% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ppc-loo.R
\name{PPC-loo}
\alias{PPC-loo}
\alias{ppc_loo_pit_overlay}
\alias{ppc_loo_pit_qq}
\alias{ppc_loo_pit}
\alias{ppc_loo_intervals}
\alias{ppc_loo_ribbon}
\title{LOO predictive checks}
\usage{
ppc_loo_pit_overlay(y, yrep, lw, pit, samples = 100, ..., size = 0.25,
  alpha = 0.7, trim = FALSE, bw = "nrd0", adjust = 1,
  kernel = "gaussian", n_dens = 1024)

ppc_loo_pit_qq(y, yrep, lw, pit, compare = c("uniform", "normal"), ...,
  size = 2, alpha = 1)

ppc_loo_pit(y, yrep, lw, pit, compare = c("uniform", "normal"), ...,
  size = 2, alpha = 1)

ppc_loo_intervals(y, yrep, lw, psis_object, intervals = NULL, ...,
  prob = 0.9, size = 1, fatten = 3, order = c("index", "median"))

ppc_loo_ribbon(y, yrep, lw, psis_object, intervals = NULL, ..., prob = 0.9,
  alpha = 0.33, size = 0.25)
}
\arguments{
\item{y}{A vector of observations. See \strong{Details}.}

\item{yrep}{An \eqn{S} by \eqn{N} matrix of draws from the posterior
predictive distribution, where \eqn{S} is the size of the posterior sample
(or subset of the posterior sample used to generate \code{yrep}) and
\eqn{N} is the number of observations (the length of \code{y}). The columns
of \code{yrep} should be in the same order as the data points in \code{y}
for the plots to make sense. See \strong{Details} for additional
instructions.}

\item{lw}{A matrix of (smoothed) log weights with the same dimensions as
\code{yrep}. If using \pkg{loo < 2.0.0} see the \code{\link[loo]{psislw}}
function in the \pkg{loo} package, which returns smoothed weights that can
be used to specify \code{lw}. If using \pkg{loo >= 2.0.0} see the
\code{psis} function and the associated \code{weights} method.}

\item{pit}{For \code{ppc_loo_pit_overlay} and \code{ppc_loo_pit_qq},
optionally a vector of precomputed PIT values that can be specified instead
of \code{y}, \code{yrep}, and \code{lw} (these are all ignored if
\code{pit} is specified). If not specified the PIT values are computed
internally before plotting.}

\item{samples}{For \code{ppc_loo_pit_overlay}, the number of data sets (each
the same size as \code{y}) to simulate from the standard uniform
distribution. The default is 100. The density estimate of each dataset is
plotted as a thin line in the plot, with the density estimate of the LOO
PITs overlaid as a thicker dark line.}

\item{...}{Currently unused.}

\item{alpha, size, fatten}{Arguments passed to code geoms to control plot
aesthetics. For \code{ppc_loo_pit_qq} and \code{ppc_loo_pit_overlay},
\code{size} and \code{alpha} are passed to
\code{\link[ggplot2]{geom_point}} and \code{\link[ggplot2]{geom_density}},
respectively. For \code{ppc_loo_intervals}, \code{size} and \code{fatten}
are passed to \code{\link[ggplot2]{geom_pointrange}}. For
\code{ppc_loo_ribbon}, \code{alpha} and \code{size} are passed to
\code{\link[ggplot2]{geom_ribbon}}.}

\item{trim}{Passed to \code{\link[ggplot2]{stat_density}}.}

\item{bw, adjust, kernel, n_dens}{Optional arguments passed to
\code{\link[stats]{density}} to override default kernel density estimation
parameters. \code{n_dens} defaults to 1024.}

\item{compare}{For \code{ppc_loo_pit_qq}, a string that can be either
\code{"uniform"} or \code{"normal"}. If \code{"uniform"} (the default) the
Q-Q plot compares computed PIT values to the standard uniform distribution.
If \code{compare="normal"}, the Q-Q plot compares standardized PIT values
to the standard normal distribution.}

\item{psis_object}{If using \pkg{loo} version \code{2.0.0} or greater, an
object returned by the \code{psis} function (or by the \code{loo} function
with argument \code{save_psis} set to \code{TRUE}).}

\item{intervals}{For \code{ppc_loo_intervals} and \code{ppc_loo_ribbon},
optionally a matrix of precomputed LOO predictive intervals intervals with
that can be specified instead of \code{yrep} and \code{lw} (these are both
ignored if \code{intervals} is specified). If not specified the intervals
are computed internally before plotting. If specified, \code{intervals}
must be a matrix with number of rows equal to the number of data points and
three columns in the following order: the first for the lower bound of the
interval, the second for median (50\%), and the third for the interval
upper bound (column names are ignored).}

\item{prob}{A value between 0 and 1 indicating the desired probability mass
to include in the intervals. The default is 0.9.}

\item{order}{For \code{ppc_loo_intervals}, a string indicating how to arrange
the plotted intervals. The default (\code{"index"}) is to plot them in the
order of the observations. The alternative (\code{"median"}) arranges them
by median value from smallest (left) to largest (right).}
}
\value{
A ggplot object that can be further customized using the
  \pkg{ggplot2} package.
}
\description{
Leave-One-Out (LOO) predictive checks. See the \strong{Plot Descriptions}
section below for details.
}
\section{Plot Descriptions}{

\describe{
\item{\code{ppc_loo_pit_qq,ppc_loo_pit_overlay}}{
 The calibration of marginal predictions can be assessed using probability
 integral transformation (PIT) checks. LOO improves the check by avoiding the
 double use of data. See the section on marginal predictive checks in Gelman
 et al. (2013, p. 152--153) and section 5 of Gabry et al. (2018) for an
 example of using \pkg{bayesplot} for these checks.

 The LOO PIT values are asymptotically uniform (for continuous data) if the
 model is calibrated. The \code{ppc_loo_pit_overlay} function creates a plot
 comparing the density of the LOO PITs (thick line) to the density estimates
 of many simulated data sets from the standard uniform distribution (thin
 lines). See Gabry et al. (2018) for an example of interpreting the shape of
 the miscalibration that can be observed in these plots.

 The \code{ppc_loo_pit_qq} function provides an alternative visualization of
 the miscalibration with a quantile-quantile (Q-Q) plot comparing the LOO
 PITs to the standard uniform distribution. Comparing to the uniform is not
 good for extreme probabilities close to 0 and 1, so it can sometimes be
 useful to set the \code{compare} argument to \code{"normal"}, which will
 produce a Q-Q plot comparing standardized PIT values to the standard normal
 distribution that can help see the (mis)calibration better for the extreme
 values. However, in most cases we have found that the overlaid density plot
 (\code{ppc_loo_pit_overlay}) function will provided a clearer picture of
 calibration problems that the Q-Q plot.
}
\item{\code{ppc_loo_intervals, ppc_loo_ribbon}}{
 Similar to \code{\link{ppc_intervals}} and \code{\link{ppc_ribbon}} but the
 intervals are for the LOO predictive distribution.
}
}
}

\examples{

\dontrun{
library(rstanarm)
library(loo)

head(radon)
fit <- stan_lmer(
  log_radon ~ floor + log_uranium + floor:log_uranium
               + (1 + floor | county),
  data = radon,
  iter = 1000,
  chains = 2  # ,cores = 2
 )
y <- radon$log_radon
yrep <- posterior_predict(fit)

if (packageVersion("loo") < "2.0.0") {
  psis1 <- psislw(-log_lik(fit), cores = 2)
  lw <- psis1$lw_smooth
} else {
  psis1 <- psis(-log_lik(fit), cores = 2)
  lw <- weights(psis1)
}

# marginal predictive check using LOO probability integral transform
color_scheme_set("orange")
ppc_loo_pit_overlay(y, yrep, lw = lw, adjust = 0.9)

ppc_loo_pit_qq(y, yrep, lw = lw)
ppc_loo_pit_qq(y, yrep, lw = lw, compare = "normal")


# loo predictive intervals vs observations
sel <- 800:900
ppc_loo_intervals(y[sel], yrep[, sel], psis1$lw_smooth[, sel],
                  prob = 0.9, size = 0.5)

color_scheme_set("gray")
ppc_loo_intervals(y[sel], yrep[, sel], psis1$lw_smooth[, sel],
                  order = "median", prob = 0.8, size = 0.5)
}

}
\references{
Gelman, A., Carlin, J. B., Stern, H. S., Dunson, D. B., Vehtari,
  A., and Rubin, D. B. (2013). \emph{Bayesian Data Analysis.} Chapman & Hall/CRC
  Press, London, third edition. (p. 152--153)

Gabry, J., Simpson, D., Vehtari, A., Betancourt, M., and Gelman,
  A. (2018). Visualization in Bayesian workflow. \emph{Journal of the Royal
  Statistical Society Series A}, accepted for publication. arXiv preprint:
  \url{http://arxiv.org/abs/1709.01449}.

Vehtari, A., Gelman, A., and Gabry, J. (2017). Practical
  Bayesian model evaluation using leave-one-out cross-validation and WAIC.
  \emph{Statistics and Computing}. 27(5), 1413--1432.
  doi:10.1007/s11222-016-9696-4. arXiv preprint:
  \url{http://arxiv.org/abs/1507.04544/}
}
\seealso{
Other PPCs: \code{\link{PPC-discrete}},
  \code{\link{PPC-distributions}},
  \code{\link{PPC-errors}}, \code{\link{PPC-intervals}},
  \code{\link{PPC-overview}},
  \code{\link{PPC-scatterplots}},
  \code{\link{PPC-test-statistics}}
}
