% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bayesfactor_parameters.R
\name{bayesfactor_parameters}
\alias{bayesfactor_parameters}
\alias{bayesfactor_pointnull}
\alias{bayesfactor_rope}
\alias{bf_parameters}
\alias{bf_pointnull}
\alias{bf_rope}
\alias{bayesfactor_parameters.numeric}
\alias{bayesfactor_parameters.stanreg}
\alias{bayesfactor_parameters.data.frame}
\title{Bayes Factors (BF) for a Single Parameter}
\usage{
bayesfactor_parameters(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  ...,
  verbose = TRUE
)

bayesfactor_pointnull(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  ...,
  verbose = TRUE
)

bayesfactor_rope(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = rope_range(posterior, verbose = FALSE),
  ...,
  verbose = TRUE
)

bf_parameters(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  ...,
  verbose = TRUE
)

bf_pointnull(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  ...,
  verbose = TRUE
)

bf_rope(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = rope_range(posterior, verbose = FALSE),
  ...,
  verbose = TRUE
)

\method{bayesfactor_parameters}{numeric}(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  ...,
  verbose = TRUE
)

\method{bayesfactor_parameters}{stanreg}(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  effects = "fixed",
  component = "conditional",
  parameters = NULL,
  ...,
  verbose = TRUE
)

\method{bayesfactor_parameters}{data.frame}(
  posterior,
  prior = NULL,
  direction = "two-sided",
  null = 0,
  rvar_col = NULL,
  ...,
  verbose = TRUE
)
}
\arguments{
\item{posterior}{A numerical vector, \code{stanreg} / \code{brmsfit} object,
\code{emmGrid} or a data frame - representing a posterior distribution(s)
from (see 'Details').}

\item{prior}{An object representing a prior distribution (see 'Details').}

\item{direction}{Test type (see 'Details'). One of \code{0},
\code{"two-sided"} (default, two tailed), \code{-1}, \code{"left"} (left
tailed) or \code{1}, \code{"right"} (right tailed).}

\item{null}{Value of the null, either a scalar (for point-null) or a range
(for a interval-null).}

\item{...}{Arguments passed to and from other methods. (Can be used to pass
arguments to internal \code{\link[logspline:logspline]{logspline::logspline()}}.)}

\item{verbose}{Toggle off warnings.}

\item{effects}{Should results for fixed effects (\code{"fixed"}, the default),
random effects (\code{"random"}) or both ("\verb{all"}) be returned? Only applies to
mixed models. May be abbreviated.}

\item{component}{Which type of parameters to return, such as parameters for
the conditional model, the zero-inflated part of the model, the dispersion
term, etc. See details in section \emph{Model Components}. May be abbreviated.
Note that the \emph{conditional} component also refers to the \emph{count} or \emph{mean}
component - names may differ, depending on the modeling package. There are
three convenient shortcuts (not applicable to \emph{all} model classes):
\itemize{
\item \code{component = "all"} returns all possible parameters.
\item If \code{component = "location"}, location parameters such as \code{conditional},
\code{zero_inflated}, \code{smooth_terms}, or \code{instruments} are returned (everything
that are fixed or random effects - depending on the \code{effects} argument -
but no auxiliary parameters).
\item For \code{component = "distributional"} (or \code{"auxiliary"}), components like
\code{sigma}, \code{dispersion}, \code{beta} or \code{precision} (and other auxiliary
parameters) are returned.
}}

\item{parameters}{Regular expression pattern that describes the parameters
that should be returned. Meta-parameters (like \code{lp__} or \code{prior_}) are
filtered by default, so only parameters that typically appear in the
\code{summary()} are returned. Use \code{parameters} to select specific parameters
for the output.}

\item{rvar_col}{A single character - the name of an \code{rvar} column in the data
frame to be processed. See example in \code{\link[=p_direction]{p_direction()}}.}
}
\value{
A data frame containing the (log) Bayes factor representing evidence
\emph{against} the null  (Use \code{as.numeric()} to extract the non-log Bayes
factors; see examples).
}
\description{
This method computes Bayes factors against the null (either a point or an
interval), based on prior and posterior samples of a single parameter. This
Bayes factor indicates the degree by which the mass of the posterior
distribution has shifted further away from or closer to the null value(s)
(relative to the prior distribution), thus indicating if the null value has
become less or more likely given the observed data.
\cr \cr
When the null is an interval, the Bayes factor is computed by comparing the
prior and posterior odds of the parameter falling within or outside the null
interval (Morey & Rouder, 2011; Liao et al., 2020); When the null is a point,
a Savage-Dickey density ratio is computed, which is also an approximation of
a Bayes factor comparing the marginal likelihoods of the model against a
model in which the tested parameter has been restricted to the point null
(Wagenmakers et al., 2010; Heck, 2019).
\cr \cr
Note that the \code{logspline} package is used for estimating densities and
probabilities, and must be installed for the function to work.
\cr \cr
\code{bayesfactor_pointnull()} and \code{bayesfactor_rope()} are wrappers
around \code{bayesfactor_parameters} with different defaults for the null to
be tested against (a point and a range, respectively). Aliases of the main
functions are prefixed with \verb{bf_*}, like \code{bf_parameters()} or
\code{bf_pointnull()}.
\cr \cr
\strong{For more info, in particular on specifying correct priors for factors
with more than 2 levels, see
\href{https://easystats.github.io/bayestestR/articles/bayes_factors.html}{the Bayes factors vignette}.}
}
\details{
This method is used to compute Bayes factors based on prior and posterior
distributions.

\subsection{One-sided & Dividing Tests (setting an order restriction)}{
One sided tests (controlled by \code{direction}) are conducted by restricting
the prior and posterior of the non-null values (the "alternative") to one
side of the null only (\cite{Morey & Wagenmakers, 2014}). For example, if we
have a prior hypothesis that the parameter should be positive, the
alternative will be restricted to the region to the right of the null (point
or interval). For example, for a Bayes factor comparing the "null" of \code{0-0.1}
to the alternative \verb{>0.1}, we would set
\code{bayesfactor_parameters(null = c(0, 0.1), direction = ">")}.
\cr\cr
It is also possible to compute a Bayes factor for \strong{dividing}
hypotheses - that is, for a null and alternative that are complementary,
opposing one-sided hypotheses (\cite{Morey & Wagenmakers, 2014}). For
example, for a Bayes factor comparing the "null" of \verb{<0} to the alternative
\verb{>0}, we would set \code{bayesfactor_parameters(null = c(-Inf, 0))}.
}
}
\note{
There is also a
\href{https://easystats.github.io/see/articles/bayestestR.html}{\code{plot()}-method}
implemented in the
\href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\section{Setting the correct \code{prior}}{

For the computation of Bayes factors, the model priors must be proper priors
(at the very least they should be \emph{not flat}, and it is preferable that
they be \emph{informative}); As the priors for the alternative get wider, the
likelihood of the null value(s) increases, to the extreme that for completely
flat priors the null is infinitely more favorable than the alternative (this
is called \emph{the Jeffreys-Lindley-Bartlett paradox}). Thus, you should
only ever try (or want) to compute a Bayes factor when you have an informed
prior.
\cr\cr
(Note that by default, \code{brms::brm()} uses flat priors for fixed-effects;
See example below.)
\cr\cr
It is important to provide the correct \code{prior} for meaningful results,
to match the \code{posterior}-type input:
\itemize{
\item \strong{A numeric vector} - \code{prior} should also be a \emph{numeric vector}, representing the prior-estimate.
\item \strong{A data frame} - \code{prior} should also be a \emph{data frame}, representing the prior-estimates, in matching column order.
\itemize{
\item If \code{rvar_col} is specified, \code{prior} should be \emph{the name of an \code{rvar} column} that represents the prior-estimates.
}
\item \strong{Supported Bayesian model (\code{stanreg}, \code{brmsfit}, etc.)}
\itemize{
\item \code{prior} should be \emph{a model an equivalent model with MCMC samples from the priors \strong{only}}. See \code{\link[=unupdate]{unupdate()}}.
\item If \code{prior} is set to \code{NULL}, \code{\link[=unupdate]{unupdate()}} is called internally (not supported for \code{brmsfit_multiple} model).
}
\item \strong{Output from a \code{{marginaleffects}} function} - \code{prior} should also be \emph{an equivalent output} from a \code{{marginaleffects}} function based on a prior-model
(See \code{\link[=unupdate]{unupdate()}}).
\item \strong{Output from an \code{{emmeans}} function}
\itemize{
\item \code{prior} should also be \emph{an equivalent output} from an \code{{emmeans}} function based on a prior-model (See \code{\link[=unupdate]{unupdate()}}).
\item \code{prior} can also be \emph{the original (posterior) model}, in which case the function
will try to "unupdate" the estimates (not supported if the estimates have undergone
any transformations -- \code{"log"}, \code{"response"}, etc. -- or any \code{regrid}ing).
}
}
}

\section{Interpreting Bayes Factors}{

A Bayes factor greater than 1 can be interpreted as evidence against the
null, at which one convention is that a Bayes factor greater than 3 can be
considered as "substantial" evidence against the null (and vice versa, a
Bayes factor smaller than 1/3 indicates substantial evidence in favor of the
null-model) (\cite{Wetzels et al. 2011}).
}

\section{Model components}{


Possible values for the \code{component} argument depend on the model class.
Following are valid options:
\itemize{
\item \code{"all"}: returns all model components, applies to all models, but will only
have an effect for models with more than just the conditional model
component.
\item \code{"conditional"}: only returns the conditional component, i.e. "fixed
effects" terms from the model. Will only have an effect for models with
more than just the conditional model component.
\item \code{"smooth_terms"}: returns smooth terms, only applies to GAMs (or similar
models that may contain smooth terms).
\item \code{"zero_inflated"} (or \code{"zi"}): returns the zero-inflation component.
\item \code{"location"}: returns location parameters such as \code{conditional},
\code{zero_inflated}, or \code{smooth_terms} (everything that are fixed or random
effects - depending on the \code{effects} argument - but no auxiliary
parameters).
\item \code{"distributional"} (or \code{"auxiliary"}): components like \code{sigma},
\code{dispersion}, \code{beta} or \code{precision} (and other auxiliary parameters) are
returned.
}

For models of class \code{brmsfit} (package \strong{brms}), even more options are
possible for the \code{component} argument, which are not all documented in detail
here. See also \href{https://easystats.github.io/insight/reference/find_parameters.BGGM.html}{\code{?insight::find_parameters}}.
}

\examples{
\dontshow{if (require("logspline")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(bayestestR)
prior <- distribution_normal(1000, mean = 0, sd = 1)
posterior <- distribution_normal(1000, mean = .5, sd = .3)
(BF_pars <- bayesfactor_parameters(posterior, prior, verbose = FALSE))

as.numeric(BF_pars)
\dontshow{\}) # examplesIf}
\dontshow{if (require("rstanarm") && require("emmeans") && require("logspline")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
\donttest{
# rstanarm models
# ---------------
contrasts(sleep$group) <- contr.equalprior_pairs # see vingette
stan_model <- suppressWarnings(stan_lmer(
  extra ~ group + (1 | ID),
  data = sleep,
  refresh = 0
))
bayesfactor_parameters(stan_model, verbose = FALSE)
bayesfactor_parameters(stan_model, null = rope_range(stan_model))

# emmGrid objects
# ---------------
group_diff <- pairs(emmeans(stan_model, ~group, data = sleep))
bayesfactor_parameters(group_diff, prior = stan_model, verbose = FALSE)

# Or
# group_diff_prior <- pairs(emmeans(unupdate(stan_model), ~group))
# bayesfactor_parameters(group_diff, prior = group_diff_prior, verbose = FALSE)
}
\dontshow{\}) # examplesIf}
\dontshow{if (require("brms") && require("logspline")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# brms models
# -----------
\dontrun{
contrasts(sleep$group) <- contr.equalprior_pairs # see vingette
my_custom_priors <-
  set_prior("student_t(3, 0, 1)", class = "b") +
  set_prior("student_t(3, 0, 1)", class = "sd", group = "ID")

brms_model <- suppressWarnings(brm(extra ~ group + (1 | ID),
  data = sleep,
  prior = my_custom_priors,
  refresh = 0
))
bayesfactor_parameters(brms_model, verbose = FALSE)
}
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Wagenmakers, E. J., Lodewyckx, T., Kuriyal, H., and Grasman, R. (2010).
Bayesian hypothesis testing for psychologists: A tutorial on the
Savage-Dickey method. Cognitive psychology, 60(3), 158-189.
\item Heck, D. W. (2019). A caveat on the Savage–Dickey density ratio: The
case of computing Bayes factors for regression parameters. British Journal of
Mathematical and Statistical Psychology, 72(2), 316-333.
\item Morey, R. D., & Wagenmakers, E. J. (2014). Simple relation between
Bayesian order-restricted and point-null hypothesis tests. Statistics &
Probability Letters, 92, 121-124.
\item Morey, R. D., & Rouder, J. N. (2011). Bayes factor approaches for
testing interval null hypotheses. Psychological methods, 16(4), 406.
\item Liao, J. G., Midya, V., & Berg, A. (2020). Connecting and contrasting
the Bayes factor and a modified ROPE procedure for testing interval null
hypotheses. The American Statistician, 1-19.
\item Wetzels, R., Matzke, D., Lee, M. D., Rouder, J. N., Iverson, G. J., and
Wagenmakers, E.-J. (2011). Statistical Evidence in Experimental Psychology:
An Empirical Comparison Using 855 t Tests. Perspectives on Psychological
Science, 6(3), 291–298. \doi{10.1177/1745691611406923}
}
}
\author{
Mattan S. Ben-Shachar
}
