% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ROCsurface.R
\name{ROCsurface}
\alias{ROCsurface}
\alias{ROCs.tcf}
\alias{ROCs}
\title{Receiver operating characteristics surface for a continuous diagnostic test}
\usage{
ROCs.tcf(method = "full", T, Dvec, V = NULL, rhoEst = NULL,
  piEst = NULL, cps)

ROCs(method = "full", T, Dvec, V, A = NULL, rhoEst = NULL,
  piEst = NULL, ncp = 100, plot = TRUE, ellipsoid = FALSE,
  cpst = NULL, level = 0.95, sur.col = c("gray40", "green"),
  BOOT = FALSE, nR = 250, parallel = FALSE,
  ncpus = ifelse(parallel, detectCores()/2, NULL), ...)
}
\arguments{
\item{method}{a estimation method to be used for estimating the true class fractions in presence of verification bias. See 'Details'.}

\item{T}{a numeric vector containing the diagnostic test values. \code{NA} values are not allowed.}

\item{Dvec}{a n * 3  binary matrix with the three columns, corresponding to three classes of the disease status. In row i, 1 in column j indicates that the i-th subject belongs to class j, with j = 1, 2, 3. A row of \code{NA} values indicates a non-verified subject.}

\item{V}{a binary vector containing the verification status (1 verified, 0 not verified).}

\item{rhoEst}{a result of a call to \code{\link{rhoMLogit}} of \code{\link{rhoKNN}} to fit the disease model.}

\item{piEst}{a result of a call to \code{\link{psglm}} to fit the verification model.}

\item{cps}{a cut point \eqn{(c_1, c_2)}, with \eqn{c_1 < c_2}, which used to estimate TCFs. If \code{m} estimates of TCFs are required, \code{cps} must be matrix with \code{m} rows and 2 columns.}

\item{A}{a vector/matrix of dimension n * q containing the values of the covariate(s). If the method is \code{"knn"} and \code{ellipsoid = TRUE}, \code{A} is needed to compute the asymptotic covariance of TCFs at a fixed cut point. The default \code{NULL} is suitable for the remaining methods.}

\item{ncp}{the dimension of cut point grid. It is used to determine the cut points (see 'Details'). Default 100.}

\item{plot}{if \code{TRUE}(the default), a 3D plot of ROC surface is produced.}

\item{ellipsoid}{a logical value. If TRUE, adds an ellipsoidal confidence region for TCFs at a specified cut point to current plot of ROC surface.}

\item{cpst}{a specified cut point, which used to construct the ellipsoid confidence region. If \code{m} ellipsoid confidence regions are required, \code{cpst} must be matrix with \code{m} rows and 2 columns. Default \code{NULL}.}

\item{level}{an confidence level to be used for constructing the ellipsoid confidence region; default 0.95.}

\item{sur.col}{color to be used for plotting ROC surface and ellipsoid.}

\item{BOOT}{a logical value. Default = \code{FALSE}. If set to \code{TRUE}, a bootstrap resampling is employed to estimate the asymptotic variance-covariance matrix of TCFs at the cut point \code{cpst}. See more details in \code{\link{asyCovTCF}}.}

\item{nR}{the number of bootstrap replicates, which is used for FULL estimator, or option \code{BOOT = TRUE}. Usually this will be a single positive integer. Default 250.}

\item{parallel}{a logical value. If \code{TRUE}, a parallel computing is employed to the bootstrap resampling process.}

\item{ncpus}{number of processes to be used in parallel computing. Default is half of of available cores.}

\item{...}{optional arguments to be passed to \code{\link[rgl]{plot3d}}, \code{\link[rgl]{surface3d}}.}
}
\value{
\code{ROCs} returns a list, with the following components:
\item{vals}{the estimates of TCFs at all cut points.}
\item{cpoint}{the cut points are used to construct the ROC surface.}
\item{ncp}{dimension of the cut point grid.}
\item{cpst}{the cut points are used to construct the ellipsoidal confidence regions.}
\item{tcf}{the estimates of TCFs at the cut points \code{cpst}.}
\item{message}{an integer code or vector. 1 indicates the ellipsoidal confidence region is available.}

\code{ROCs.tcf} returns a vector having estimates of TCFs at a cut point when \code{cps} is a vector with two elements, or a list of estimates of TCFs at \code{m} cut points when \code{cps} is a \code{m*2} matrix. In addition, some attributes called \code{theta}, \code{beta}, \code{cp} and \code{name} are given. Here, \code{theta} is a probability vector, with 3 element, corresponding to the disease prevalence rates of three classes. \code{beta} is also a probability vector having 4 components, which are used to compute TCFs, see To Duc el al. (2016a, 2016b) for more details. \code{cp} is the specified cut point that is used to estimate TCFs. \code{name} indicates the method used to estimate TCFs. These attributes are required to compute the asymptotic variance-covariance matrix of TCFs at the given cut point.
}
\description{
\code{ROCs.tcf} is used to obtain bias-corrected estimates of the true class fractions (TCFs) for evaluating the accuracy of a continuous diagnostic test for a given cut point \eqn{(c_1, c_2)}, with \eqn{c_1 < c_2}.

\code{ROCs} provides bias-corrected estimates of the ROC surfaces of the continuous diagnostic test by using TCF.
}
\details{
In a three-class diagnostic problem, quantities used to evaluate the accuracy of a diagnostic test are the true class fractions (TCFs). For a given pair of cut points \eqn{(c_1, c_2)} such that \eqn{c_1 < c_2}, subjects are classified into class 1 (\eqn{D_1}) if \eqn{T < c_1}; class 2 (\eqn{D_2}) if \eqn{c_1 \le T < c_2}; class 3 (\eqn{D_3}) otherwise. The true class fractions of the test \eqn{T} at \eqn{(c_1, c_2)} are defined as
\deqn{TCF_1(c_1) = P(T < c_1| D_1 = 1) = 1 - P(T \ge c_1| D_1 = 1),}
\deqn{TCF_2(c_1, c_2) = P(c_1 \le T < c_2| D_2 = 1) = P(T \ge c_1| D_2 = 1) - P(T \ge c_2| D_2 = 1),}
\deqn{TCF_3(c_2) = P(T > c_2| D_3 = 1) = P(T \ge c_2| D_3 = 1). }

The receiver operating characteristic (ROC) surface is the plot of \eqn{TCF_1}, \eqn{TCF_2} and \eqn{TCF_3} by varying the cut point \eqn{(c_1, c_2)} in the domain of the diagnostic test. The cut points \eqn{(c_1, c_2)} are produced by designing a cut point grid with \code{ncp} dimension. In this grid, the points satisfying \eqn{c_1 < c_2} are selected as the cut points. The number of the cut points are obtained as \eqn{ncp(ncp - 1)/2}, for example, the default is 4950.

These functions implement the bias-corrected estimators in To Duc et al (2016a, 2016b) for estimating TCF of a three-class continuous diagnostic test in presence of verification bias. The estimators work under MAR assumption. Five methods are provided, namely:
\itemize{
   \item Full imputation (FI): uses the fitted values of the disease model to replace the true disease status (both of missing and non-missing values).
   \item Mean score imputation (MSI): replaces only the missing values by the fitted values of the disease model.
   \item Inverse probability weighted (IPW): weights each observation in the verification sample by the inverse of the sampling fraction (i.e. the probability that the subject was selected for verification).
   \item Semiparametric efficient (SPE): replaces the true disease status by the double robust estimates.
   \item K nearest-neighbor (KNN): uses K nearest-neighbor imputation to obtain the missing values of the true disease status.
}

The argument \code{method} must be selected from the collection of the bias-corrected methods, i.e., \code{"full"}, \code{"fi"}, \code{"msi"}, \code{"ipw"}, \code{"spe"} and \code{"knn"}.

The ellipsoidal confidence region of TCFs at a given cut point can be constructed by using a normal approximation and plotted in the ROC surface space. The confidence level (default) is 0.95.

Note that, before using the functions \code{ROCs} and \code{ROCs.tcf}, the use of \code{\link{preDATA}} might be needed to check the monotone ordering disease classes and to create the matrix format for disease status.
}
\examples{
data(EOC)
head(EOC)

\dontrun{
# FULL data estimator
Dfull <- preDATA(EOC$D.full, EOC$CA125)
Dvec.full <- Dfull$Dvec
ROCs("full", T = EOC$CA125, Dvec = Dvec.full, , ncp = 30, ellipsoid = TRUE,
     cpst = c(-0.56, 2.31))
}

# Preparing the missing disease status
Dna <- preDATA(EOC$D, EOC$CA125)
Dvec.na <- Dna$Dvec
Dfact.na <- Dna$D

# FI estimator
rho.out <- rhoMLogit(Dfact.na ~ CA125 + CA153 + Age, data = EOC, test = TRUE)
ROCs("fi", T = EOC$CA125, Dvec = Dvec.na, V = EOC$V, rhoEst = rho.out, ncp = 30)

\dontrun{
# Plot ROC surface and add ellipsoid confidence region
ROCs("fi", T = EOC$CA125, Dvec = Dvec.na, V = EOC$V, rhoEst = rho.out, ncp = 30,
     ellipsoid = TRUE, cpst = c(-0.56, 2.31))

# MSI estimator
ROCs("msi", T = EOC$CA125, Dvec = Dvec.na, V = EOC$V, rhoEst = rho.out, ncp = 30,
     ellipsoid = TRUE, cpst = c(-0.56, 2.31))

# IPW estimator
pi.out <- psglm(V ~ CA125 + CA153 + Age, data = EOC, test = TRUE)
ROCs("ipw", T = EOC$CA125, Dvec = Dvec.na, V = EOC$V, piEst = pi.out, ncp = 30,
     ellipsoid = TRUE, cpst = c(-0.56, 2.31))

# SPE estimator
ROCs("spe", T = EOC$CA125, Dvec = Dvec.na, V = EOC$V, rhoEst = rho.out, ncp = 30,
     piEst = pi.out, ellipsoid = TRUE, cpst = c(-0.56, 2.31))

# 1NN estimator
XX <- cbind(EOC$CA125, EOC$CA153, EOC$Age)
K.opt <- CVknn(X = XX, Dvec = Dvec.na, V = EOC$V, type = "mahala", plot = TRUE)
rho.1nn <- rhoKNN(X = XX, Dvec = Dvec.na, V = EOC$V, K = K.opt, type = "mahala")
ROCs("knn", T = EOC$CA125, Dvec = Dvec.na, V = EOC$V, rhoEst = rho.1nn, ncp = 30,
     ellipsoid = TRUE, cpst = c(-0.56, 2.31))

## Compute TCFs at three cut points
cutps <- rbind(c(0,0.5), c(0,1), c(0.5,1))
ROCs.tcf("spe", T = EOC$CA125, Dvec = Dvec.na, V = EOC$V, rhoEst = rho.out, ncp = 30,
         piEst = pi.out, cps = cutps)
}

}
\references{
To Duc, K., Chiogna, M. and Adimari, G. (2016a)
Bias-corrected methods for estimating the receiver operating characteristic surface of continuous diagnostic tests.
\emph{Electronic Journal of Statistics}, \bold{10}, 3063-3113.

To Duc, K., Chiogna, M. and Adimari, G. (2016b)
Nonparametric Estimation of ROC Surfaces Under Verification Bias.
\url{https://arxiv.org/abs/1604.04656v1}. Submitted.
}
\seealso{
\code{\link{psglm}}, \code{\link{rhoMLogit}}, \code{\link[rgl]{plot3d}}.
}
