#' Estimate Differences of Standardized Slopes
#' and Generate the Corresponding Sampling Distribution
#' Using the Monte Carlo Method
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of differences of standardized regression slopes
#'   is derived from each randomly generated vector of parameter estimates.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of differences of standardized regression slopes,
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betamc` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{object}{The function argument `object`.}
#'     \item{thetahatstar}{Sampling distribution of
#'       differences of standardized regression slopes.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       differences of standardized regression slopes.}
#'     \item{est}{Vector of estimated
#'       differences of standardized regression slopes.}
#'     \item{fun}{Function used ("DiffBetaMC").}
#'   }
#'
#' @inheritParams BetaMC
#'
#' @examples
#' # Fit the regression model
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#' # Generate the sampling distribution of parameter estimates
#' # (use a large R, for example, R = 20000 for actual research)
#' mc <- MC(object, R = 100)
#' # Generate confidence intervals
#' # for differences of standardized regression slopes
#' diff <- DiffBetaMC(mc)
#' # Methods --------------------------------------------------------
#' print(diff)
#' summary(diff)
#' coef(diff)
#' vcov(diff)
#' confint(diff, level = 0.95)
#' @export
#' @family Beta Monte Carlo Functions
#' @keywords betaMC diff
DiffBetaMC <- function(object) {
  stopifnot(
    methods::is(
      object,
      "mc"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  std <- BetaMC(object)
  std <- std$thetahatstar
  std <- do.call(
    what = "rbind",
    args = std
  )
  p_dif <- dim(object$lm_process$dif_idx)[2]
  thetahatstar <- matrix(
    data = 0.0,
    ncol = p_dif,
    nrow = dim(std)[1]
  )
  for (i in seq_len(p_dif)) {
    thetahatstar[, i] <- std[
      ,
      object$lm_process$dif_idx[1, i]
    ] - std[
      ,
      object$lm_process$dif_idx[2, i]
    ]
  }
  colnames(thetahatstar) <- names(object$lm_process$dif_betastar)
  thetahatstar <- as.data.frame(
    t(
      thetahatstar
    )
  )
  out <- list(
    call = match.call(),
    object = object,
    thetahatstar = thetahatstar,
    est = object$lm_process$dif_betastar,
    fun = "DiffBetaMC"
  )
  class(out) <- c(
    "betamc",
    class(out)
  )
  return(
    out
  )
}
