#' Estimate Differences of Standardized Slopes
#' and Generate the Corresponding Sampling Distribution
#' Using Nonparametric Bootstrapping
#'
#' @author Ivan Jacob Agaloos Pesigan
#'
#' @details The vector of differences of standardized regression slopes
#'   is estimated from bootstrap samples.
#'   Confidence intervals are generated by obtaining
#'   percentiles corresponding to \eqn{100(1 - \alpha)\%}
#'   from the generated sampling
#'   distribution of differences of standardized regression slopes,
#'   where \eqn{\alpha} is the significance level.
#'
#' @return Returns an object
#'   of class `betanb` which is a list with the following elements:
#'   \describe{
#'     \item{call}{Function call.}
#'     \item{args}{Function arguments.}
#'     \item{thetahatstar}{Sampling distribution of
#'       differences of standardized regression slopes.}
#'     \item{vcov}{Sampling variance-covariance matrix of
#'       differences of standardized regression slopes.}
#'     \item{est}{Vector of estimated
#'       differences of standardized regression slopes.}
#'     \item{fun}{Function used ("DiffBetaNB").}
#'   }
#'
#' @inheritParams BetaNB
#'
#' @examples
#' # Data ---------------------------------------------------------------------
#' data("nas1982", package = "betaNB")
#'
#' # Fit Model in lm ----------------------------------------------------------
#' object <- lm(QUALITY ~ NARTIC + PCTGRT + PCTSUPP, data = nas1982)
#'
#' # NB -----------------------------------------------------------------------
#' nb <- NB(
#'   object,
#'   R = 100, # use a large value e.g., 5000L for actual research
#'   seed = 0508
#' )
#'
#' # DiffBetaNB ---------------------------------------------------------------
#' out <- DiffBetaNB(nb, alpha = 0.05)
#'
#' ## Methods -----------------------------------------------------------------
#' print(out)
#' summary(out)
#' coef(out)
#' vcov(out)
#' confint(out, level = 0.95)
#'
#' @family Beta Nonparametric Bootstrap Functions
#' @keywords betaNB diff
#' @export
DiffBetaNB <- function(object,
                       alpha = c(0.05, 0.01, 0.001)) {
  stopifnot(
    inherits(
      object,
      "nb"
    )
  )
  if (object$lm_process$p < 2) {
    stop("Two or more regressors is required.")
  }
  fun <- "DiffBetaNB"
  est <- object$lm_process$dif_betastar
  p_dif <- dim(object$lm_process$dif_idx)[2]
  foo <- function(x) {
    std <- .BetaStarofSigma(
      sigmacap = x,
      q = 1 / sqrt(diag(x)),
      k = object$lm_process$k
    )
    diff <- rep(x = 0.0, times = p_dif)
    for (i in seq_len(p_dif)) {
      diff[i] <- std[
        object$lm_process$dif_idx[1, i]
      ] - std[
        object$lm_process$dif_idx[2, i]
      ]
    }
    return(diff)
  }
  thetahatstar <- lapply(
    X = object$thetahatstar,
    FUN = foo
  )
  vcov <- stats::var(
    do.call(
      what = "rbind",
      args = thetahatstar
    )
  )
  colnames(vcov) <- rownames(vcov) <- names(est)
  out <- list(
    call = match.call(),
    args = list(
      object = object,
      alpha = alpha
    ),
    thetahatstar = thetahatstar,
    jackknife = lapply(
      X = object$jackknife,
      FUN = foo
    ),
    vcov = vcov,
    est = est,
    fun = fun
  )
  class(out) <- c(
    "betanb",
    class(out)
  )
  return(
    out
  )
}
