% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/classification.R
\name{Beta.tp.fit}
\alias{Beta.tp.fit}
\title{Estimate Beta true-score distribution based on observed-score raw-moments and the effective test length.}
\usage{
Beta.tp.fit(
  x,
  min,
  max,
  etl,
  reliability = NULL,
  true.model = "4P",
  failsafe = FALSE,
  l = 0,
  u = 1,
  alpha = NA,
  beta = NA,
  output = "parameters"
)
}
\arguments{
\item{x}{Vector of observed-scores.}

\item{min}{The minimum possible score to attain on the test.}

\item{max}{The maximum possible score to attain on the test.}

\item{etl}{The value of Livingston and Lewis' effective test length. See ?ETL().}

\item{reliability}{Optional specification of the test-score reliability coefficient. If specified, overrides the input of the \code{etl} argument.}

\item{true.model}{The type of Beta distribution which is to be fit to the moments of the true-score distribution. Options are \code{"4P"} and \code{"2P"}, where "4P" refers to the four-parameter (with the same mean, variance, skewness, and kurtosis), and "2P" the two-parameter solution where both location-parameters are specified (with the same mean and variance).}

\item{failsafe}{Logical. Whether to revert to a failsafe two-parameter solution should the four-parameter solution contain invalid parameter estimates.}

\item{l}{If \code{failsafe = TRUE} or \code{true.model = "2P"}: The lower-bound of the Beta distribution. Default is 0 (i.e., the lower-bound of the Standard, two-parameter Beta distribution).}

\item{u}{If \code{failsafe = TRUE} or \code{true.model = "2P"}: The upper-bound of the Beta distribution. Default is 1 (i.e., the upper-bound of the Standard, two-parameter Beta distribution).}

\item{alpha}{If \code{failsafe = TRUE} or \code{true.model = "2P"}: The Alpha shape-parameter of the Beta distribution. Default is NA (i.e., estimate).}

\item{beta}{If \code{failsafe = TRUE} or \code{true.model = "2P"}: The Beta shape-parameter of the Beta distribution. Default is NA (i.e., estimate).}

\item{output}{Option to specify true-score distribution moments as output if the value of the output argument does not equal \code{"parameters"}.}
}
\value{
A list with the parameter values of a four-parameter Beta distribution. "l" is the lower location-parameter, "u" the upper location-parameter, "alpha" the first shape-parameter, and "beta" the second shape-parameter.
}
\description{
Estimator for the Beta true-score distribution shape-parameters from the observed-score distribution and Livingston and Lewis' effective test length. Returns a list with entries representing the lower- and upper shape parameters (l and u), and the shape parameters (alpha and beta) of the four-parameters beta distribution.
}
\note{
This estimator is based on the S-Plus code provided by Rogosa and Finkelman (2004). It includes an option for implementing a failsafe should the four-parameter solution be invalid (e.g., l < 0 or u > 1, alpha < 1 or beta < 1).
}
\examples{
# Generate some fictional data. Say 1000 individuals take a 100-item test
# where all items are equally difficult, and the true-score distribution
# is a four-parameter Beta distribution with location parameters l = .25,
# u = .75, alpha = 5, and beta = 3:
set.seed(12)
testdata <- rbinom(1000, 100, rBeta.4P(1000, .25, .75, 5, 3))

# Since this test contains items which are all equally difficult, the true
# effective test length (etl) is the actual test length. I.e., etl = 100.
# To estimate the four-parameter Beta distribution parameters underlying
# the draws from the binomial distribution:
Beta.tp.fit(testdata, 0, 100, 100)

# Imagine a case where the fitting procedure produces an impermissible
# estimate (e.g., l < 0 or u > 1).
set.seed(1234)
testdata <- rbinom(1000, 50, rBeta.4P(1000, .25, .75, 5, 3))
Beta.tp.fit(testdata, 0, 50, 50)

# This example produced an l-value estimate less than 0. One way of
# dealing with such an occurrence is to revert to a two-parameter
# model, specifying the l and u parameters and estimating the
# alpha and beta parameters necessary to produce a Beta distribution
# with the same mean and variance as the estimated true-score distribution.

# Suppose you have good theoretical reasons to fix the l parameter at a
# value of 0.25 (e.g., the test is composed of multiple-choice questions
# with four response-options, resulting in a 25\% chance of guessing the
# correct answer). The l-parameter could be specified to this theoretically
# justified value, and the u-parameter could be specified to be equal to the
# estimate above (u = 0.7256552) as such:
Beta.tp.fit(testdata, 0, 50, 50, true.model = "2P", l = 0.25, u = 0.7256552)
}
\references{
Hanson, B. A. (1991). Method of Moments Estimates for the Four-Parameter Beta Compound Binomial Model and the Calculation of Classification Consistency Indexes. American College Testing Research Report Series. Retrieved from https://files.eric.ed.gov/fulltext/ED344945.pdf

Lord, F. M. (1965). A strong true-score theory, with applications. Psychometrika. 30(3). pp. 239--270. doi: 10.1007/BF02289490

Rogosa, D. &  Finkelman, M. (2004). How Accurate Are the STAR Scores for Individual Students? – An Interpretive Guide. Retrieved from http://statweb.stanford.edu/~rag/accguide/guide04.pdf
}
