# Plotting function for OTC1() and OTC2()
###############################################################################
#' @title Plot method for optimal testing configuration results
#' 
#' @description Produce a plot for objects of class \kbd{"OTC"} 
#' returned by \code{\link{OTC1}} or \code{\link{OTC2}}.
#' 
#' @param x an object of class \kbd{"OTC"}, providing operating 
#' characteristics for the optimal testing configuration and similar 
#' configurations for a group testing algorithm.
#' @param ... currently not used.
#' 
#' @details This function produces a plot for objects of class \kbd{"OTC"} 
#' returned by \code{\link{OTC1}} or \code{\link{OTC2}}. It plots the expected 
#' number of tests per individual for each similar testing configuration 
#' in the object. 
#' 
#' In addition to the OTC, the \code{\link{OTC1}} and \code{\link{OTC2}} 
#' functions provide operating characteristics for other configurations 
#' corresponding to each initial group size provided by the user. For 
#' algorithms where there is only one configuration for each initial group size 
#' (non-informative two-stage hierarchical and all array testing algorithms), 
#' results for each initial group size are plotted. For algorithms where there 
#' is more than one possible configuration for each initial group size 
#' (informative two-stage hierarchical and all three-stage hierarchical 
#' algorithms), the results corresponding to the best configuration for each 
#' initial group size are plotted.
#' 
#' If a single value is provided for the \kbd{group.sz} argument in the 
#' \code{\link{OTC1}} or \code{\link{OTC2}} functions, no plot will be 
#' produced.
#' 
#' The plot is produced using the \code{ggplot2} package. Customization 
#' features from \code{ggplot2} are available once the package is loaded. 
#' Examples are shown in the 'Examples' section. 
#' 
#' @return A plot of the expected number of tests per individual for similar 
#' configurations provided in the object.
#' 
#' @author Brianna D. Hitt
#' 
#' @seealso
#' \code{\link{OTC1}} and \code{\link{OTC2}} for creating an object of class 
#' \kbd{"OTC"}.
#' 
#' @examples 
#' # Estimated running time for all examples was calculated 
#' #   using a computer with 16 GB of RAM and one core of 
#' #   an Intel i7-6500U processor. Please take this into 
#' #   account when interpreting the run time given.
#' 
#' # Find the optimal testing configuration for 
#' #   non-informative two-stage hierarchical testing.
#' res1 <- OTC1(algorithm = "D2", p = 0.01, Se = 0.99, Sp = 0.99, 
#'              group.sz = 3:100, obj.fn = c("ET", "MAR", "GR1"), 
#'              weights = matrix(data = c(1, 1), nrow = 1, ncol = 2))
#' plot(res1)
#' 
#' # Customize the plot using the ggplot2 package.
#' library(ggplot2)
#' plot(res1) + ylim(0,1) + 
#'   ggtitle("Similar configurations for Dorfman testing") + 
#'   theme(plot.title = element_text(hjust = 0.5))
#' 
#' # Find the optimal testing configuration for 
#' #   informative three-stage hierarchical testing
#' res2 <- OTC1(algorithm = "ID3", p = 0.025, 
#'              Se = c(0.95, 0.95, 0.99), Sp = c(0.96, 0.96, 0.98), 
#'              group.sz = 3:15, obj.fn = "ET", alpha = 2)
#' plot(res2)
#' 
#' # Find the optimal testing configuration for 
#' #   informative array testing without master pooling.
#' # This example takes approximately 30 seconds to run.
#' \donttest{
#' res3 <- OTC1(algorithm = "IA2", p = 0.05, alpha = 2, 
#'              Se = 0.90, Sp = 0.90, group.sz = 3:20, obj.fn = "ET")
#' plot(res3)}
#' 
#' # Find the optimal testing configuration for 
#' #   informative two-stage hierarchical testing.
#' # This example takes approximately 30 seconds to run.
#' Se <- matrix(data = c(rep(0.95, 2), rep(0.99, 2)), 
#'              nrow = 2, ncol = 2, byrow = FALSE)
#' Sp <- matrix(data = c(rep(0.96, 2), rep(0.98, 2)), 
#'              nrow = 2, ncol = 2, byrow = FALSE)
#' \donttest{
#' res4 <- OTC2(algorithm = "ID2", alpha = c(18.25, 0.75, 0.75, 0.25), 
#'                 Se = Se, Sp = Sp, group.sz = 12:20)
#' plot(res4)}
#' 
#' # Find the optimal testing configuration for 
#' #   non-informative three-stage hierarchical testing.
#' # This example takes approximately 1 minute to run.
#' Se <- matrix(data = c(rep(0.95, 6)), nrow = 2, ncol = 3)
#' Sp <- matrix(data = c(rep(0.99, 6)), nrow = 2, ncol = 3)
#' \donttest{
#' res5 <- OTC2(algorithm = "D3", p.vec = c(0.95, 0.0275, 0.0175, 0.005), 
#'                 Se = Se, Sp = Sp, group.sz = 5:20)
#' plot(res5)}
#' 
#' # Find the optimal testing configuration for
#' #   non-informative array testing with master pooling.
#' # This example takes approximately 10 seconds to run.
#' \donttest{
#' res6 <- OTC2(algorithm = "A2M", p.vec = c(0.90, 0.04, 0.04, 0.02), 
#'              Se = rep(0.99, 2), Sp = rep(0.99, 2), group.sz = 3:20)
#' plot(res6)}

plot.OTC <- function(x, ...) {
  
  if (length(x$Configs) == 1 && is.na(x$Configs)) {
    stop("No similar configurations provided. No plot will be produced.")
  }
  
  if (x$algorithm %in% c("Informative two-stage hierarchical testing")) {
    xlabel <- "Block size"
    
    ggplot(data = x$Configs, aes(x = x$Configs$N, y = x$Configs$value)) +
      geom_point() + geom_line() + 
      xlab(xlabel) + 
      ylab("Expected number of tests\n per individual\n") + 
      scale_x_continuous(breaks = pretty_breaks()) +
      theme(axis.title = element_text(size = 14), 
            axis.text = element_text(size = 12))
  } else {
    if (x$algorithm %in% c("Non-informative two-stage hierarchical testing", 
                           "Non-informative three-stage hierarchical testing", 
                           "Informative three-stage hierarchical testing")) {
      xlabel <- "Initial group size"
      
      ggplot(data = x$Configs, aes(x = x$Configs$I, y = x$Configs$value)) +
        geom_point() + geom_line() + 
        xlab(xlabel) + 
        ylab("Expected number of tests\n per individual\n") + 
        scale_x_continuous(breaks = pretty_breaks()) +
        theme(axis.title = element_text(size = 14), 
              axis.text = element_text(size = 12))
    } else {
      xlabel <- "Initial row/column size"
      
      ggplot(data = x$Configs, aes(x = x$Configs$I, y = x$Configs$value)) +
        geom_point() + geom_line() + 
        xlab(xlabel) + 
        ylab("Expected number of tests\n per individual\n") + 
        scale_x_continuous(breaks = pretty_breaks()) +
        theme(axis.title = element_text(size = 14), 
              axis.text = element_text(size = 12))
    }
  }
}
