\name{train}
\alias{train}
\alias{train.default}
\title{Fit Predictive Models over Different Tuning Parameters}
\description{
  This function sets up a grid of tuning parameters for a number of classification and regression routines,
  fits each model and calculates a resampling based performance measure. 
}
\usage{
train(x, ...)

\method{train}{default}(x, y, method = "rf",  ..., 
   metric = ifelse(is.factor(y), "Accuracy", "RMSE"),   
   trControl = trainControl(), tuneGrid = NULL, 
   tuneLength = 3)
}
\arguments{
  \item{x}{a data frame containing training data where samples are in rows and features are in columns.}
  \item{y}{a numeric or factor vector containing the outcome for each sample.}
  \item{method}{a string specifying which classification or regression model to use. Possible values are: \code{lm}, \code{rda}, \code{lda}, \code{gbm}, \code{rf},  \code{nnet}, \code{multinom}, \code{gpls}, \code{lvq}, \code{rpart}, \code{knn}, \code{pls}, \code{pam}, \code{nb},  \code{earth}, \code{treebag},
\code{svmpoly}, \code{svmradial}, \code{fda}, \code{bagEarth}, \code{bagFDA}, \code{glmboost}, \code{gamboost}, \code{blackboost},  \code{ada}, \code{ctree}, \code{cforest}, \code{sddaLDA}, \code{sddaQDA}, \code{enet} and \code{lasso}. See the Details section below.}  
  \item{\ldots}{ arguments passed to the classification or regression routine (such as \code{\link[randomForest]{randomForest}}). Errors will occur if values
         for tuning parameters are passed here.}  
  \item{metric}{a string that specifies what summary metric will be used to select the optimal model. Possible values are "RMSE" and "Rsquared" for 
     regression and "Accuracy" and "Kappa" for classification.(NOTE: If given, this argument must be named.)}         
  \item{trControl}{a list of values that define how this function acts. See \code{\link{trainControl}}. (NOTE: If given, this argument must be named.)}
  \item{tuneGrid}{a data frame with possible tuning values. The columns are named the same as the tuning parameters in each
         method preceded by a period (e.g. .decay, .lambda). See the function \code{\link{createGrid}} in this package for more details. 
         (NOTE: If given, this argument must be named.)}  
  \item{tuneLength}{an integer denoting the number of levels for each tuning parameters that should be
         generated by \code{\link{createGrid}}. (NOTE: If given, this argument must be named.)}
}
\details{
\code{train} can be used to tune models by picking the complexity parameters that are associated with the optimal resampling statistics. For particular model, a grid of parameters (if any) is created and the model is trained on slightly different data for each candidate combination of tuning parameters. Across each data set, the performance of held-out samples is calculated and the mean and standard deviation is summarized for each combination. The combination with the optimal resampling statistic is chosen as the final model and the entire training set is used to fit a final model.

Currently, the \code{train} function does not support model specification via a formula. It assumes that all of the predictors are numeric (perhaps generated by \code{\link[stats]{model.matrix}}).

A variety of models are currently available. The table below enumerates the models and the values of the \code{method} argument, as well as the complexity parameters used by \code{train}.

  
\tabular{lccc}{
\bold{Model} \tab \bold{\code{method} Value} \tab \bold{Package} \tab \bold{Tuning Parameter(s)} \cr
Recursive partitioning \tab \code{rpart} \tab \pkg{rpart} \tab \code{maxdepth} \cr  
\tab \code{ctree} \tab \pkg{party} \tab \code{mincriterion} \cr   
Boosted Trees \tab \code{gbm} \tab \pkg{gbm} \tab \code{interaction depth}, \cr
\tab \tab \tab \code{n.trees}, \code{shrinkage}  \cr
\tab \code{blackboost} \tab \pkg{mboost} \tab \code{maxdepth}, \code{mstop}\cr
\tab \code{ada} \tab \pkg{ada} \tab \code{maxdepth}, \code{iter}, \code{nu}\cr            
Boosted regression models \tab \code{glmboost} \tab \pkg{mboost} \tab \code{mstop}\cr   
\tab \code{gamboost} \tab \pkg{mboost} \tab  \code{mstop}\cr    
\tab \code{logitboost} \tab \pkg{caTools} \tab  \code{nIter}\cr    
Random forests \tab  \code{rf} \tab \pkg{randomForest} \tab \code{mtry} \cr
\tab \code{cforest} \tab \pkg{party} \tab \code{mtry} \cr   
Bagged Trees \tab \code{treebag} \tab \pkg{ipred} \tab None \cr 
Neural networks \tab  \code{nnet} \tab \pkg{nnet} \tab \code{decay}, \code{size} \cr
Partial least squares \tab \code{pls} \tab \pkg{pls}, \pkg{caret} \tab \code{ncomp} \cr
Support Vector Machines (RBF)  \tab \code{svmradial} \tab \pkg{kernlab} \tab \code{sigma}, \code{C}\cr 
Support Vector Machines (polynomial)  \tab \code{svmpoly} \tab \pkg{kernlab} \tab \code{scale}, \code{degree}, \code{C}\cr 
Relevance Vector Machines (RBF)  \tab \code{rvmradial} \tab \pkg{kernlab} \tab \code{sigma}\cr 
Relevance Vector Machines (polynomial)  \tab \code{rvmpoly} \tab \pkg{kernlab} \tab \code{scale}, \code{degree}\cr 
Least Squares Support Vector Machines (RBF)  \tab \code{lssvmradial} \tab \pkg{kernlab} \tab \code{sigma}\cr 
Gaussian Processes (RBF)  \tab \code{guassprRadial} \tab \pkg{kernlab} \tab \code{sigma}\cr 
Gaussian Processes (polynomial)  \tab \code{guassprPoly} \tab \pkg{kernlab} \tab \code{scale}, \code{degree}\cr 
Linear least squares \tab \code{lm} \tab \pkg{stats} \tab None \cr 
Multivariate adaptive regression splines \tab \code{earth} \tab \pkg{earth} \tab \code{degree}, \code{nprune} \cr
Bagged MARS \tab \code{bagEarth} \tab \pkg{caret},  \pkg{earth}\tab \code{degree}, \code{nprune} \cr  
Elastic Net \tab \code{enet} \tab \pkg{elasticnet} \tab \code{lambda}, \code{fraction} \cr
The Lasso \tab \code{enet} \tab \pkg{elasticnet} \tab \code{fraction} \cr
Linear discriminant analysis \tab \code{lda} \tab \pkg{MASS} \tab  None\cr 
Stepwise Diagonal Discriminant Analysis \tab \code{sddaLDA}, \code{sddaQDA} \tab \pkg{SDDA} \tab None \cr
Logistic/multinomial regression \tab \code{multinom} \tab \pkg{nnet} \tab \code{decay}\cr 
Regularized discriminant analysis \tab \code{rda} \tab \pkg{klaR} \tab \code{lambda}, \code{gamma} \cr
Flexible discriminant analysis (MARS) \tab \code{fda} \tab \pkg{mda}, \pkg{earth} \tab \code{degree}, \code{nprune} \cr
Bagged FDA \tab \code{bagFDA} \tab \pkg{caret},  \pkg{earth} \tab \code{degree}, \code{nprune} \cr  
k nearest neighbors \tab \code{knn3} \tab \pkg{caret} \tab \code{k} \cr 
Nearest shrunken centroids \tab \code{pam} \tab \pkg{pamr} \tab \code{threshold} \cr  
Naive Bayes \tab \code{nb} \tab \pkg{klaR} \tab \code{usekernel} \cr
Generalized partial least squares\tab \code{gpls} \tab \pkg{gpls} \tab \code{K.prov} \cr
Learned vector quantization \tab \code{lvq} \tab \pkg{class} \tab  \code{k} 
}


By default, the function \code{\link{createGrid}} is used to define the candidate values of the tuning parameters. The user can also specify their own. To do this, a data fame is created with columns for each tuning parameter in the model. The column names must be the same as those listed in the table above with a leading dot. For example, \code{ncomp} would have the column heading \code{.ncomp}. This data frame can then be passed to \code{\link{createGrid}}.

In some cases, models may require control arguments. These can be passed via the three dots argument. Note that some models can specify tuning parameters in the control objects. If specified, these values will be superseded by those given in the \code{\link{createGrid}} argument. 

The vignette entitled "caret Manual -- Model Building" has more details and examples related to this function.
}



\value{
  A list is returned of class \code{train} containing:
  \item{modelType }{an identifier of the model type.}  
  \item{results }{a data frame the training error rate and values of the tuning parameters.}
  \item{call }{the (matched) function call with dots expanded}
  \item{dots}{a list containing any ... values passed to the original call}
  \item{metric}{a string that specifies what summary metric will be used to select the optimal model.}  
  \item{trControl}{the list of control parameters.}
   \item{finalModel}{an fit object using the best parameters}
   \item{trainingData}{a data frame}
   \item{resample}{A data frame with columns for each performance
     metric. Each row corresponds to each resample. If leave-group-out
     cross-validation or out-of-bag estimation methods are requested,
     this will be \code{NULL}}
}

\author{Max Kuhn}

\seealso{\code{\link{createGrid}}, \code{\link{createFolds}}}
\examples{
data(iris)
TrainData <- iris[,1:4]
TrainClasses <- iris[,5]

knnFit1 <- train(TrainData, TrainClasses, "knn", tuneLength = 10,
   trControl = trainControl(method = "cv"))

knnFit2 <- train(TrainData, TrainClasses, "knn", tuneLength = 10, 
   trControl = trainControl(method = "boot"))


library(MASS)
nnetFit <- train(TrainData, TrainClasses, "nnet", 
   tuneLength = 2, trace = FALSE, maxit = 100)

library(mlbench)
data(BostonHousing)

# for illustration, converting factors
trainX <- model.matrix(medv ~ . - 1, BostonHousing)

lmFit <- train(trainX, BostonHousing$medv, "lm")

library(rpart)
rpartFit <- train(trainX, BostonHousing$medv, "rpart", tuneLength = 9)
}

\keyword{models}
