#' Generate Rust Code to Register Rust Functions
#'
#' This function generates Rust code to register Rust functions accessed in R
#' through .Call(). If a package's usage of .Call() functions changes, rerun
#' this function to update the \code{src/rustlib/src/registration.rs} file.
#'
#' @param pkgroot The root directory of the package.
#'
#' @return \code{NULL}, invisibly.
#' @export
#'
register_calls <- function(pkgroot=".") {
  map <- hunt_for_calls(pkgroot)
  outfilename <- file.path(pkgroot,"src","rustlib","src","registration.rs")
  outfile <- file(outfilename, open="w")
  cat(
"// Generated by cargo: do not edit by hand

// If usage of .Call()/.Kall() functions in the package's R code changes, update
// this file by rerunning \"cargo::register_calls(DIR)\", where DIR is the root
// directory of this package.

/*
// Below is commented-out skeleton code that you can copy to your
// 'src/rustlib/src/lib.rs' file. You can change the body and arguments
// names of the functions, but changing the function name necessitates
// a corresponding change in the R code.

mod registration;
use roxido::*;
", file=outfile)
  for ( x in names(map) ) {
    longjmp <- if ( map[[x]]$isKall ) "(longjmp = false)" else ""
    cat("\n#[roxido",longjmp,"]\nfn ",x,"(",sep="",file=outfile)
    a <- map[[x]]$args
    for ( i in seq_along(a) ) {
      cat(a[i],": Rval",sep="",file=outfile)
      if ( i != length(a) ) cat(", ",file=outfile)
    }
    cat(") -> Rval {\n    ",file=outfile)
    cat("Rval::nil()\n",file=outfile)
    cat("}\n",file=outfile)
  }
  cat("*/\n\n",file=outfile)
  description_file <- file.path(pkgroot, "DESCRIPTION")
  header <- gsub("YYY", length(map), gsub("XXX", read.dcf(description_file)[,'Package'],
'use roxido::*;

#[no_mangle]
extern "C" fn R_init_XXX_librust(info: *mut rbindings::DllInfo) {
    let mut call_routines = Vec::with_capacity(YYY);
    let mut _names: Vec<std::ffi::CString> = Vec::with_capacity(YYY);
'))
  footer <-
'    call_routines.push(rbindings::R_CallMethodDef {
        name: std::ptr::null(),
        fun: None,
        numArgs: 0,
    });
    unsafe {
        rbindings::R_registerRoutines(
            info,
            std::ptr::null(),
            call_routines.as_ptr(),
            std::ptr::null(),
            std::ptr::null(),
        );
        rbindings::R_useDynamicSymbols(info, 1);
        rbindings::R_forceSymbols(info, 1);
    }
    roxido::r::set_custom_panic_hook();
}'
  entry <-
'    _names.push(std::ffi::CString::new(".#@_NAME_@#").unwrap());
    call_routines.push(rbindings::R_CallMethodDef {
        name: _names.last().unwrap().as_ptr(),
        fun: unsafe { std::mem::transmute(crate::#@_NAME_@# as *const u8) },
        numArgs: #@_NARGS_@#,
    });
'
  cat(header,file=outfile)
  for ( x in names(map) ) {
    cat(gsub("#@_NARGS_@#",length(map[[x]]$args),gsub("#@_NAME_@#",x,entry)),file=outfile)
  }
  cat(footer,file=outfile)
  cat("\n",file=outfile)
  close(outfile)
  invisible()
}

hunt_for_calls <- function(pkgroot) {
  results <- list()
  engine <- function(x) {
    cl <- class(x)
    if ( cl == "call" ) {
      if ( toString(x[[1]]) %in% c(".Call",".Kall") ) {
        name <- toString(x[[2]])
        if ( grepl("\\..*", name) ) {
          name <- sub("\\.","",name)
          args <- sapply(x[-(1:2)],toString)
          okay <- grepl("^[a-z,A-Z][a-z,A-Z,0-9,_]*$",args)
          args[!okay] <- paste0("unnamed", seq_len(sum(!okay)))
          y <- list(name=toString(x[[2]]), args=sapply(x[-(1:2)],toString))
          results[[length(results)+1]] <<- list(name=name, args=args, isKall=toString(x[[1]])==".Kall")
        }
      }
      lapply(x[-1], engine)
    } else if ( cl %in% c("<-","{","(","for","if","while","repeat") ) {
      lapply(as.list(x)[-1], engine)
    } else if ( cl %in% c("numeric","character","logical","integer","name","pairlist","srcref","NULL") ) {
      NULL
    } else warning(paste0("Unknown class: ",cl))
  }
  description_file <- file.path(pkgroot, "DESCRIPTION")
  r_dir <- file.path(pkgroot, "R")
  if ( ! file.exists(description_file) || ! dir.exists(r_dir) ) {
    stop(sprintf("Oops, '%s' does not appear to be a package root directory.", pkgroot))
  }
  files <- list.files(r_dir, full.names=TRUE)
  env <- new.env()
  for ( f in files ) source(f, local=env)
  funcs <- mget(ls(env), envir=env)
  map <- list()
  w <- sapply(funcs, is.function)
  if ( length(w) == 0 ) return(map)
  funcs <- funcs[w]
  for ( func in funcs ) {
    engine(body(func))
  }
  for ( r in results ) {
    if ( ! is.null(map[[r$name]]) ) {
      if ( length(map[[r$name]]$args) != length(r$args) ) {
        stop(paste0("Inconsistent number of arguments for function ",r$name))
      }
      pattern <- "^unnamed[0-9]+$"
      if ( sum(grepl(pattern, r$args)) < sum(grepl(pattern, map[[r$name]])) ) {
        map[[r$name]] <- list(args=r$args, isKall=r$isKall)
      }
    } else {
      map[[r$name]] <- list(args=r$args, isKall=r$isKall)
    }
  }
  map
}
