% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cgalMesh.R
\name{cgalMesh}
\alias{cgalMesh}
\title{R6 class to represent a CGAL mesh}
\description{
R6 class to represent a CGAL mesh.
}
\examples{

## ------------------------------------------------
## Method `cgalMesh$new`
## ------------------------------------------------

library(cgalMeshes)
meshFile <- system.file(
  "extdata", "bigPolyhedron.off", package = "cgalMeshes"
)
mesh <- cgalMesh$new(meshFile)
rglmesh <- mesh$getMesh()
\donttest{library(rgl)
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglmesh, color = "tomato")
plotEdges(
  mesh$getVertices(), mesh$getEdges(), color = "darkred"
)}

# this one has colors: ####
\donttest{meshFile <- system.file(
  "extdata", "pentagrammicDipyramid.ply", package = "cgalMeshes"
)
mesh <- cgalMesh$new(meshFile)
rmesh <- mesh$getMesh()
library(rgl)
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.85)
shade3d(rmesh, meshColor = "faces")}

## ------------------------------------------------
## Method `cgalMesh$area`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$area()

## ------------------------------------------------
## Method `cgalMesh$boundingBox`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
rmesh <- cyclideMesh(a = 97, c = 32, mu = 57)
mesh <- cgalMesh$new(rmesh)
bbox <- mesh$boundingBox()
bxmesh <- isoCuboidMesh(bbox[["lcorner"]], bbox[["ucorner"]])
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(0, -60)
shade3d(rmesh, color = "gold")
wire3d(bxmesh, color = "black")}

## ------------------------------------------------
## Method `cgalMesh$boundsVolume`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$boundsVolume() # TRUE
mesh$reverseOrientation()
mesh$boundsVolume() # TRUE

## ------------------------------------------------
## Method `cgalMesh$CatmullClark`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
hopfMesh <- HopfTorusMesh(nu = 80, nv = 40)
mesh <- cgalMesh$new(hopfMesh)
mesh$CatmullClark(iterations = 2)
mesh$computeNormals()
rmesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 800, 400))
mfrow3d(1, 2)
view3d(0, 0, zoom = 0.9)
shade3d(hopfMesh, color = "red")
wire3d(hopfMesh, color = "black")
next3d()
view3d(0, 0, zoom = 0.9)
shade3d(rmesh, color = "red")
wire3d(rmesh, color = "black")}

## ------------------------------------------------
## Method `cgalMesh$centroid`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(icosahedron3d())
mesh$centroid()}

## ------------------------------------------------
## Method `cgalMesh$clip`
## ------------------------------------------------

# cube clipped to sphere ####
\donttest{library(cgalMeshes)
library(rgl)
mesh    <- cgalMesh$new(cube3d())$triangulate()
clipper <- cgalMesh$new(sphereMesh(r= sqrt(2)))
mesh$assignFaceColors("blue")
clipper$assignFaceColors("red")
meshes <- mesh$clip(clipper, clipVolume = TRUE)
mesh1 <- meshes[[1]]
mesh2 <- meshes[[2]]
mesh2$computeNormals()
rglmesh1 <- mesh1$getMesh()
rglmesh2 <- mesh2$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(45, 45, zoom = 0.9)
shade3d(rglmesh1, meshColor = "faces")
shade3d(rglmesh2, meshColor = "faces")}

# Togliatti surface clipped to a ball ####
\donttest{library(rmarchingcubes)
library(rgl)
library(cgalMeshes)
# Togliatti surface equation: f(x,y,z) = 0
f <- function(x, y, z) {
  64*(x-1) *
    (x^4 - 4*x^3 - 10*x^2*y^2 - 4*x^2 + 16*x - 20*x*y^2 + 5*y^4 + 16 - 20*y^2) - 
    5*sqrt(5-sqrt(5))*(2*z - sqrt(5-sqrt(5))) * 
    (4*(x^2 + y^2 - z^2) + (1 + 3*sqrt(5)))^2
}
# grid
n <- 200L
x <- y <- seq(-5, 5, length.out = n)
z <- seq(-4, 4, length.out = n)
Grid <- expand.grid(X = x, Y = y, Z = z)
# calculate voxel
voxel <- array(with(Grid, f(X, Y, Z)), dim = c(n, n, n))
# calculate isosurface
contour_shape <- contour3d(
  griddata = voxel, level = 0, x = x, y = y, z = z
)
# make rgl mesh (plotted later)
rglMesh <- tmesh3d(
  vertices = t(contour_shape[["vertices"]]),
  indices  = t(contour_shape[["triangles"]]),
  normals  = contour_shape[["normals"]],
  homogeneous = FALSE
)
# make CGAL mesh
mesh <- cgalMesh$new(rglMesh)
# clip to sphere of radius 4.8
sphere <- sphereMesh(r = 4.8)
clipper <- cgalMesh$new(sphere)
mesh$clip(clipper, clipVolume = FALSE)
rglClippedMesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 900, 450))
mfrow3d(1L, 2L)
view3d(0, -70, zoom = 0.8)
shade3d(rglMesh, color = "firebrick")
next3d()
view3d(0, -70, zoom = 0.8)
shade3d(rglClippedMesh, color = "firebrick")
shade3d(sphere, color = "yellow", alpha = 0.15)}

## ------------------------------------------------
## Method `cgalMesh$clipToPlane`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
rmesh <- sphereMesh()
mesh <- cgalMesh$new(rmesh)
nfaces <- nrow(mesh$getFaces())
if(require("randomcoloR")) {
  colors <- 
    randomColor(nfaces, hue = "random", luminosity = "dark")
} else {
  colors <- rainbow(nfaces)
}
mesh$assignFaceColors(colors)
meshes <- mesh$clipToPlane(
  planePoint  = c(0, 0, 0), 
  planeNormal = c(0, 0, 1), 
  clipVolume = TRUE
)
mesh1 <- meshes[[1]]
mesh2 <- meshes[[2]]
mesh1$computeNormals()
rClippedMesh1 <- mesh1$getMesh()
rClippedMesh2 <- mesh2$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(70, 0)
shade3d(rClippedMesh1, meshColor = "faces")
shade3d(rClippedMesh2, color = "orange")}

## ------------------------------------------------
## Method `cgalMesh$clipToIsoCuboid`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
rmesh <- HopfTorusMesh(nu = 200, nv = 200)
mesh <- cgalMesh$new(rmesh)
mesh$assignFaceColors("orangered")
lcorner <- c(-7, -7, -5)
ucorner <- c(7, 6, 5)
bxmesh <- isoCuboidMesh(lcorner, ucorner)
mesh$clipToIsoCuboid(
  lcorner, ucorner, clipVolume = FALSE
)
mesh$computeNormals()
rClippedMesh <- mesh$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(-40, 0)
shade3d(rClippedMesh, meshColor = "faces")
shade3d(bxmesh, color = "cyan", alpha = 0.3)}

## ------------------------------------------------
## Method `cgalMesh$connectedComponents`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rmarchingcubes)
# isosurface function (slice of a seven-dimensional toratope)
f <- function(x, y, z, a) {
  (sqrt(
    (sqrt((sqrt((x*sin(a))^2 + (z*cos(a))^2) - 5)^2 + (y*sin(a))^2) - 2.5)^2 + 
      (x*cos(a))^2) - 1.25
  )^2 + (sqrt((sqrt((z*sin(a))^2 + (y*cos(a))^2) - 2.5)^2) - 1.25)^2
}
# make grid
n <- 200L
x <- seq(-10, 10, len = n)
y <- seq(-10, 10, len = n)
z <- seq(-10, 10, len = n)
Grid <- expand.grid(X = x, Y = y, Z = z)
# compute isosurface
voxel <- array(with(Grid, f(X, Y, Z, a = pi/2)), dim = c(n, n, n))
isosurface <- contour3d(voxel, level = 0.25, x = x, y = y, z = z)
# make CGAL mesh
mesh <- cgalMesh$new(
  vertices = isosurface[["vertices"]], 
  faces = isosurface[["triangles"]],
  normals = isosurface[["normals"]]
)
# connected components
components <- mesh$connectedComponents()
ncc <- length(components)
# plot
library(rgl)
colors <- rainbow(ncc)
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(30, 50)
for(i in 1L:ncc) {
  rglMesh <- components[[i]]$getMesh()
  shade3d(rglMesh, color = colors[i])
}}

## ------------------------------------------------
## Method `cgalMesh$convexParts`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(pentagrammicPrism)$triangulate()
cxparts <- mesh$convexParts()
ncxparts <- length(cxparts)
colors <- hcl.colors(ncxparts, palette = "plasma")
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(20, -20, zoom = 0.8)
for(i in 1L:ncxparts) {
  cxmesh <- cxparts[[i]]$getMesh()
  shade3d(cxmesh, color = colors[i])
}}

## ------------------------------------------------
## Method `cgalMesh$copy`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())
tmesh <- mesh$copy()$triangulate()
tmesh$isTriangle() # TRUE
mesh$isTriangle() # FALSE

## ------------------------------------------------
## Method `cgalMesh$distance`
## ------------------------------------------------

# cube example ####
library(cgalMeshes)
mesh <- cgalMesh$new(rgl::cube3d())$triangulate()
points <- rbind(
  c(0, 0, 0),
  c(1, 1, 1)
)
mesh$distance(points) # should be 1 and 0

# cyclide example ####
\donttest{library(cgalMeshes)
a <- 100; c <- 30; mu <- 80
mesh <- cgalMesh$new(cyclideMesh(a, c, mu, nu = 100L, nv = 100L))
O2 <- c(c, 0, 0)
# should be a - mu = 20 (see ?cyclideMesh):
mesh$distance(O2)}    

## ------------------------------------------------
## Method `cgalMesh$DooSabin`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
hopfMesh <- HopfTorusMesh(nu = 80, nv = 40)
mesh <- cgalMesh$new(hopfMesh)
mesh$DooSabin(iterations = 2)
mesh$triangulate()
mesh$computeNormals()
rmesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 800, 400))
mfrow3d(1, 2)
view3d(0, 0, zoom = 0.9)
shade3d(hopfMesh, color = "brown")
wire3d(hopfMesh, color = "black")
next3d()
view3d(0, 0, zoom = 0.9)
shade3d(rmesh, color = "brown")
wire3d(rmesh, color = "black")}

## ------------------------------------------------
## Method `cgalMesh$fair`
## ------------------------------------------------

\donttest{library(cgalMeshes)
rglHopf <- HopfTorusMesh(nu = 100, nv = 100)
hopf <- cgalMesh$new(rglHopf)
# squared norms of the vertices
normsq <- apply(hopf$getVertices(), 1L, crossprod)
# fair the region where the squared norm is > 19
indices <- which(normsq > 19)
hopf$fair(indices)
rglHopf_faired <- hopf$getMesh()
# plot
library(rgl)
open3d(windowRect = 50 + c(0, 0, 900, 450))
mfrow3d(1L, 2L)
view3d(0, 0, zoom = 0.8)
shade3d(rglHopf, color = "orangered")
next3d()
view3d(0, 0, zoom = 0.8)
shade3d(rglHopf_faired, color = "orangered")}

## ------------------------------------------------
## Method `cgalMesh$fillBoundaryHole`
## ------------------------------------------------

library(cgalMeshes)
library(rgl)
# make a sphere
sphere <- sphereMesh()
mesh <- cgalMesh$new(sphere)
# make a hole in this sphere
mesh$clipToPlane(
  planePoint  = c(0.5, 0, 0),
  planeNormal = c(1, 0, 0),
  clipVolume  = FALSE
)
mesh$computeNormals()
rmesh <- mesh$getMesh()
# fill the hole
hole <- mesh$fillBoundaryHole(1, fair = TRUE)
hole$computeNormals()
rhole <- hole$getMesh()
# plot
\donttest{open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(30, 30)
shade3d(rmesh, color = "red")
shade3d(rhole, color = "blue")}

## ------------------------------------------------
## Method `cgalMesh$filterMesh`
## ------------------------------------------------

\donttest{library(rgl)
library(cgalMeshes)
rmesh <- HopfTorusMesh(nu = 80, nv = 60)
mesh <- cgalMesh$new(rmesh)
areas <- mesh$getFacesInfo()[, "area"]
bigFaces <- which(areas > 1)
meshes <- mesh$filterMesh(bigFaces)
rmesh1 <- meshes[[1]]$getMesh()
rmesh2 <- meshes[[2]]$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(0, 0)
shade3d(rmesh1, color = "red")
shade3d(rmesh2, color = "blue")
wire3d(rmesh)}

## ------------------------------------------------
## Method `cgalMesh$geoDists`
## ------------------------------------------------

\donttest{# torus ####
library(cgalMeshes)
library(rgl)
rglmesh <- torusMesh(R = 3, r = 2, nu = 90, nv = 60)
mesh <- cgalMesh$new(rglmesh)
# estimated geodesic distances
geodists <- mesh$geoDists(1L)
# normalization to (0, 1)
geodists <- geodists / max(geodists)
# color each vertex according to its geodesic distance from the source
fcolor <- colorRamp(viridisLite::turbo(200L))
colors <- fcolor(geodists)
colors <- rgb(colors[, 1L], colors[, 2L], colors[, 3L], maxColorValue = 255)
rglmesh[["material"]] <- list("color" = colors)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(rglmesh)
wire3d(rglmesh, color = "black")
if(!rgl.useNULL()) {
  play3d(spin3d(axis = c(1, 1, 1), rpm = 5), duration = 20)  
}}

# a trefoil knot (taken from `?rgl::cylinder3d`) ####
\donttest{library(cgalMeshes)
library(rgl)
theta <- seq(0, 2*pi, length.out = 50L)
knot <- cylinder3d(
  center = cbind(
    sin(theta) + 2*sin(2*theta), 
    2*sin(3*theta), 
    cos(theta) - 2*cos(2*theta)),
  e1 = cbind(
    cos(theta) + 4*cos(2*theta), 
    6*cos(3*theta), 
    sin(theta) + 4*sin(2*theta)),
  radius = 0.8, 
  closed = TRUE)
knot <- subdivision3d(knot, depth = 2)
mesh <- cgalMesh$new(knot)$triangulate()
rglmesh <- mesh$getMesh()
# estimated geodesic distances
geodists <- mesh$geoDists(1L)
# normalization to (0, 1)
geodists <- geodists / max(geodists)
# color each vertex according to its geodesic distance from the source
fcolor <- colorRamp(viridisLite::inferno(200L))
colors <- fcolor(geodists)
colors <- rgb(colors[, 1L], colors[, 2L], colors[, 3L], maxColorValue = 255)
rglmesh[["material"]] <- list("color" = colors)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(rglmesh)
if(!rgl.useNULL()) {
  play3d(spin3d(axis = c(1, 1, 0), rpm = 5), duration = 20)  
}}

## ------------------------------------------------
## Method `cgalMesh$getBorders`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
# isosurface f=0
f <- function(x, y, z) {
  sin_x <- sin(x)
  sin_y <- sin(y)
  sin_z <- sin(z)
  cos_x <- cos(x)
  cos_y <- cos(y)
  cos_z <- cos(z)
  d <- sqrt(
    (-sin_x * sin_y + cos_x * cos_z) ** 2
    + (-sin_y * sin_z + cos_y * cos_x) ** 2
    + (-sin_z * sin_x + cos_z * cos_y) ** 2
  )
  (
    cos(
      x - (-sin_x * sin_y + cos_x * cos_z) / d
    )
    * sin(
      y - (-sin_y * sin_z + cos_y * cos_x) / d
    )
    + cos(
      y - (-sin_y * sin_z + cos_y * cos_x) / d
    )
    * sin(
      z - (-sin_z * sin_x + cos_z * cos_y)/ d
    )
    + cos(
      z - (-sin_z * sin_x + cos_z * cos_y) / d
    )
    * sin(
      x - (-sin_x * sin_y + cos_x * cos_z) / d
    )
  ) * (
    (
      cos(
        x + (-sin_x * sin_y + cos_x * cos_z) / d
      )
      * sin(
        y + (-sin_y * sin_z + cos_y * cos_x) / d
      )
      + cos(
        y + (-sin_y * sin_z + cos_y * cos_x) / d
      )
      * sin(
        z + (-sin_z * sin_x + cos_z * cos_y) / d
      )
      + cos(
        z + (-sin_z * sin_x + cos_z * cos_y) / d
      )
      * sin(
        x + (-sin_x * sin_y + cos_x * cos_z) / d
      )
    )
  )
}
# construct the isosurface f=0
ngrid <- 200L
x <- y <- z <- seq(-8.1, 8.1, length.out = ngrid)
Grid <- expand.grid(X = x, Y = y, Z = z)
voxel <- array(
  with(Grid, f(X, Y, Z)), dim = c(ngrid, ngrid, ngrid)
)
library(rmarchingcubes)
contour_shape <- contour3d(
  griddata = voxel, level = 0,
  x = x, y = y, z = z
)
# make mesh
mesh <- cgalMesh$new(
  list(
    "vertices" = contour_shape[["vertices"]],
    "faces"    = contour_shape[["triangles"]]
  )
)
# clip the mesh to the ball of radius 8
spheremesh <- cgalMesh$new(sphereMesh(r = 8))
mesh$clip(spheremesh, clipVolume = FALSE)
# compute normals
mesh$computeNormals()
# we will plot the borders
borders <- mesh$getBorders()
# plot
rmesh <- mesh$getMesh()
open3d(windowRect = c(50, 50, 562, 562), zoom = 0.7)
shade3d(rmesh, color = "darkred")
vertices <- mesh$getVertices()
for(border in borders){
  plotEdges(
    vertices, border[, c("v1", "v2")], color = "gold",
    lwd = 3, edgesAsTubes = FALSE, verticesAsSpheres = FALSE
  )
}}

## ------------------------------------------------
## Method `cgalMesh$getEdges`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(dodecahedron3d())
head(mesh$getEdges())

## ------------------------------------------------
## Method `cgalMesh$getMesh`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$getMesh()

## ------------------------------------------------
## Method `cgalMesh$intersection`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# intersection
imesh <- mesh1$intersection(mesh2)
rglimesh <- imesh$getMesh()
# extract edges for plotting
extEdges <- exteriorEdges(imesh$getEdges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglimesh, color = "red")
plotEdges(imesh$getVertices(), extEdges)
shade3d(rglmesh1, color = "yellow", alpha = 0.2)
shade3d(rglmesh2, color = "cyan", alpha = 0.2)}

## ------------------------------------------------
## Method `cgalMesh$isotropicRemeshing`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(HopfTorusMesh(nu = 80, nv = 50))
mesh$isotropicRemeshing(targetEdgeLength = 0.7)
# squared norms of the vertices
normsq <- apply(mesh$getVertices(), 1L, crossprod)
# fair the region where the squared norm is > 19
mesh$fair(which(normsq > 19))
# plot
mesh$computeNormals()
rmesh <- mesh$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(0, 0)
shade3d(rmesh, color = "maroon")
wire3d(rmesh)}

## ------------------------------------------------
## Method `cgalMesh$isOutwardOriented`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$isOutwardOriented() # TRUE
mesh$reverseOrientation()
mesh$isOutwardOriented() # FALSE

## ------------------------------------------------
## Method `cgalMesh$isTriangle`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())
mesh$isTriangle()

## ------------------------------------------------
## Method `cgalMesh$LoopSubdivision`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
hopfMesh <- HopfTorusMesh(nu = 80, nv = 40)
mesh <- cgalMesh$new(hopfMesh)
mesh$LoopSubdivision(iterations = 2)
mesh$computeNormals()
rmesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 800, 400))
mfrow3d(1, 2)
view3d(0, 0, zoom = 0.9)
shade3d(hopfMesh, color = "gold")
wire3d(hopfMesh, color = "black")
next3d()
view3d(0, 0, zoom = 0.9)
shade3d(rmesh, color = "gold")
wire3d(rmesh, color = "black")}

## ------------------------------------------------
## Method `cgalMesh$merge`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
mesh1 <- cgalMesh$new(sphereMesh())
mesh1$assignFaceColors("red")
mesh2 <- cgalMesh$new(sphereMesh(x = 3))
mesh2$assignFaceColors("blue")
mesh1$merge(mesh2)
rmesh <- mesh1$getMesh()
open3d(windowRect = c(50, 50, 562, 562))
shade3d(rmesh, meshColor = "faces")}

## ------------------------------------------------
## Method `cgalMesh$orientToBoundVolume`
## ------------------------------------------------

# two disjoint tetrahedra ####
vertices <- rbind(
  c(0, 0, 0),
  c(2, 2, 0),
  c(2, 0, 2),
  c(0, 2, 2),
  c(3, 3, 3),
  c(5, 5, 3),
  c(5, 3, 5),
  c(3, 5, 5)
)
faces <- rbind(
  c(3, 2, 1),
  c(3, 4, 2),
  c(1, 2, 4),
  c(4, 3, 1),
  c(5, 6, 7),
  c(6, 8, 7),
  c(8, 6, 5),
  c(5, 7, 8)
)
mesh <- cgalMesh$new(vertices = vertices, faces = faces)
mesh$boundsVolume() # FALSE
mesh$orientToBoundVolume()
mesh$boundsVolume() # TRUE

## ------------------------------------------------
## Method `cgalMesh$reverseOrientation`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$isOutwardOriented() # TRUE
mesh$reverseOrientation()
mesh$isOutwardOriented() # FALSE

## ------------------------------------------------
## Method `cgalMesh$selfIntersects`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(dodecahedron3d())
mesh$selfIntersects()

## ------------------------------------------------
## Method `cgalMesh$sharpEdges`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
# astroidal ellipsoid
f <- function(u, v) {
  rbind(
    cos(u)^3 * cos(v)^3,
    sin(u)^3 * cos(v)^3,
    sin(v)^3
  )
}
rmesh <- parametricMesh(
  f, urange = c(0, 2*pi), vrange = c(0, 2*pi), 
  periodic = c(TRUE, TRUE), nu = 120, nv = 110
)
mesh <- cgalMesh$new(rmesh)
sharpEdges <- mesh$sharpEdges(30)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(addNormals(rmesh), color = "chartreuse")
plotEdges(
  mesh$getVertices(), sharpEdges[, c("v1", "v2")], 
  edgesAsTubes = FALSE, lwd = 5, verticesAsSpheres = FALSE
)}

## ------------------------------------------------
## Method `cgalMesh$Sqrt3Subdivision`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
hopfMesh <- HopfTorusMesh(nu = 80, nv = 40)
mesh <- cgalMesh$new(hopfMesh)
mesh$Sqrt3Subdivision(iterations = 2)
mesh$computeNormals()
rmesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 800, 400))
mfrow3d(1, 2)
view3d(0, 0, zoom = 0.9)
shade3d(hopfMesh, color = "cyan")
wire3d(hopfMesh, color = "black")
next3d()
view3d(0, 0, zoom = 0.9)
shade3d(rmesh, color = "cyan")
wire3d(rmesh, color = "black")}

## ------------------------------------------------
## Method `cgalMesh$subtract`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# assign colors
mesh1$assignFaceColors("red")
mesh2$assignFaceColors("navy")
# difference
mesh <- mesh1$subtract(mesh2)
rglmesh <- mesh$getMesh()
# extract edges for plotting
extEdges <- exteriorEdges(mesh$getEdges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglmesh, meshColor = "faces")
plotEdges(mesh$getVertices(), extEdges)
shade3d(rglmesh2, color = "cyan", alpha = 0.2)}

## ------------------------------------------------
## Method `cgalMesh$triangulate`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())
mesh$isTriangle() # FALSE
# warning: triangulating the mesh modifies it
mesh$triangulate()
mesh$isTriangle() # TRUE

## ------------------------------------------------
## Method `cgalMesh$union`
## ------------------------------------------------

\donttest{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# assign a color to the faces; they will be retrieved in the union
mesh1$assignFaceColors("yellow")
mesh2$assignFaceColors("navy")
# union
umesh <- mesh1$union(mesh2)
rglumesh <- umesh$getMesh()
# extract edges for plotting
extEdges <- exteriorEdges(umesh$getEdges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglumesh, meshColor = "faces")
plotEdges(umesh$getVertices(), extEdges)}

## ------------------------------------------------
## Method `cgalMesh$volume`
## ------------------------------------------------

library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$volume()

## ------------------------------------------------
## Method `cgalMesh$whereIs`
## ------------------------------------------------

library(cgalMeshes)
mesh <- cgalMesh$new(sphereMesh())
pt1 <- c(0, 0, 0) # inside
pt2 <- c(2, 0, 0) # outside
mesh$whereIs(rbind(pt1, pt2))
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-cgalMesh-new}{\code{cgalMesh$new()}}
\item \href{#method-cgalMesh-print}{\code{cgalMesh$print()}}
\item \href{#method-cgalMesh-area}{\code{cgalMesh$area()}}
\item \href{#method-cgalMesh-assignFaceColors}{\code{cgalMesh$assignFaceColors()}}
\item \href{#method-cgalMesh-assignFaceScalars}{\code{cgalMesh$assignFaceScalars()}}
\item \href{#method-cgalMesh-assignNormals}{\code{cgalMesh$assignNormals()}}
\item \href{#method-cgalMesh-assignVertexColors}{\code{cgalMesh$assignVertexColors()}}
\item \href{#method-cgalMesh-assignVertexScalars}{\code{cgalMesh$assignVertexScalars()}}
\item \href{#method-cgalMesh-boundingBox}{\code{cgalMesh$boundingBox()}}
\item \href{#method-cgalMesh-boundsVolume}{\code{cgalMesh$boundsVolume()}}
\item \href{#method-cgalMesh-CatmullClark}{\code{cgalMesh$CatmullClark()}}
\item \href{#method-cgalMesh-centroid}{\code{cgalMesh$centroid()}}
\item \href{#method-cgalMesh-clip}{\code{cgalMesh$clip()}}
\item \href{#method-cgalMesh-clipToPlane}{\code{cgalMesh$clipToPlane()}}
\item \href{#method-cgalMesh-clipToIsoCuboid}{\code{cgalMesh$clipToIsoCuboid()}}
\item \href{#method-cgalMesh-computeNormals}{\code{cgalMesh$computeNormals()}}
\item \href{#method-cgalMesh-connectedComponents}{\code{cgalMesh$connectedComponents()}}
\item \href{#method-cgalMesh-convexParts}{\code{cgalMesh$convexParts()}}
\item \href{#method-cgalMesh-copy}{\code{cgalMesh$copy()}}
\item \href{#method-cgalMesh-distance}{\code{cgalMesh$distance()}}
\item \href{#method-cgalMesh-DooSabin}{\code{cgalMesh$DooSabin()}}
\item \href{#method-cgalMesh-facesAroundVertex}{\code{cgalMesh$facesAroundVertex()}}
\item \href{#method-cgalMesh-fair}{\code{cgalMesh$fair()}}
\item \href{#method-cgalMesh-fillBoundaryHole}{\code{cgalMesh$fillBoundaryHole()}}
\item \href{#method-cgalMesh-fillFaceColors}{\code{cgalMesh$fillFaceColors()}}
\item \href{#method-cgalMesh-filterMesh}{\code{cgalMesh$filterMesh()}}
\item \href{#method-cgalMesh-fixManifoldness}{\code{cgalMesh$fixManifoldness()}}
\item \href{#method-cgalMesh-geoDists}{\code{cgalMesh$geoDists()}}
\item \href{#method-cgalMesh-getBorders}{\code{cgalMesh$getBorders()}}
\item \href{#method-cgalMesh-getEdges}{\code{cgalMesh$getEdges()}}
\item \href{#method-cgalMesh-getFaces}{\code{cgalMesh$getFaces()}}
\item \href{#method-cgalMesh-getFacesInfo}{\code{cgalMesh$getFacesInfo()}}
\item \href{#method-cgalMesh-getFaceColors}{\code{cgalMesh$getFaceColors()}}
\item \href{#method-cgalMesh-getFaceScalars}{\code{cgalMesh$getFaceScalars()}}
\item \href{#method-cgalMesh-getVertexColors}{\code{cgalMesh$getVertexColors()}}
\item \href{#method-cgalMesh-getVertexScalars}{\code{cgalMesh$getVertexScalars()}}
\item \href{#method-cgalMesh-getNormals}{\code{cgalMesh$getNormals()}}
\item \href{#method-cgalMesh-getMesh}{\code{cgalMesh$getMesh()}}
\item \href{#method-cgalMesh-getVertices}{\code{cgalMesh$getVertices()}}
\item \href{#method-cgalMesh-HausdorffApproximate}{\code{cgalMesh$HausdorffApproximate()}}
\item \href{#method-cgalMesh-HausdorffEstimate}{\code{cgalMesh$HausdorffEstimate()}}
\item \href{#method-cgalMesh-intersection}{\code{cgalMesh$intersection()}}
\item \href{#method-cgalMesh-isClosed}{\code{cgalMesh$isClosed()}}
\item \href{#method-cgalMesh-isotropicRemeshing}{\code{cgalMesh$isotropicRemeshing()}}
\item \href{#method-cgalMesh-isOutwardOriented}{\code{cgalMesh$isOutwardOriented()}}
\item \href{#method-cgalMesh-isTriangle}{\code{cgalMesh$isTriangle()}}
\item \href{#method-cgalMesh-isValid}{\code{cgalMesh$isValid()}}
\item \href{#method-cgalMesh-isValidFaceGraph}{\code{cgalMesh$isValidFaceGraph()}}
\item \href{#method-cgalMesh-isValidHalfedgeGraph}{\code{cgalMesh$isValidHalfedgeGraph()}}
\item \href{#method-cgalMesh-isValidPolygonMesh}{\code{cgalMesh$isValidPolygonMesh()}}
\item \href{#method-cgalMesh-LoopSubdivision}{\code{cgalMesh$LoopSubdivision()}}
\item \href{#method-cgalMesh-merge}{\code{cgalMesh$merge()}}
\item \href{#method-cgalMesh-orientToBoundVolume}{\code{cgalMesh$orientToBoundVolume()}}
\item \href{#method-cgalMesh-removeSelfIntersections}{\code{cgalMesh$removeSelfIntersections()}}
\item \href{#method-cgalMesh-reverseOrientation}{\code{cgalMesh$reverseOrientation()}}
\item \href{#method-cgalMesh-sample}{\code{cgalMesh$sample()}}
\item \href{#method-cgalMesh-selfIntersects}{\code{cgalMesh$selfIntersects()}}
\item \href{#method-cgalMesh-sharpEdges}{\code{cgalMesh$sharpEdges()}}
\item \href{#method-cgalMesh-Sqrt3Subdivision}{\code{cgalMesh$Sqrt3Subdivision()}}
\item \href{#method-cgalMesh-subtract}{\code{cgalMesh$subtract()}}
\item \href{#method-cgalMesh-triangulate}{\code{cgalMesh$triangulate()}}
\item \href{#method-cgalMesh-union}{\code{cgalMesh$union()}}
\item \href{#method-cgalMesh-volume}{\code{cgalMesh$volume()}}
\item \href{#method-cgalMesh-whereIs}{\code{cgalMesh$whereIs()}}
\item \href{#method-cgalMesh-writeMeshFile}{\code{cgalMesh$writeMeshFile()}}
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-new"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-new}{}}}
\subsection{Method \code{new()}}{
Creates a new \code{cgalMesh} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$new(mesh, vertices, faces, normals = NULL, clean = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh}}{there are four possibilities for this argument: it can be 
missing, in which case the arguments \code{vertices} and \code{faces} 
must be given, or it can be the path to a mesh file (accepted formats: 
\code{off}, \code{obj}, \code{stl}, \code{ply}, \code{ts}, \code{vtp}),
or it can be a \strong{rgl} mesh (i.e. a \code{mesh3d} object), or it 
can be a list containing (at least) the fields \code{vertices} 
(numeric matrix with three columns) and \code{faces} (matrix of 
integers or list of vectors of integers), and optionally a field 
\code{normals} (numeric matrix with three columns); if 
this argument is a \strong{rgl} mesh containing some colors, these 
colors will be assigned to the created \code{cgalMesh} object}

\item{\code{vertices}}{if \code{mesh} is missing, must be a numeric matrix with 
three columns}

\item{\code{faces}}{if \code{mesh} is missing, must be either a matrix of 
integers (each row gives the vertex indices of a face) or a list of 
vectors of integers (each one gives the vertex indices of a face)}

\item{\code{normals}}{if \code{mesh} is missing, must be \code{NULL} or a 
numeric matrix with three columns and as many rows as vertices}

\item{\code{clean}}{Boolean value indicating whether to clean the mesh (merge 
duplicated vertices and duplicated faces, remove isolated vertices); 
set to \code{FALSE} if you know your mesh is already clean}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
meshFile <- system.file(
  "extdata", "bigPolyhedron.off", package = "cgalMeshes"
)
mesh <- cgalMesh$new(meshFile)
rglmesh <- mesh$getMesh()
library(rgl)
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglmesh, color = "tomato")
plotEdges(
  mesh$getVertices(), mesh$getEdges(), color = "darkred"
)

# this one has colors: ####
meshFile <- system.file(
  "extdata", "pentagrammicDipyramid.ply", package = "cgalMeshes"
)
mesh <- cgalMesh$new(meshFile)
rmesh <- mesh$getMesh()
library(rgl)
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.85)
shade3d(rmesh, meshColor = "faces")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-print"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-print}{}}}
\subsection{Method \code{print()}}{
Print a \code{cgalMesh} object.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$print(...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{...}}{ignored}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
No value returned, just prints some information about the mesh.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-area"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-area}{}}}
\subsection{Method \code{area()}}{
Compute the area of the mesh. The mesh must be triangle 
  and must not self-intersect.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$area()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A number, the mesh area.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$area()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-assignFaceColors"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-assignFaceColors}{}}}
\subsection{Method \code{assignFaceColors()}}{
Assign colors (or any character strings) to the faces of 
  the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$assignFaceColors(colors)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{colors}}{a character vector whose length equals the number 
of faces, or a single character string to be assigned to each face}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The current \code{cgalMesh} object, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-assignFaceScalars"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-assignFaceScalars}{}}}
\subsection{Method \code{assignFaceScalars()}}{
Assign scalars to the faces of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$assignFaceScalars(scalars)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{scalars}}{a numeric vector whose length equals the number of faces}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The current \code{cgalMesh} object, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-assignNormals"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-assignNormals}{}}}
\subsection{Method \code{assignNormals()}}{
Assign per-vertex normals to the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$assignNormals(normals)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{normals}}{a numeric matrix with three columns and as many rows as 
the number of vertices}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The current \code{cgalMesh} object, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-assignVertexColors"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-assignVertexColors}{}}}
\subsection{Method \code{assignVertexColors()}}{
Assign colors (or any character strings) to the vertices of 
  the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$assignVertexColors(colors)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{colors}}{a character vector whose length equals the number of vertices}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The current \code{cgalMesh} object, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-assignVertexScalars"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-assignVertexScalars}{}}}
\subsection{Method \code{assignVertexScalars()}}{
Assign scalars to the vertices of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$assignVertexScalars(scalars)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{scalars}}{a numeric vector whose length equals the number 
of vertices}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The current \code{cgalMesh} object, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-boundingBox"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-boundingBox}{}}}
\subsection{Method \code{boundingBox()}}{
Bounding box of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$boundingBox()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A list containing the smallest corner point and the largest 
  corner point of the bounding box, named \code{lcorner} and 
 \code{ucorner} respectively. Use \code{\link{isoCuboidMesh}} to get a 
 mesh of this bounding box.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
rmesh <- cyclideMesh(a = 97, c = 32, mu = 57)
mesh <- cgalMesh$new(rmesh)
bbox <- mesh$boundingBox()
bxmesh <- isoCuboidMesh(bbox[["lcorner"]], bbox[["ucorner"]])
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(0, -60)
shade3d(rmesh, color = "gold")
wire3d(bxmesh, color = "black")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-boundsVolume"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-boundsVolume}{}}}
\subsection{Method \code{boundsVolume()}}{
Check whether the mesh bounds a volume. The mesh must be 
  triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$boundsVolume()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh bounds a volume.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$boundsVolume() # TRUE
mesh$reverseOrientation()
mesh$boundsVolume() # TRUE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-CatmullClark"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-CatmullClark}{}}}
\subsection{Method \code{CatmullClark()}}{
Performs the Catmull-Clark subdivision and deformation. 
  The mesh must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$CatmullClark(iterations = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{iterations}}{number of iterations}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The modified reference mesh, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
hopfMesh <- HopfTorusMesh(nu = 80, nv = 40)
mesh <- cgalMesh$new(hopfMesh)
mesh$CatmullClark(iterations = 2)
mesh$computeNormals()
rmesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 800, 400))
mfrow3d(1, 2)
view3d(0, 0, zoom = 0.9)
shade3d(hopfMesh, color = "red")
wire3d(hopfMesh, color = "black")
next3d()
view3d(0, 0, zoom = 0.9)
shade3d(rmesh, color = "red")
wire3d(rmesh, color = "black")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-centroid"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-centroid}{}}}
\subsection{Method \code{centroid()}}{
Centroid of the mesh. The mesh must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$centroid()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The Cartesian coordinates of the centroid of the mesh.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(icosahedron3d())
mesh$centroid()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-clip"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-clip}{}}}
\subsection{Method \code{clip()}}{
Clip mesh to the volume bounded by another mesh. The
  mesh must be triangle. Face properties (colors and scalars) 
  are preserved. 
  \strong{WARNING}: the reference mesh is then replaced by its 
  clipped version.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$clip(clipper, clipVolume)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{clipper}}{a \code{cgalMesh} object; it must represent a closed 
triangle mesh which doesn't self-intersect}

\item{\code{clipVolume}}{Boolean, whether the clipping has to be done on the 
volume bounded by the reference mesh rather than on its surface (i.e. 
the reference mesh will be kept closed if it is closed); if 
\code{TRUE}, the mesh to be clipped must not self-intersect}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The reference mesh is always replaced by the result of the 
  clipping. If \code{clipVolume=TRUE}, this function returns two 
  \code{cgalMesh} objects: the two parts of the clipped mesh contained 
  in the reference mesh and the clipping mesh respectively. Otherwise, 
  this function returns the modified reference mesh.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# cube clipped to sphere ####
library(cgalMeshes)
library(rgl)
mesh    <- cgalMesh$new(cube3d())$triangulate()
clipper <- cgalMesh$new(sphereMesh(r= sqrt(2)))
mesh$assignFaceColors("blue")
clipper$assignFaceColors("red")
meshes <- mesh$clip(clipper, clipVolume = TRUE)
mesh1 <- meshes[[1]]
mesh2 <- meshes[[2]]
mesh2$computeNormals()
rglmesh1 <- mesh1$getMesh()
rglmesh2 <- mesh2$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(45, 45, zoom = 0.9)
shade3d(rglmesh1, meshColor = "faces")
shade3d(rglmesh2, meshColor = "faces")

# Togliatti surface clipped to a ball ####
library(rmarchingcubes)
library(rgl)
library(cgalMeshes)
# Togliatti surface equation: f(x,y,z) = 0
f <- function(x, y, z) {
  64*(x-1) *
    (x^4 - 4*x^3 - 10*x^2*y^2 - 4*x^2 + 16*x - 20*x*y^2 + 5*y^4 + 16 - 20*y^2) - 
    5*sqrt(5-sqrt(5))*(2*z - sqrt(5-sqrt(5))) * 
    (4*(x^2 + y^2 - z^2) + (1 + 3*sqrt(5)))^2
}
# grid
n <- 200L
x <- y <- seq(-5, 5, length.out = n)
z <- seq(-4, 4, length.out = n)
Grid <- expand.grid(X = x, Y = y, Z = z)
# calculate voxel
voxel <- array(with(Grid, f(X, Y, Z)), dim = c(n, n, n))
# calculate isosurface
contour_shape <- contour3d(
  griddata = voxel, level = 0, x = x, y = y, z = z
)
# make rgl mesh (plotted later)
rglMesh <- tmesh3d(
  vertices = t(contour_shape[["vertices"]]),
  indices  = t(contour_shape[["triangles"]]),
  normals  = contour_shape[["normals"]],
  homogeneous = FALSE
)
# make CGAL mesh
mesh <- cgalMesh$new(rglMesh)
# clip to sphere of radius 4.8
sphere <- sphereMesh(r = 4.8)
clipper <- cgalMesh$new(sphere)
mesh$clip(clipper, clipVolume = FALSE)
rglClippedMesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 900, 450))
mfrow3d(1L, 2L)
view3d(0, -70, zoom = 0.8)
shade3d(rglMesh, color = "firebrick")
next3d()
view3d(0, -70, zoom = 0.8)
shade3d(rglClippedMesh, color = "firebrick")
shade3d(sphere, color = "yellow", alpha = 0.15)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-clipToPlane"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-clipToPlane}{}}}
\subsection{Method \code{clipToPlane()}}{
Clip the mesh to a plane. The mesh must be triangle. Face 
  properties (colors, scalars) are preserved.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$clipToPlane(planePoint, planeNormal, clipVolume)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{planePoint}}{numeric vector of length three, a point belonging 
to the plane}

\item{\code{planeNormal}}{numeric vector of length three, a vector orthogonal
to the plane}

\item{\code{clipVolume}}{Boolean, whether to clip on the volume}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
If \code{clipVolume=FALSE}, the modified reference mesh is 
  invisibly returned. If \code{clipVolume=TRUE}, a list of two 
  \code{cgalMesh} objects is returned: the first one is the part of 
  the clipped mesh corresponding to the original mesh, the second 
  one is the part of the clipped mesh corresponding to the plane.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
rmesh <- sphereMesh()
mesh <- cgalMesh$new(rmesh)
nfaces <- nrow(mesh$getFaces())
if(require("randomcoloR")) {
  colors <- 
    randomColor(nfaces, hue = "random", luminosity = "dark")
} else {
  colors <- rainbow(nfaces)
}
mesh$assignFaceColors(colors)
meshes <- mesh$clipToPlane(
  planePoint  = c(0, 0, 0), 
  planeNormal = c(0, 0, 1), 
  clipVolume = TRUE
)
mesh1 <- meshes[[1]]
mesh2 <- meshes[[2]]
mesh1$computeNormals()
rClippedMesh1 <- mesh1$getMesh()
rClippedMesh2 <- mesh2$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(70, 0)
shade3d(rClippedMesh1, meshColor = "faces")
shade3d(rClippedMesh2, color = "orange")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-clipToIsoCuboid"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-clipToIsoCuboid}{}}}
\subsection{Method \code{clipToIsoCuboid()}}{
Clip the mesh to an iso-oriented cuboid. The mesh must be 
  triangle. Face properties (colors, scalars) are preserved.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$clipToIsoCuboid(lcorner, ucorner, clipVolume)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{lcorner, ucorner}}{two diagonally opposite vertices of the 
iso-oriented cuboid, the smallest and the largest (see 
\code{\link{isoCuboidMesh}})}

\item{\code{clipVolume}}{Boolean, whether to clip on the volume}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
If \code{clipVolume=FALSE}, the modified reference mesh is 
  invisibly returned. If \code{clipVolume=TRUE}, a list of two 
  \code{cgalMesh} objects is returned: the first one is the part of 
  the clipped mesh corresponding to the original mesh, the second 
  one is the part of the clipped mesh corresponding to the cuboid.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
rmesh <- HopfTorusMesh(nu = 200, nv = 200)
mesh <- cgalMesh$new(rmesh)
mesh$assignFaceColors("orangered")
lcorner <- c(-7, -7, -5)
ucorner <- c(7, 6, 5)
bxmesh <- isoCuboidMesh(lcorner, ucorner)
mesh$clipToIsoCuboid(
  lcorner, ucorner, clipVolume = FALSE
)
mesh$computeNormals()
rClippedMesh <- mesh$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(-40, 0)
shade3d(rClippedMesh, meshColor = "faces")
shade3d(bxmesh, color = "cyan", alpha = 0.3)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-computeNormals"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-computeNormals}{}}}
\subsection{Method \code{computeNormals()}}{
Compute per-vertex normals of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$computeNormals()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The current \code{cgalMesh} object, invisibly. 
 To get the normals, use the \code{getNormals} method.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-connectedComponents"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-connectedComponents}{}}}
\subsection{Method \code{connectedComponents()}}{
Decomposition into connected components. All face 
  properties (colors, scalars) and vertex properties 
  (colors, scalars, normals) are preserved.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$connectedComponents(triangulate = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{triangulate}}{Boolean, whether to triangulate the connected 
components.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list of \code{cgalMesh} objects, one for each connected 
  component.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rmarchingcubes)
# isosurface function (slice of a seven-dimensional toratope)
f <- function(x, y, z, a) {
  (sqrt(
    (sqrt((sqrt((x*sin(a))^2 + (z*cos(a))^2) - 5)^2 + (y*sin(a))^2) - 2.5)^2 + 
      (x*cos(a))^2) - 1.25
  )^2 + (sqrt((sqrt((z*sin(a))^2 + (y*cos(a))^2) - 2.5)^2) - 1.25)^2
}
# make grid
n <- 200L
x <- seq(-10, 10, len = n)
y <- seq(-10, 10, len = n)
z <- seq(-10, 10, len = n)
Grid <- expand.grid(X = x, Y = y, Z = z)
# compute isosurface
voxel <- array(with(Grid, f(X, Y, Z, a = pi/2)), dim = c(n, n, n))
isosurface <- contour3d(voxel, level = 0.25, x = x, y = y, z = z)
# make CGAL mesh
mesh <- cgalMesh$new(
  vertices = isosurface[["vertices"]], 
  faces = isosurface[["triangles"]],
  normals = isosurface[["normals"]]
)
# connected components
components <- mesh$connectedComponents()
ncc <- length(components)
# plot
library(rgl)
colors <- rainbow(ncc)
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(30, 50)
for(i in 1L:ncc) {
  rglMesh <- components[[i]]$getMesh()
  shade3d(rglMesh, color = colors[i])
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-convexParts"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-convexParts}{}}}
\subsection{Method \code{convexParts()}}{
Decomposition into convex parts. The mesh must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$convexParts(triangulate = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{triangulate}}{Boolean, whether to triangulate the convex parts}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A list of \code{cgalMesh} objects, one for each convex part.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(pentagrammicPrism)$triangulate()
cxparts <- mesh$convexParts()
ncxparts <- length(cxparts)
colors <- hcl.colors(ncxparts, palette = "plasma")
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(20, -20, zoom = 0.8)
for(i in 1L:ncxparts) {
  cxmesh <- cxparts[[i]]$getMesh()
  shade3d(cxmesh, color = colors[i])
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-copy"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-copy}{}}}
\subsection{Method \code{copy()}}{
Copy the mesh. This can change the order of the vertices.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$copy()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A new \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())
tmesh <- mesh$copy()$triangulate()
tmesh$isTriangle() # TRUE
mesh$isTriangle() # FALSE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-distance"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-distance}{}}}
\subsection{Method \code{distance()}}{
Distance from one or more points to the mesh. The mesh 
  must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$distance(points)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{points}}{either one point given as a numeric vector or several 
points given as a numeric matrix with three columns}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A numeric vector providing the distances between the given 
  point(s) to the mesh.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# cube example ####
library(cgalMeshes)
mesh <- cgalMesh$new(rgl::cube3d())$triangulate()
points <- rbind(
  c(0, 0, 0),
  c(1, 1, 1)
)
mesh$distance(points) # should be 1 and 0

# cyclide example ####
library(cgalMeshes)
a <- 100; c <- 30; mu <- 80
mesh <- cgalMesh$new(cyclideMesh(a, c, mu, nu = 100L, nv = 100L))
O2 <- c(c, 0, 0)
# should be a - mu = 20 (see ?cyclideMesh):
mesh$distance(O2)    
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-DooSabin"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-DooSabin}{}}}
\subsection{Method \code{DooSabin()}}{
Performs the Doo-Sabin subdivision and deformation.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$DooSabin(iterations = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{iterations}}{number of iterations}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The modified reference mesh, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
hopfMesh <- HopfTorusMesh(nu = 80, nv = 40)
mesh <- cgalMesh$new(hopfMesh)
mesh$DooSabin(iterations = 2)
mesh$triangulate()
mesh$computeNormals()
rmesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 800, 400))
mfrow3d(1, 2)
view3d(0, 0, zoom = 0.9)
shade3d(hopfMesh, color = "brown")
wire3d(hopfMesh, color = "black")
next3d()
view3d(0, 0, zoom = 0.9)
shade3d(rmesh, color = "brown")
wire3d(rmesh, color = "black")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-facesAroundVertex"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-facesAroundVertex}{}}}
\subsection{Method \code{facesAroundVertex()}}{
Faces containing a given vertex.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$facesAroundVertex(v)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{v}}{a vertex index}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An integer vector, the indices of the faces containing \code{v}.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-fair"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-fair}{}}}
\subsection{Method \code{fair()}}{
Fair a region of the mesh, i.e. make it smooth. The mesh 
  must be triangle. This modifies the reference mesh. All 
  face properties and vertex properties except the normals 
  are preserved.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$fair(indices)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{indices}}{the indices of the vertices in the region to be faired}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The modified \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
rglHopf <- HopfTorusMesh(nu = 100, nv = 100)
hopf <- cgalMesh$new(rglHopf)
# squared norms of the vertices
normsq <- apply(hopf$getVertices(), 1L, crossprod)
# fair the region where the squared norm is > 19
indices <- which(normsq > 19)
hopf$fair(indices)
rglHopf_faired <- hopf$getMesh()
# plot
library(rgl)
open3d(windowRect = 50 + c(0, 0, 900, 450))
mfrow3d(1L, 2L)
view3d(0, 0, zoom = 0.8)
shade3d(rglHopf, color = "orangered")
next3d()
view3d(0, 0, zoom = 0.8)
shade3d(rglHopf_faired, color = "orangered")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-fillBoundaryHole"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-fillBoundaryHole}{}}}
\subsection{Method \code{fillBoundaryHole()}}{
Fill a hole in the mesh. The face properties and the 
  vertex properties are preserved.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$fillBoundaryHole(border, fair = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{border}}{index of the boundary cycle forming the hole to be 
filled; the boundary cycles can be identified with 
\code{$getBorders()}}

\item{\code{fair}}{Boolean, whether to fair (i.e. smooth) the filled hole}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The filled hole as a \code{cgalMesh} object. The reference 
  mesh is updated.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# make a sphere
sphere <- sphereMesh()
mesh <- cgalMesh$new(sphere)
# make a hole in this sphere
mesh$clipToPlane(
  planePoint  = c(0.5, 0, 0),
  planeNormal = c(1, 0, 0),
  clipVolume  = FALSE
)
mesh$computeNormals()
rmesh <- mesh$getMesh()
# fill the hole
hole <- mesh$fillBoundaryHole(1, fair = TRUE)
hole$computeNormals()
rhole <- hole$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(30, 30)
shade3d(rmesh, color = "red")
shade3d(rhole, color = "blue")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-fillFaceColors"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-fillFaceColors}{}}}
\subsection{Method \code{fillFaceColors()}}{
Replace missing face colors with a color.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$fillFaceColors(color)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{color}}{the color to replace the missing face colors}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The reference mesh, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-filterMesh"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-filterMesh}{}}}
\subsection{Method \code{filterMesh()}}{
Split the mesh into two meshes according to a 
  given set of selected faces. Face properties are 
  preserved.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$filterMesh(faces)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{faces}}{a vector of face indices}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Two \code{cgalMesh} objects. The first one is the mesh consisting 
 of the faces of the reference mesh given in the \code{faces} 
 argument. The second one is the complementary mesh.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
library(cgalMeshes)
rmesh <- HopfTorusMesh(nu = 80, nv = 60)
mesh <- cgalMesh$new(rmesh)
areas <- mesh$getFacesInfo()[, "area"]
bigFaces <- which(areas > 1)
meshes <- mesh$filterMesh(bigFaces)
rmesh1 <- meshes[[1]]$getMesh()
rmesh2 <- meshes[[2]]$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(0, 0)
shade3d(rmesh1, color = "red")
shade3d(rmesh2, color = "blue")
wire3d(rmesh)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-fixManifoldness"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-fixManifoldness}{}}}
\subsection{Method \code{fixManifoldness()}}{
Duplicate non-manifold vertices.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$fixManifoldness()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The possibly modified reference mesh, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-geoDists"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-geoDists}{}}}
\subsection{Method \code{geoDists()}}{
Estimated geodesic distances between vertices. The mesh 
  must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$geoDists(index)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{index}}{index of the source vertex}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The estimated geodesic distances from the source vertex to each
  vertex.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# torus ####
library(cgalMeshes)
library(rgl)
rglmesh <- torusMesh(R = 3, r = 2, nu = 90, nv = 60)
mesh <- cgalMesh$new(rglmesh)
# estimated geodesic distances
geodists <- mesh$geoDists(1L)
# normalization to (0, 1)
geodists <- geodists / max(geodists)
# color each vertex according to its geodesic distance from the source
fcolor <- colorRamp(viridisLite::turbo(200L))
colors <- fcolor(geodists)
colors <- rgb(colors[, 1L], colors[, 2L], colors[, 3L], maxColorValue = 255)
rglmesh[["material"]] <- list("color" = colors)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(rglmesh)
wire3d(rglmesh, color = "black")
if(!rgl.useNULL()) {
  play3d(spin3d(axis = c(1, 1, 1), rpm = 5), duration = 20)  
}

# a trefoil knot (taken from `?rgl::cylinder3d`) ####
library(cgalMeshes)
library(rgl)
theta <- seq(0, 2*pi, length.out = 50L)
knot <- cylinder3d(
  center = cbind(
    sin(theta) + 2*sin(2*theta), 
    2*sin(3*theta), 
    cos(theta) - 2*cos(2*theta)),
  e1 = cbind(
    cos(theta) + 4*cos(2*theta), 
    6*cos(3*theta), 
    sin(theta) + 4*sin(2*theta)),
  radius = 0.8, 
  closed = TRUE)
knot <- subdivision3d(knot, depth = 2)
mesh <- cgalMesh$new(knot)$triangulate()
rglmesh <- mesh$getMesh()
# estimated geodesic distances
geodists <- mesh$geoDists(1L)
# normalization to (0, 1)
geodists <- geodists / max(geodists)
# color each vertex according to its geodesic distance from the source
fcolor <- colorRamp(viridisLite::inferno(200L))
colors <- fcolor(geodists)
colors <- rgb(colors[, 1L], colors[, 2L], colors[, 3L], maxColorValue = 255)
rglmesh[["material"]] <- list("color" = colors)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(rglmesh)
if(!rgl.useNULL()) {
  play3d(spin3d(axis = c(1, 1, 0), rpm = 5), duration = 20)  
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getBorders"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getBorders}{}}}
\subsection{Method \code{getBorders()}}{
Get the borders of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getBorders()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A list of matrices representing the boundary cycles. Each matrix 
 has three columns: \code{"edge"}, an edge index, and 
 \code{"v1"} and \code{"v2"}, the vertex indices of this edge.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# isosurface f=0
f <- function(x, y, z) {
  sin_x <- sin(x)
  sin_y <- sin(y)
  sin_z <- sin(z)
  cos_x <- cos(x)
  cos_y <- cos(y)
  cos_z <- cos(z)
  d <- sqrt(
    (-sin_x * sin_y + cos_x * cos_z) ** 2
    + (-sin_y * sin_z + cos_y * cos_x) ** 2
    + (-sin_z * sin_x + cos_z * cos_y) ** 2
  )
  (
    cos(
      x - (-sin_x * sin_y + cos_x * cos_z) / d
    )
    * sin(
      y - (-sin_y * sin_z + cos_y * cos_x) / d
    )
    + cos(
      y - (-sin_y * sin_z + cos_y * cos_x) / d
    )
    * sin(
      z - (-sin_z * sin_x + cos_z * cos_y)/ d
    )
    + cos(
      z - (-sin_z * sin_x + cos_z * cos_y) / d
    )
    * sin(
      x - (-sin_x * sin_y + cos_x * cos_z) / d
    )
  ) * (
    (
      cos(
        x + (-sin_x * sin_y + cos_x * cos_z) / d
      )
      * sin(
        y + (-sin_y * sin_z + cos_y * cos_x) / d
      )
      + cos(
        y + (-sin_y * sin_z + cos_y * cos_x) / d
      )
      * sin(
        z + (-sin_z * sin_x + cos_z * cos_y) / d
      )
      + cos(
        z + (-sin_z * sin_x + cos_z * cos_y) / d
      )
      * sin(
        x + (-sin_x * sin_y + cos_x * cos_z) / d
      )
    )
  )
}
# construct the isosurface f=0
ngrid <- 200L
x <- y <- z <- seq(-8.1, 8.1, length.out = ngrid)
Grid <- expand.grid(X = x, Y = y, Z = z)
voxel <- array(
  with(Grid, f(X, Y, Z)), dim = c(ngrid, ngrid, ngrid)
)
library(rmarchingcubes)
contour_shape <- contour3d(
  griddata = voxel, level = 0,
  x = x, y = y, z = z
)
# make mesh
mesh <- cgalMesh$new(
  list(
    "vertices" = contour_shape[["vertices"]],
    "faces"    = contour_shape[["triangles"]]
  )
)
# clip the mesh to the ball of radius 8
spheremesh <- cgalMesh$new(sphereMesh(r = 8))
mesh$clip(spheremesh, clipVolume = FALSE)
# compute normals
mesh$computeNormals()
# we will plot the borders
borders <- mesh$getBorders()
# plot
rmesh <- mesh$getMesh()
open3d(windowRect = c(50, 50, 562, 562), zoom = 0.7)
shade3d(rmesh, color = "darkred")
vertices <- mesh$getVertices()
for(border in borders){
  plotEdges(
    vertices, border[, c("v1", "v2")], color = "gold",
    lwd = 3, edgesAsTubes = FALSE, verticesAsSpheres = FALSE
  )
}
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getEdges"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getEdges}{}}}
\subsection{Method \code{getEdges()}}{
Get the edges of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getEdges()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A dataframe with seven columns; the first two ones give the 
  vertex indices of each edge (one edge per row), the third one gives 
  the lengths of each edge, the fourth one indicates whether the edges 
  is a border edge, the fifth one gives the dihedral angles 
  in degrees between the two faces adjacent to each edge, and the last 
  two ones gives the indices of the faces the edge belongs to (the 
  second index is \code{NA} if the edge is a border edge).
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(dodecahedron3d())
head(mesh$getEdges())
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getFaces"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getFaces}{}}}
\subsection{Method \code{getFaces()}}{
Get the faces of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getFaces()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The faces in a matrix if the mesh is triangle or quad, 
  otherwise in a list.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getFacesInfo"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getFacesInfo}{}}}
\subsection{Method \code{getFacesInfo()}}{
Get the centroids, the circumcenters, and the areas of the 
  faces, for a triangle mesh only.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getFacesInfo()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A matrix with seven columns: the first three ones provide the 
  Cartesian coordinates of the centroids, the three next ones provide 
  the Cartesian coordinates of the circumcenters, and the last one 
  provides the areas.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getFaceColors"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getFaceColors}{}}}
\subsection{Method \code{getFaceColors()}}{
Get the face colors (if there are).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getFaceColors()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The vector of colors (or any character vector) attached to 
  the faces of the mesh, or \code{NULL} if nothing is assigned to 
  the faces.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getFaceScalars"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getFaceScalars}{}}}
\subsection{Method \code{getFaceScalars()}}{
Get the face scalars (if there are).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getFaceScalars()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The vector of scalars attached to 
  the faces of the mesh, or \code{NULL} if nothing is assigned to 
  the faces.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getVertexColors"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getVertexColors}{}}}
\subsection{Method \code{getVertexColors()}}{
Get the vertex colors (if there are).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getVertexColors()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The vector of colors (or any character vector) attached to 
  the vertices of the mesh, or \code{NULL} if nothing is assigned to 
  the vertices.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getVertexScalars"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getVertexScalars}{}}}
\subsection{Method \code{getVertexScalars()}}{
Get the vertex scalars (if there are).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getVertexScalars()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The vector of scalars attached to 
  the vertices of the mesh, or \code{NULL} if nothing is assigned to 
  the vertices.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getNormals"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getNormals}{}}}
\subsection{Method \code{getNormals()}}{
Get the per-vertex normals (if there are).
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getNormals()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The matrix of per-vertex normals if they have been given or 
  computed (see \code{computeNormals}, or \code{NULL} otherwise.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getMesh"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getMesh}{}}}
\subsection{Method \code{getMesh()}}{
Get the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getMesh(rgl = TRUE, ...)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{rgl}}{Boolean, whether to return a \strong{rgl} mesh if possible, 
i.e. if the mesh only has triangular or quadrilateral faces}

\item{\code{...}}{arguments passed to \code{\link[rgl:mesh3d]{mesh3d}} (if 
a \strong{rgl} mesh is returned)}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \strong{rgl} mesh or a list with two or three fields: 
  \code{vertices}, \code{faces}, and \code{normals} if XXXXXXXXXXXXXXXXXXXXXXXXX the argument 
  \code{normals} is set to \code{TRUE}
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$getMesh()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-getVertices"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-getVertices}{}}}
\subsection{Method \code{getVertices()}}{
Get the vertices of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$getVertices()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The vertices in a matrix.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-HausdorffApproximate"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-HausdorffApproximate}{}}}
\subsection{Method \code{HausdorffApproximate()}}{
Approximate Hausdorff distance between the reference mesh
  and another mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$HausdorffApproximate(mesh2, symmetric = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}

\item{\code{symmetric}}{Boolean, whether to consider the symmetric Hausdorff 
distance.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A number. The algorithm uses some simulations and thus the 
  result can vary.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-HausdorffEstimate"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-HausdorffEstimate}{}}}
\subsection{Method \code{HausdorffEstimate()}}{
Estimate of Hausdorff distance between the reference mesh
  and another mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$HausdorffEstimate(mesh2, errorBound = 1e-04, symmetric = TRUE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}

\item{\code{errorBound}}{a positive number, a bound on the error of the 
estimate}

\item{\code{symmetric}}{Boolean, whether to consider the symmetric Hausdorff 
distance.}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A number.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-intersection"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-intersection}{}}}
\subsection{Method \code{intersection()}}{
Intersection with another mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$intersection(mesh2)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{cgalMesh} object.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# intersection
imesh <- mesh1$intersection(mesh2)
rglimesh <- imesh$getMesh()
# extract edges for plotting
extEdges <- exteriorEdges(imesh$getEdges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglimesh, color = "red")
plotEdges(imesh$getVertices(), extEdges)
shade3d(rglmesh1, color = "yellow", alpha = 0.2)
shade3d(rglmesh2, color = "cyan", alpha = 0.2)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isClosed"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isClosed}{}}}
\subsection{Method \code{isClosed()}}{
Check whether the mesh is closed.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isClosed()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is closed.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isotropicRemeshing"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isotropicRemeshing}{}}}
\subsection{Method \code{isotropicRemeshing()}}{
Isotropic remeshing.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isotropicRemeshing(targetEdgeLength, iterations = 1, relaxSteps = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{targetEdgeLength}}{positive number, the target edge length of the
remeshed mesh}

\item{\code{iterations}}{number of iterations, a positive integer}

\item{\code{relaxSteps}}{number of relaxation steps, a positive integer}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
mesh <- cgalMesh$new(HopfTorusMesh(nu = 80, nv = 50))
mesh$isotropicRemeshing(targetEdgeLength = 0.7)
# squared norms of the vertices
normsq <- apply(mesh$getVertices(), 1L, crossprod)
# fair the region where the squared norm is > 19
mesh$fair(which(normsq > 19))
# plot
mesh$computeNormals()
rmesh <- mesh$getMesh()
open3d(windowRect = 50 + c(0, 0, 512, 512))
view3d(0, 0)
shade3d(rmesh, color = "maroon")
wire3d(rmesh)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isOutwardOriented"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isOutwardOriented}{}}}
\subsection{Method \code{isOutwardOriented()}}{
Check whether the mesh is outward oriented. The mesh must 
  be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isOutwardOriented()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is outward oriented.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$isOutwardOriented() # TRUE
mesh$reverseOrientation()
mesh$isOutwardOriented() # FALSE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isTriangle"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isTriangle}{}}}
\subsection{Method \code{isTriangle()}}{
Check whether the mesh is triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isTriangle()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is triangle.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())
mesh$isTriangle()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isValid"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isValid}{}}}
\subsection{Method \code{isValid()}}{
Check whether the mesh is valid.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isValid()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is valid.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isValidFaceGraph"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isValidFaceGraph}{}}}
\subsection{Method \code{isValidFaceGraph()}}{
Check whether the mesh is valid.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isValidFaceGraph()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is valid.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isValidHalfedgeGraph"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isValidHalfedgeGraph}{}}}
\subsection{Method \code{isValidHalfedgeGraph()}}{
Check whether the mesh is valid.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isValidHalfedgeGraph()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is valid.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-isValidPolygonMesh"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-isValidPolygonMesh}{}}}
\subsection{Method \code{isValidPolygonMesh()}}{
Check whether the mesh is valid.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$isValidPolygonMesh()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh is valid.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-LoopSubdivision"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-LoopSubdivision}{}}}
\subsection{Method \code{LoopSubdivision()}}{
Performs the Loop subdivision and deformation.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$LoopSubdivision(iterations = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{iterations}}{number of iterations}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The modified reference mesh, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
hopfMesh <- HopfTorusMesh(nu = 80, nv = 40)
mesh <- cgalMesh$new(hopfMesh)
mesh$LoopSubdivision(iterations = 2)
mesh$computeNormals()
rmesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 800, 400))
mfrow3d(1, 2)
view3d(0, 0, zoom = 0.9)
shade3d(hopfMesh, color = "gold")
wire3d(hopfMesh, color = "black")
next3d()
view3d(0, 0, zoom = 0.9)
shade3d(rmesh, color = "gold")
wire3d(rmesh, color = "black")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-merge"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-merge}{}}}
\subsection{Method \code{merge()}}{
Merge the mesh and another mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$merge(mesh2)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The updated reference mesh, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
mesh1 <- cgalMesh$new(sphereMesh())
mesh1$assignFaceColors("red")
mesh2 <- cgalMesh$new(sphereMesh(x = 3))
mesh2$assignFaceColors("blue")
mesh1$merge(mesh2)
rmesh <- mesh1$getMesh()
open3d(windowRect = c(50, 50, 562, 562))
shade3d(rmesh, meshColor = "faces")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-orientToBoundVolume"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-orientToBoundVolume}{}}}
\subsection{Method \code{orientToBoundVolume()}}{
Reorient the connected components of the mesh in order that 
it bounds a volume. The mesh must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$orientToBoundVolume()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly. \strong{WARNING}: 
  even if you store the result in a new variable, the original mesh is 
  modified.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{# two disjoint tetrahedra ####
vertices <- rbind(
  c(0, 0, 0),
  c(2, 2, 0),
  c(2, 0, 2),
  c(0, 2, 2),
  c(3, 3, 3),
  c(5, 5, 3),
  c(5, 3, 5),
  c(3, 5, 5)
)
faces <- rbind(
  c(3, 2, 1),
  c(3, 4, 2),
  c(1, 2, 4),
  c(4, 3, 1),
  c(5, 6, 7),
  c(6, 8, 7),
  c(8, 6, 5),
  c(5, 7, 8)
)
mesh <- cgalMesh$new(vertices = vertices, faces = faces)
mesh$boundsVolume() # FALSE
mesh$orientToBoundVolume()
mesh$boundsVolume() # TRUE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-removeSelfIntersections"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-removeSelfIntersections}{}}}
\subsection{Method \code{removeSelfIntersections()}}{
Remove self-intersections (experimental). The mesh must 
  be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$removeSelfIntersections()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-reverseOrientation"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-reverseOrientation}{}}}
\subsection{Method \code{reverseOrientation()}}{
Reverse the orientation of the faces of the mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$reverseOrientation()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly. \strong{WARNING}: 
  even if you store the result in a new variable, the original mesh is 
  modified.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(tetrahedron3d())
mesh$isOutwardOriented() # TRUE
mesh$reverseOrientation()
mesh$isOutwardOriented() # FALSE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-sample"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-sample}{}}}
\subsection{Method \code{sample()}}{
Random sampling on the mesh. The mesh must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$sample(nsims)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{nsims}}{integer, the desired number of simulations}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{nsims x 3} matrix containing the simulations.
}
}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-selfIntersects"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-selfIntersects}{}}}
\subsection{Method \code{selfIntersects()}}{
Check whether the mesh self-intersects. The mesh must be 
  triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$selfIntersects()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A Boolean value, whether the mesh self-intersects.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(dodecahedron3d())
mesh$selfIntersects()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-sharpEdges"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-sharpEdges}{}}}
\subsection{Method \code{sharpEdges()}}{
Returns edges considered to be sharp according to the given 
  angle bound.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$sharpEdges(angleBound)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{angleBound}}{angle bound in degrees; an edge whose corresponding 
dihedral angle is smaller than this bound is considered as sharp}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An integer matrix with three columns: \code{"edge"}, an edge 
  index, and \code{"v1"} and \code{"v2"}, the vertex indices of this 
  edge.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# astroidal ellipsoid
f <- function(u, v) {
  rbind(
    cos(u)^3 * cos(v)^3,
    sin(u)^3 * cos(v)^3,
    sin(v)^3
  )
}
rmesh <- parametricMesh(
  f, urange = c(0, 2*pi), vrange = c(0, 2*pi), 
  periodic = c(TRUE, TRUE), nu = 120, nv = 110
)
mesh <- cgalMesh$new(rmesh)
sharpEdges <- mesh$sharpEdges(30)
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.8)
shade3d(addNormals(rmesh), color = "chartreuse")
plotEdges(
  mesh$getVertices(), sharpEdges[, c("v1", "v2")], 
  edgesAsTubes = FALSE, lwd = 5, verticesAsSpheres = FALSE
)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-Sqrt3Subdivision"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-Sqrt3Subdivision}{}}}
\subsection{Method \code{Sqrt3Subdivision()}}{
Performs the 'Sqrt3' subdivision and deformation. The mesh 
  must be triangle.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$Sqrt3Subdivision(iterations = 1)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{iterations}}{number of iterations}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
The modified reference mesh, invisibly.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
hopfMesh <- HopfTorusMesh(nu = 80, nv = 40)
mesh <- cgalMesh$new(hopfMesh)
mesh$Sqrt3Subdivision(iterations = 2)
mesh$computeNormals()
rmesh <- mesh$getMesh()
# plot
open3d(windowRect = 50 + c(0, 0, 800, 400))
mfrow3d(1, 2)
view3d(0, 0, zoom = 0.9)
shade3d(hopfMesh, color = "cyan")
wire3d(hopfMesh, color = "black")
next3d()
view3d(0, 0, zoom = 0.9)
shade3d(rmesh, color = "cyan")
wire3d(rmesh, color = "black")
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-subtract"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-subtract}{}}}
\subsection{Method \code{subtract()}}{
Subtract a mesh. Both meshes must be triangle. Face 
  properties of the two meshes are copied to the new mesh. 
 \strong{WARNING}: this modifies the reference mesh and \code{mesh2}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$subtract(mesh2)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{cgalMesh} object, the difference between the reference 
  mesh and \code{mesh2}. Both the reference mesh and \code{mesh2} are 
  modified: they are corefined.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# assign colors
mesh1$assignFaceColors("red")
mesh2$assignFaceColors("navy")
# difference
mesh <- mesh1$subtract(mesh2)
rglmesh <- mesh$getMesh()
# extract edges for plotting
extEdges <- exteriorEdges(mesh$getEdges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglmesh, meshColor = "faces")
plotEdges(mesh$getVertices(), extEdges)
shade3d(rglmesh2, color = "cyan", alpha = 0.2)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-triangulate"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-triangulate}{}}}
\subsection{Method \code{triangulate()}}{
Triangulate mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$triangulate()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
The modified \code{cgalMesh} object, invisibly. \strong{WARNING}: 
  even if you store the result in a new variable, the original mesh is 
  modified (see the example). You may want to triangulate a copy of the 
  mesh; see the \code{copy} method.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())
mesh$isTriangle() # FALSE
# warning: triangulating the mesh modifies it
mesh$triangulate()
mesh$isTriangle() # TRUE
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-union"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-union}{}}}
\subsection{Method \code{union()}}{
Union with another mesh. Both meshes must be triangle. Face 
  properties of the two united meshes are copied to the union mesh. 
 \strong{WARNING}: this modifies the reference mesh and \code{mesh2}.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$union(mesh2)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{mesh2}}{a \code{cgalMesh} object}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
A \code{cgalMesh} object, the union of the reference mesh with 
  \code{mesh2}. Both the reference mesh and \code{mesh2} are modified: 
  they are corefined.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
library(rgl)
# take two cubes
rglmesh1 <- cube3d()
rglmesh2 <- translate3d(cube3d(), 1, 1, 1)
mesh1 <- cgalMesh$new(rglmesh1)
mesh2 <- cgalMesh$new(rglmesh2)
# the two meshes must be triangle
mesh1$triangulate()
mesh2$triangulate()
# assign a color to the faces; they will be retrieved in the union
mesh1$assignFaceColors("yellow")
mesh2$assignFaceColors("navy")
# union
umesh <- mesh1$union(mesh2)
rglumesh <- umesh$getMesh()
# extract edges for plotting
extEdges <- exteriorEdges(umesh$getEdges())
# plot
open3d(windowRect = 50 + c(0, 0, 512, 512), zoom = 0.9)
shade3d(rglumesh, meshColor = "faces")
plotEdges(umesh$getVertices(), extEdges)
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-volume"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-volume}{}}}
\subsection{Method \code{volume()}}{
Compute the volume of the mesh. The mesh must be closed,
  triangle, and must not self-intersect.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$volume()}\if{html}{\out{</div>}}
}

\subsection{Returns}{
A number, the mesh volume.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(rgl)
mesh <- cgalMesh$new(cube3d())$triangulate()
mesh$volume()
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-whereIs"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-whereIs}{}}}
\subsection{Method \code{whereIs()}}{
Locate points with respect to a closed triangle mesh.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$whereIs(points)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{points}}{a numeric matrix with three columns}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
An integer vector taking values \code{-1} for outside, \code{1} 
  for inside, and \code{0} if the point is on the boundary.
}
\subsection{Examples}{
\if{html}{\out{<div class="r example copy">}}
\preformatted{library(cgalMeshes)
mesh <- cgalMesh$new(sphereMesh())
pt1 <- c(0, 0, 0) # inside
pt2 <- c(2, 0, 0) # outside
mesh$whereIs(rbind(pt1, pt2))
}
\if{html}{\out{</div>}}

}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-cgalMesh-writeMeshFile"></a>}}
\if{latex}{\out{\hypertarget{method-cgalMesh-writeMeshFile}{}}}
\subsection{Method \code{writeMeshFile()}}{
Write mesh to a file.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{cgalMesh$writeMeshFile(filename, precision = 17, comments = "", binary = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{filename}}{path to the file to be written, with extension 
\code{off} or \code{ply}; if you use the \code{ply} format, the mesh 
properties (vertex colors, face colors, normals) are written in the 
file}

\item{\code{precision}}{a positive integer, the desired number of decimal 
places}

\item{\code{comments}}{for \code{ply} extension only, a string to be included 
in the header of the PLY file}

\item{\code{binary}}{Boolean, for \code{ply} extension only, whether to write 
a binary \code{ply} file}
}
\if{html}{\out{</div>}}
}
\subsection{Returns}{
Nothing, just writes a file.
}
}
}
