# last revision: 12 Jan 2012

#' @include ChemosensorsClassMethods.R
NULL

#----------------------------
# Support Class defintion
#----------------------------


#----------------------------
# Class defintion
#----------------------------

validConcNoiseModel <- function(object)
{
  if(sum(csd(object) < 0)) return("'csd' is negative")    
  
  return(TRUE)
}

#' Class ConcNoiseModel.
#'
#' Class \code{ConcNoiseModel} emulates perturbations in the analyte delivery system.
#' The noise term is a normally distributed noise with zero-mean 
#' and diagonal covariance matrix. The diagonal structure of the covariance matrix underlines 
#' that the concentration noises of analytes do not affect each other.
#'
#' Slots of the class:
#' \tabular{rl}{
#'   \code{gases} \tab Gas indices. \cr
#'   \code{ngases} \tab The number of gases. \cr
#'   \code{gnames} \tab Names of gases. \cr
#'   \code{concUnits} \tab Concentration units external to the model, values given in an input concentration matrix. \cr
#'   \code{concUnitsInt} \tab Concentration units internal for the model, values used to induce the noise. \cr
#'   \code{csd} \tab Parameter of standard deviation used to generate the noise. The deault value is 0.1. \cr
#'   \code{cntype} \tab Noise type (\code{logconc}). \cr 
#'   \code{cnlogf} \tab Scaling factor for for log-terms if the noise type is \code{logconc}. The default value is \code{c(1, 1, 2)}. \cr
#' }
#'
#' The logarithm term expresses an additional scaling of the amplitude applied to simulate more noise on high levels of concentration.
#'
#' Methods of the class:
#' \tabular{rl}{
#'   \code{predict} \tab Generates noise into the columns of an input concentration matrix. \cr
#'   \code{csd} \tab Gets the noise level. \cr
#'   \code{csd<-} \tab Sets the noise level. \cr
#' }
#' 
#' The \code{plot} method has one type (parameter \code{y}):
#' \tabular{rl}{
#'   \code{noise} \tab Depicts the noise vectors generated by the model. \cr
#' }
#'
#' @name ConcNoiseModel-class
#' @rdname ConcNoiseModel-class
#' @example inst/examples/ConcNoiseModel-class.R
#' @exportClass ConcNoiseModel
setClass(Class="ConcNoiseModel", 
  representation=representation(
    # class-specific slots
    csd = "numeric",
    # class-common slots
    gases = "numeric", gind = "numeric", ngases = "numeric", gnames="character", 
    concUnits="character", concUnitsInt="character",
    cntype = "character", cnlogf = "numeric"),  
  validity=validConcNoiseModel
)


#----------------------------
# Print/Show Methods
#----------------------------

#' @rdname class-methods
#' @aliases print,ConcNoiseModel-method
setMethod ("print","ConcNoiseModel", function(x, ...)
{
  cat(" Concentration Noise Model\n")
  cat(" -", ngases(x), "gases", paste(gnames(x), collapse=", "), "\n")
  cat(" - csd:", paste(csd(x), collapse=", "), "\n")
  cat(" - noise type:", type(x), "\n")
  if(type(x) == "logconc") 
    cat(" - log-factor:", paste(logf(x), collapse=", "), "\n")    
})

setMethod ("show","ConcNoiseModel", function(object)
{
  cat(" Concentration Noise Model (csd ", paste(csd(object), collapse=", "), "), ", "noise type '", type(object), "'", "\n", sep='')
})


#----------------------------
# Plot Methods
#----------------------------

#----------------------------
# Get/Set Methods
#----------------------------

#' @rdname noise-methods
#' @aliases csd,ConcNoiseModel-method
setMethod("csd", "ConcNoiseModel", function(x) x@csd)

#' @rdname get-methods
#' @aliases type,ConcNoiseModel-method
setMethod("type", "ConcNoiseModel", function(x) x@cntype)

#' @rdname ConcNoiseModel-class
#' @aliases logf,ConcNoiseModel-method
setMethod("logf", "ConcNoiseModel", function(x) x@cnlogf)

#' @name csd<-
#' @aliases csd<-,ConcNoiseModel-method
#' @rdname noise-methods
setReplaceMethod("csd", "ConcNoiseModel", function(object, value) 
{
  object@csd <- value
  validObject(object)
  return (object)
})


#----------------------------
# Model Methods
#----------------------------

#----------------------------
# Predict methods
#----------------------------

#----------------------------
# Noise Methods
#----------------------------

#' @name nsd<-
#' @aliases nsd<-,ConcNoiseModel-method
#' @rdname noise-methods
setReplaceMethod("nsd", "ConcNoiseModel", function(object, value) 
{
  csd(object) <- value
  validObject(object)
  return (object)
})

