% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/interpolate_gaps_hourly.R
\name{interpolate_gaps_hourly}
\alias{interpolate_gaps_hourly}
\title{Interpolate gaps in hourly temperature records}
\usage{
interpolate_gaps_hourly(hourtemps, latitude = 50, daily_temps = NULL,
  interpolate_remaining = TRUE, return_extremes = FALSE,
  minimum_values_for_solving = 5, runn_mean_test_length = 5,
  runn_mean_test_diff = 5, daily_patch_max_mean_bias = NA,
  daily_patch_max_stdev_bias = NA)
}
\arguments{
\item{hourtemps}{data.frame containing hourly temperatures. This has to
contain columns c("Year","Month","Day","Hour","Temp").}

\item{latitude}{the geographic latitude (in decimal degrees) of the location
of interest}

\item{daily_temps}{list of (chillR compliant) daily temperature data sets
for patching gaps in the record.}

\item{interpolate_remaining}{boolean parameter indicating whether gaps remaining
after the daily record has been patched (or after solving temperature equations,
if (daily_temps==NULL)) should be linearly interpolated.}

\item{return_extremes}{boolean parameters indicating whether daily minimum
and maximum temperatures used for the interpolation should be part of the
output table. Defaults to FALSE.}

\item{minimum_values_for_solving}{integer specifying the minimum number of hourly
temperature values that must be available for the solving function to be
applied. Must be greater than 1 (otherwise you get an error). Since according to
the idealized temperature curves used here, a given daily extreme temperature
is related to hourly temperatures of about a 12-hour period, values above 12
are not useful. Note that relatively large numbers for this parameter raise the
reliability of the interpolated values, but they restrict the number of missing
values in a day, for which the procedure produces results.}

\item{runn_mean_test_length}{integer specifying the length of the period, for
which a running mean test for is applied to daily records after the solving
procedure. This aims to remove spurious values that can sometimes arise during
solving. This test checks for all daily minimum and maximum temperature values,
if they differ from the mean of the surrounding values by more than
runn_mean_test_diff. If this is the case, they are set to NA, and have to be
filled by other means (from proxy data or by interpolation). Defaults to 5,
which means each value is compared to the mean of the 2 previous and 2
following days.}

\item{runn_mean_test_diff}{integer specifying the maximum tolerable difference
between solved daily extreme temperature values and the mean for the
surrounding days. See description of runn_mean_test_length for more details.
Defaults to 5.}

\item{daily_patch_max_mean_bias}{maximum acceptable mean difference between
the daily extreme temperatures of daily temperature records used as proxy and
daily extreme temperatures in the dataset that is to be interpolated. If the
bias between stations is greater than this, the station is not considered
a useful proxy and not used for filling gaps.}

\item{daily_patch_max_stdev_bias}{maximum acceptable standard deviation of
the difference between the daily extreme temperatures of daily temperature
records used as proxy and daily extreme temperatures in the dataset that is
to be interpolated. If the bias between stations is greater than this,
the station is not considered a useful proxy and not used for filling gaps.}
}
\value{
data frame containing interpolated temperatures for all hours within
the interval defined by the first and last day of the hourtemps input.
}
\description{
Using idealized temperature curves for guidance, this function interpolated hourly
temperature data.
}
\details{
Many agroclimatic metrics are calculated from hourly temperature data. chillR
provides functions for generating hourly data from daily records, which are
often available. Small gaps in such daily records can easily be closed
through linear interpolation, with relatively small errors, so that complete
hourly records can be generated. However, many sites have recorded actual
hourly temperatures, which allow much more accurate site-specific assessments.
Such records quite often have gaps, which need to be closed before
calculating most agroclimatic metrics (such as Chill Portions). Linear
interpolation is not a good option for this, because daily temperature curves
are not linear. Moreover, when gaps exceed a certain number of hours,
important featured would be missed (e.g. interpolating between temperatures
at 8 pm and 8 am may miss all the cool hours of the day, which would greatly
distort chill estimates).

This function solves this problem by using an idealized daily temperature
curve as guide to the interpolation of hourly temperature data.

These are the steps:
1) produce an idealized temperature curve for the site (which requires
site latitude as an input), assuming minimum and maximum temperatures of
0 and 1 degrees C, respectively. The calculations are based on equations published
by Spencer (1971), Almorox et al. (2005) and Linvill (1990, though I
modified these slightly to produce a smooth curve). This curve describes
the expected relationship of the temperature for the respective hour with
minimum and maximum temperatures of the same, previous or next day
(depending on the time of day), according to idealized temperature curve.
At this point, however, these daily minimum or maximum temperatures aren't
known yet.

2) determine minimum and maximum temperatures for each day. For each minimum
and maximum daily temperature, the expected relationships between hourly
temperatures and daily extremes determined in step 1, combined with the
hourly temperatures that were observed can be interpreted as an
overdetermined set of equations that define these temperatures. Since few
days will follow the ideal curve precisely, and there are usually more than
two equations that define the same daily temperature extreme value, these
equations can only be solved numerically. This is implemented with the
qr.solve function, which can provide estimates of the minimum and maximum
temperatures for all days from the available hourly records.

3) interpolate gaps in the record of estimated daily temperature extremes.
There can be days, when the number of recorded hourly temperatures isn't
sufficient for inferring daily minimum or maximum temperatures. The resulting
gaps are closed by linear interpolation (this may produce poor results if
gaps are really large, but this isn't currently addressed).

4) compute an idealized daily temperature
curve for all days, based on estimated daily temperature extremes (using
the make_hourly_temperatures function).

5) calculate deviation of recorded temperatures from idealized curve.

6) linearly interpolate deviation values using the interpolate_gaps
function.

7) add interpolated deviation values to idealized temperature curve.
}
\examples{


Winters_gaps<-make_JDay(Winters_hours_gaps[1:2000,])
colnames(Winters_gaps)[5:6]<-c("Temp","original_Temp")
interp<-interpolate_gaps_hourly(hourtemps=Winters_gaps,latitude=38.5)

#plot results: interpolated temperatures are shown in red, measured temperatures in black.
plot(interp$weather$Temp[1:120]~c(interp$weather$JDay[1:120]+
   interp$weather$Hour[1:120]/24),type="l",
   col="RED",lwd=2,xlab="JDay",ylab="Temperature")
lines(interp$weather$Temp_measured[1:120]~c(interp$weather$JDay[1:120]+
   interp$weather$Hour[1:120]/24),lwd=2)

}
\references{
Linvill DE, 1990. Calculating chilling hours and chill units from daily
maximum and minimum temperature observations. HortScience 25(1), 14-16.

Spencer JW, 1971. Fourier series representation of the position of the Sun.
Search 2(5), 172.

Almorox J, Hontoria C and Benito M, 2005. Statistical validation of
daylength definitions for estimation of global solar radiation in Toledo,
Spain. Energy Conversion and Management 46(9-10), 1465-1471)
}
\author{
Eike Luedeling
}
\keyword{utility}
