% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/chisquare.r
\name{chisquare}
\alias{chisquare}
\title{R function for chi-square and G-square test of independence, measures of association, and standardized/moment-corrected
standardized/adjusted standardized residuals, visualisation of odds ratio in 2xk tables (where k >= 2)}
\usage{
chisquare(
  data,
  B = 999,
  plot.or = FALSE,
  reference.level = 1,
  row.level = 1,
  or.alpha = 0.05,
  adj.alpha = FALSE,
  format = "short",
  graph = FALSE,
  tfs = 14
)
}
\arguments{
\item{data}{Dataframe containing the input contingency table.}

\item{B}{Number of simulated tables to be used to calculate the Monte Carlo-based p value (999 by default).}

\item{plot.or}{Takes TRUE or FALSE (default) if the user wants a plot of the odds ratios to be rendered (only for 2xk tables, where k >= 2).}

\item{reference.level}{The index of the column reference level for odds ratio calculations (default: 1).
The user must select the column level to serve as the reference level (only for 2xk tables, where k >= 2).}

\item{row.level}{The index of the row category to be used in odds ratio calculations (1 or 2; default: 1).
The user must select the row level to which the calculation of the odds ratios make reference (only for 2xk tables, where k >= 2).}

\item{or.alpha}{The significance level used for the odds ratios' confidence intervals (default: 0.05).}

\item{adj.alpha}{Takes TRUE or FALSE (default) if the user wants or does not want the significance level of the
residuals (both standarized and adjusted standardized) to be corrected using the Sidak's adjustment method (see Details).}

\item{format}{Takes \emph{short} (default) if the dataset is a dataframe storing a contingency table; if the
input dataset is a dataframe storing two columns that list the levels of the two categorical variables,
\emph{long} will preliminarily cross-tabulate the levels of the categorical variable in the 1st column against
the levels of the variable stored in the 2nd column.}

\item{graph}{Takes TRUE or FALSE (default) if the user wants or does not want to chart the distribution of the permuted
chi-square statistic accross the number of randomized tables set by the B parameter.}

\item{tfs}{Numerical value to set the size of the font used in the main body of the various output tables (14 by default).}
}
\value{
The function produces \strong{optional charts} (distribution of the permuted chi-square statistic
and a plot of the odds ratios between a reference column level and the other ones, the latter only for 2xk tables where k >= 2), and
a number of \strong{output tables} that are nicely formatted with the help of the \emph{gt} package.
The output tables are listed below:

 \itemize{
  \item Input contingency table (with some essential analytical results annotated at the bottom)
  \item Expected frequencies
  \item Cells' chi-square value
  \item Cells' relative contribution (in percent) to the chi-square statistic (cells in RED feature a larger-than-average
  contribution)
  \item Cells' absolute contribution (in percent) to the chi-square statistic (colour same as above)
  \item Standardized residuals (RED for large significant residuals, BLUE for small significant residuals)
  \item Moment-corrected standardized residuals (colour same as above)
  \item Adjusted standardized residuals (colour same as above)
  \item Table of output statistics, p values, and association measures
}

Also, the function returns a \strong{list containing the following elements}:
 \itemize{
  \item \emph{crosstab}: input contingency table
  \item \emph{exp.freq.}: table of expected frequencies
  \item \emph{chisq.values}: cells' chi-square value
  \item \emph{chisq.relat.contrib.}: cells' relative contribution (in percent) to the chi-square statistic
  \item \emph{chisq.abs.contrib.}: cells' absolute contribution (in percent) to the chi-square statistic
  \item \emph{chisq.statistic}: observed chi-square value
  \item \emph{chisq.p.value}: p value of the chi-square statistic
  \item \emph{chisq.p.value.perm.}: p value based on B permuted tables
  \item \emph{Gsq.statistic}: observed G-square value
  \item \emph{Gsq.p.value}: p value of the G-square statistic
  \item \emph{stand.resid.}: table of standardized residuals
  \item \emph{mom.corr.stand.resid.}: table of moment-corrected standardized residuals
  \item \emph{adj.stand.resid.}: table of adjusted standardized residuals
  \item \emph{Phi}: Phi coefficient (only for 2x2 tables)
  \item \emph{Phi signed}: signed Phi coefficient (only for 2x2 tables)
  \item \emph{Yule's Q}: Q coefficient (only for 2x2 tables)
  \item \emph{Yule's Q p.value}: 2-tailed p value of Yule's Q
  \item \emph{Odds ratio}: odds ratio (only for 2x2 tables)
  \item \emph{Odds ratio CI lower boundary}: lower boundary of the 95perc CI
  \item \emph{Odds ratio CI upper boundary}: upper boundary of the 95perc CI
  \item \emph{Odds ratio p.value}: p value of the odds ratio
  \item \emph{Cadj}: adjusted contigency coefficient C
  \item \emph{Cramer's V}: Cramer's V coefficient
  \item \emph{Cramer's V CI lower boundary}: lower boundary of the 95perc CI
  \item \emph{Cramer's V CI upper boundary}: upper boundary of the 95perc CI
  \item \emph{Cramer's Vbc}: bias-corrected Cramer's V coefficient
  \item \emph{w}: Cohen's w
  \item \emph{lambda (rows dep.)}: Goodman-Kruskal's lambda coefficient (considering the rows being the dependent variable)
  \item \emph{lambda (cols dep.)}: Goodman-Kruskal's lambda coefficient (considering the columns being the dependent variable)
  \item \emph{lambda.symmetric}: Goodman-Kruskal's symmetric lambda coefficient
  \item \emph{tau (rows dep.)}: Goodman-Kruskal's tau coefficient (considering the rows being the dependent variable)
  \item \emph{tau (cols dep.)}: Goodman-Kruskal's tau coefficient (considering the columns being the dependent variable)
  \item \emph{gamma}: Goodman-Kruskal's gamma coefficient
  \item \emph{gamma.p.value}: 2-sided p value for the Goodman-Kruskal's gamma coefficient
  \item \emph{k}: Cohen'k
  \item \emph{k CI lower boundary}: lower boundary of the 95perc CI
  \item \emph{k CI upper boundary}: upper boundary of the 95perc CI
}

\strong{Note} that the \emph{p values} returned in the above list are expressed in scientific notation, whereas the ones reported in the
output table featuring the tests' result and measures of association are reported as broken down into classes (e.g., <0.05, or <0.01, etc).\cr

The \strong{following examples}, which use in-built datasets, can be run to fimiliarise with the function:\cr

-perform the test on the in-built 'social_class' dataset\cr
result <- chisquare(social_class)\cr

-perform the test on a 2x2 subset of the 'diseases' dataset\cr
mytable <- diseases[3:4,1:2]\cr
result <- chisquare(mytable)\cr

-perform the test on a 2x2 subset of the 'safety' dataset\cr
mytable <- safety[c(4,1),c(1,6)]\cr
result <- chisquare(mytable)\cr

-build a toy dataset in 'long' format (gender vs. opinion about death sentence)\cr
mytable <- data.frame(GENDER=c(rep("F", 360), rep("M", 340)),\cr
OPINION=c(rep("oppose", 235),\cr
         rep("favour", 125),\cr
         rep("oppose", 160),\cr
         rep("favour", 180)))\cr

-perform the test specifying that the input table is in 'long' format\cr
result <- chisquare(mytable, format="long")\cr
}
\description{
The function performs the chi-square (and the G-square) test of independence on the input contingency table,
calculates various measures of categorical association, returns standardized, moment-corrected standardized, and adjusted
standardized residuals (with indication of their significance), and calculates relative and absolute contributions to the chi-square.
The p value associated to the chi-square statistic is also calculated on the basis of a permutation-based procedure.
Nicely-formatted output tables are rendered. Optionally, in 2xk tables (where k >= 2), a plot of the odds ratios can be rendered.
}
\details{
The following \strong{measures of categorical associations} are produced by the function:
 \itemize{
  \item Phi (only for 2x2 tables)
  \item Phi signed (only for 2x2 tables)
  \item Yule's Q (and p value; only for 2x2 tables)
  \item Odds ratio (with 95perc confidence interval and p value; only for 2x2 tables)
  \item Adjusted contingency coefficient C
  \item Cramer's V (with 95perc confidence interval; indication of the magnitude of the effect size according to Cohen is provided for tables with up to 5 degrees of freedom)
  \item Bias-corrected Cramer's V (indication of the magnitude of the effect size according to Cohen is provided for tables with up to 5 degrees of freedom)
  \item Cohen's w (with indication of the magnitude of the effect size)
  \item Goodman-Kruskal's lambda (asymmetric)
  \item Goodman-Kruskal's lambda (symmetric)
  \item Goodman-Kruskal's tau (asymmetric)
  \item Goodman-Kruskal's gamma (and p value)
  \item Cohen's k (and 95perc confidence interval)
}

\strong{P value of the chi-square statistic}\cr
The p value of the observed chi-square statistic is also calculated on the basis of a permutation-based approach,
using \emph{B} random tables created under the Null Hypothesis of independence. For the rationale of this approach,
see for instance the description in Beh-Lombardo 2014: 62-64. The permutation-based p value is calculated as follows: \cr

\eqn{(1 + sum (chistat.perm > chisq.stat)) / (1 + B)}, where\cr

\emph{chistat.perm} is a vector storing the B chi-square statistics generated under the Null Hypothesis, and
\emph{chisq.stat} is the observed chi-square statistic. For the logic of the calculation, see for example
Baddeley et al. 2016: 387.\cr


\strong{Moment-corrected standardized residuals}\cr
The moment-corrected standardized residuals are calculated as follows: \cr

\eqn{stand.res / (sqrt((nr-1)*(nc-1)/(nr*nc)))}, where\cr

\emph{stand.res} is each cell's standardized residual, \emph{nr} and
\emph{nc} are the number of rows and columns respectively; see
Garcia-Perez-Nunez-Anton 2003: 827.\cr


\strong{Adjusted standardized residuals}\cr
The adjusted standardized residuals are calculated as follows: \cr

\eqn{stand.res[i,j] / sqrt((1-sr[i]/n)*(1-sc[j]/n))}, where\cr

\emph{stand.res} is the standardized residual for cell \emph{ij},
\emph{sr} is the row sum for row \emph{i}, \emph{sc} is the column sum for column \emph{j}, and
\emph{n} is the table grand total. The \emph{adjusted standardized residuals} may prove useful since it has
been demonstrated that the standardized residuals tend to underestimate the significance of differences in small samples.
The adjusted standardized residuals correct that deficiency.\cr

\strong{Significance of the residuals}\cr
The significance of the residuals (standardized, moment-corrected standardized, and adjusted standardized) is assessed using alpha 0.05 or, optionally
(by setting the parameter 'adj.alpha' to TRUE),
using an adjusted alpha calculated using the Sidak's method:\cr

\eqn{alpha.adj = 1-(1 - 0.05)^(1/(nr*nc))}, where\cr

\emph{nr} and \emph{nc} are the number of rows and columns in the table respectively. The adjusted
alpha is then converted into a critical two-tailed z value. See: Beasley-Schumacker 1995: 86, 89.\cr


\strong{Cells' relative contribution (in percent) to the chi-square statistic}\cr
The cells' relative contribution (in percent) to the chi-square statistic is calculated as:\cr

\eqn{chisq.values / chisq.stat * 100}, where\cr

\emph{chisq.values} and \emph{chisq.stat} are the chi-square
value in each individual cell of the table and the value of the chi-square statistic, respectively. The
\emph{average contribution} is calculated as \eqn{100 / (nr*nc)}, where \emph{nr} and \emph{nc} are the
number of rows and columns in the table respectively.\cr

\strong{Cells' absolute contribution (in percent) to the chi-square statistic}\cr
The cells' absolute contribution (in percent) to the chi-square statistic is calculated as:\cr

\eqn{chisq.values / n * 100}, where\cr

\emph{chisq.values} and \emph{n} are the chi-square
value in each individual cell of the table and the table's grant total, respectively. The
\emph{average contribution} is calculated as sum of all the absolute contributions divided by the number of cells in
the table. For both the relative and absolute contributions to the chi-square, see:
Beasley-Schumacker 1995: 90.\cr


\strong{95perc confidence interval around Cramer's V}\cr
The calculation of the 95perc confidence interval around Cramer's V is based on Smithson 2003: 39-41, and builds on the R code made
available by the author on the web (http://www.michaelsmithson.online/stats/CIstuff/CI.html).\cr


\strong{Bias-corrected Cramer's V}\cr
The bias-corrected Cramer's V is based on Bergsma 2013: 323–328.\cr


\strong{Other measures of categorical association}\cr
For the other measures of categorical association provided by the function, see for example Sheskin 2011: 1415-1427.\cr


\strong{Odds Ratio}\cr
The odds ratio is calculated for 2x2 tables. In case of zeros along any of the table's diagonal,
the \emph{Haldane-Anscombe} correction is applied. It consists in adding 0.5 to every cell of the table before calculating the odds ratio.
For tables of size 2xk (where k >= 2), pairwise odds ratios can be plotted (along with their confidence interval) by
setting the \code{or.alpha} parameter to \code{TRUE}. The mentioned correction
is also applied to the calculation of those pairwise odds ratios (for more information on the plot, see further below).
For the Haldane-Anscombe correction see, for instance, Fleiss-Levin-Paik 2003: 102-103.\cr


\strong{Odd Ratios plot}\cr
For 2xk table, where k >= 2:\cr
by setting the \code{plor.or} parameter to \code{TRUE}, a plot showing the odds ratios and their 95percent confidence interval will be rendered.
The confidence level can be modified via the \code{or.alpha} parameter. The odds ratios are calculated for the column levels, and one of them
is to be selected by the user as a reference for comparison via the \code{reference.level} parameter (set to 1 by default).
Also, the user may want to select the row category to which the calculation of the odds ratios makes reference (using the \code{row.level} parameter,
which is set to 1 by default). If any of the pairwisely-generated 2x2 tables on which the odds ratio is calculated
features zeros along any of the diagonal, the \emph{Haldane-Anscombe} correction is applied (see above). \cr

To better understand the rationale of plotting the odds ratios, consider the following example, which uses on the famous Titanic data:\cr

Create a 2x3 contingency table:\cr
mytable <- matrix(c(123, 158, 528, 200, 119, 181), nrow = 2, byrow = TRUE)\cr
colnames(mytable) <- c("1st", "2nd", "3rd")\cr
rownames(mytable) <- c("Died", "Survived")\cr

Now, we perform the test and visualise the odds ratios:\cr
chisquare(mytable, plot.or=TRUE, reference.level=1, row.level=1)\cr

In the rendered plot, we can see the odds ratios and confidence intervals for the second and third column level
(i.e., 2nd class and 3rd class) because the first column level has been selected as reference level. The odds ratios are calculated
making reference to the first row category (i.e., \emph{Died}). From the plot, we can see that, compared to the 1st class,
passengers on the 2nd class have 2.16 times larger odds of dying; passengers on the 3rd class have 4.74 times larger odds of dying
compared to the 1st class.\cr

Note that if we set the \code{row.level} parameter to \code{2}, we make reference to the second row category, i.e. \emph{Survived}:\cr
chisquare(mytable, plot.or=TRUE, reference.level=1, row.level=2)\cr

In the plot, we can see that passengers in the 2nd class have 0.46 times the odds of surviving of passengers in the 1st class, while
passengers from the 3rd class have 0.21 times the odds of surviving of those travelling in the 1st class.\cr



\strong{Note}\cr
-the \strong{Phi} coefficient is based on the chi-square statistic as per Sheskin 2011's equation 16.21, whereas the
\strong{Phi signed} is after Sheskin's equation 16.20;\cr

-the \strong{2-sided p value of Yule's Q} is calculated following Sheskin 2011's equation 16.24;\cr

-\strong{Cohen's w} is calculate as \eqn{V * sqrt(min(nr, nc)-1)}, where \emph{V} is Cramer's V, and  \emph{nr} and  \emph{nc}
are the number of rows and columns respectively; see Sheskin 2011: 679;\cr

-in the output table reporting the result of the chi-square test and the various measures of association, the
magnitude of the \strong{effect size} according to \emph{Cohen}'s guidelines is reported for Cramer's V, Cramer's V biase-corrected, and for Cohen's w;
the effect size for the former two coefficients is only reported for tables featuring up to 5 degrees of freedom
(see Cohen 1988);\cr

-the \strong{2-tailed p value} of \strong{Goodman-Kruskal's gamma} is based on the
associated z-score calculated as per Sheskin 2011's equation 32.2;\cr

-the \strong{symmetric} version of \strong{Goodman-Kruskal's lambda} is calculated
as per Reynolds 1984: 55-57;\cr

-\strong{Goodman-Kruskal's tau} is calculated as per Reynolds 1984: 57-60;\cr

-\strong{Cohen's k} is calculated as per Sheskin 2011: 688-689 (equation 16.30).\cr
}
\examples{
# Perform the test on the in-built 'social_class' dataset
result <- chisquare(social_class, B=99)



}
\references{
Baddeley et al. 2016. Spatial Point Patterns. Methodology and Applications with R, CRC Press.

Beh E.J., Lombardo R. 2014. Correspondence Analysis: Theory, Practice and New Strategies, Chichester, Wiley.

Beasley TM and Schumacker RE. 1995. Multiple Regression Approach to Analyzing Contingency Tables: Post Hoc and Planned Comparison Procedures. The Journal of Experimental Education, 64(1).

Bergsma, W. 2013. A bias correction for Cramér's V and Tschuprow's T. Journal of the Korean Statistical Society. 42 (3).

Cohen, J. 1988. Statistical power analysis for the behavioral sciences (2nd ed). Hillsdale, N.J: L. Erlbaum Associates.

Fleiss, J. L., Levin, B., & Paik, M. C. 2003. Statistical Methods for Rates and Proportions (3rd ed.). Wiley.

Garcia-Perez, MA, and Nunez-Anton, V. 2003. Cellwise Residual Analysis in Two-Way Contingency Tables. Educational and Psychological Measurement, 63(5).

Reynolds, H. T. 1984. Analysis of Nominal Data (Quantitative Applications in the Social Sciences) (1st ed.). SAGE Publications.

Sheskin, D. J. 2011. Handbook of Parametric and Nonparametric Statistical Procedures, Fifth Edition (5th ed.). Chapman and Hall/CRC.

Smithson M.J. 2003. Confidence Intervals, Quantitative Applications in the Social Sciences Series, No. 140. Thousand Oaks, CA: Sage.
}
\keyword{chiperm}
