% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/repositionReferencePoint.R
\name{repositionReferencePoint}
\alias{repositionReferencePoint}
\title{Move Data Point a Specified Distance}
\usage{
repositionReferencePoint(
  x = NULL,
  id = NULL,
  dateTime = NULL,
  point.x = NULL,
  point.y = NULL,
  direction = NULL,
  repositionAngle = 0,
  repositionDist = 1,
  immobThreshold = 0,
  parallel = FALSE,
  nCores = (parallel::detectCores()/2),
  modelOrientation = 90
)
}
\arguments{
\item{x}{Data frame or list of data frames containing real-time-location 
point data.}

\item{id}{Vector of length nrow(data.frame(x)) or singular character data, 
detailing the relevant colname in x, that denotes what unique ids for 
tracked individuals will be used. If argument == NULL, the function 
assumes a column with the colname "id" exists in x. Defaults to NULL.}

\item{dateTime}{Vector of length nrow(data.frame(x)) or singular character 
data, detailing the relevant colname in x, that denotes what dateTime 
information will be used. If argument == NULL, the function assumes a 
column with the colname "dateTime" exists in x. Defaults to NULL.}

\item{point.x}{Vector of length nrow(data.frame(x)) or singular character 
data, detailing the relevant colname in x, that denotes what planar-x or 
longitude coordinate information will be used. If argument == NULL, the 
function assumes a column with the colname "x" exists in x. Defaults to 
NULL.}

\item{point.y}{Vector of length nrow(data.frame(x)) or singular character 
data, detailing the relevant colname in x, that denotes what planar-y or 
lattitude coordinate information will be used. If argument == NULL, the 
function assumes a column with the colname "y" exists in x. Defaults to 
NULL.}

\item{direction}{Numerical vector of length nrow(data.frame(x)) or singular 
character data detailing the relevant colname in x, that denotes what 
movement-direction information will be used. Observations in this vector
represent the direction (in degrees) that tracked individuals moved to 
reach their position at each time point, NOT the direction that they will
move to reach their subsequent position (i.e., values represent known
orientations at each time point). Note that for the purposes of this 
function, observations of 0, 90, 180, and 270 degrees indicates that an 
individual moved straight Eastward, Northward, Westward, and Southward,
respectively. If NULL, direction will be calculated using observed 
point-locations. Defaults to NULL.}

\item{repositionAngle}{Numerical. Describes the angle (in degrees) 
between empirical point-locations and the desired vertex location as 
represented in a planar model (see vignette or Farthing et al. in Review 
(note: when this manuscript is officially published, we will update this 
citation/reference information)). Essentially, this is the direction you
want new points to be from orginal points. Note that for the purposes of 
this function, observations of 0, 90, 180, and 270 degrees indicates that
an individual moved straight Eastward, Northward, Westward, and 
Southward, respectively. Defaults to 0.}

\item{repositionDist}{Numerical. Describes the distance from the empirical 
point-locations to desired vertex locations in planar units (e.g., 
meters) inherent to the real-time-location input. Defaults to 1.}

\item{immobThreshold}{Numerical. Describes what we call, the immobility 
threshold, which is a movement distance (in planar units) within which we
assume individuals’ physical locations and orientations remain unchanged.
This immobility threshold allows us to discount observed movements so 
miniscule that the majority of animals’ physical-space usage is likely 
unaffected (e.g., head shaking). Defaults to 0.}

\item{parallel}{Logical. If TRUE, sub-functions within the 
repositionReferencePoint wrapper will be parallelized. Note that this can
significantly speed up processing of relatively small data sets, but may 
cause R to crash due to lack of available memory when attempting to 
process large datasets. Defaults to FALSE.}

\item{nCores}{Integer. Describes the number of cores to be dedicated to 
parallel processes. Defaults to half of the maximum number of cores 
available (i.e., (parallel::detectCores()/2)).}

\item{modelOrientation}{Numerical. Describes the relative orientation (in 
degrees) of a planar model (see vignette or Farthing et al. in Press 
(note: when this manuscript is officially published, we will update this 
citation/reference information)) describing vertex locations relative to
tracking-device point-locations. Defaults to 90.}
}
\value{
Output is a data frame with the following columns:
      
   \item{id}{Unique ID of tracked individuals.}
   \item{x.original}{Original x coordinates from input.}
   \item{y.original}{Original y coordinates from input.}
   \item{distance.original}{Original planar distance (m) between 
   point-location i to point-location i + 1.}
   \item{dx.original}{Original difference between point-location 
   x-coordinate i to x-coordinate i + 1.}
   \item{dy.original}{Original difference between point-location 
   y-coordinate i to y-coordinate i + 1.}
   \item{x.adjusted}{Translated x coordinates.}
   \item{y.adjusted}{Translated y coordinates.}
   \item{dist.adjusted}{Planar distance (m) between translated
   point-location i to translated point-location i + 1.}
   \item{dx.adjusted}{Difference between translated point-location 
   x-coordinate i to translated x-coordinate i + 1.}
   \item{dy.adjusted}{Difference between translated point-location 
   y-coordinate i to translated y-coordinate i + 1.}
   \item{movementDirection}{Describes the angle of movement (in degrees) 
   required to translate point-locations to be congruent with planar-model
   adjustments.}
   \item{repositionAngle}{Describes the value \code{repositionAngle} of the
   argument.}
   \item{repositionDist}{Describes the value \code{repositionDist} of the
   argument.}
   \item{immob}{If "0", distance between observed movements is 
   < \code{immobThreshold}.}
   \item{immobThreshold}{Returns the value from the \code{immobThreshold}
   argument.}
   \item{dateTime}{Timepoint at which polygons were observed.}
   \item{dt}{The the time between reported xy coordinates in row i to row 
   i + 1 in each individuals' movement path.}
}
\description{
Translates locations of a single rfid tag/gps transmitter to a different 
   location a fixed distance away, given a known angular offset (in 
   degrees), while maintaining orientations associated with observed
   movements (see vignette or Farthing et al. in Review (note: when this 
   manuscript is officially published, we will update this 
   citation/reference information)) For example, calves in our study (see 
   calves2018) were equiped with RFID tags on their left ear. With this 
   function, we can move this reference point somewhere else on the body of 
   each individual. This might be done for a number of reasons, but is very 
   useful for use in the referencePoint2Polygon function later on (for 
   delineating polygons representing entire individuals). Currently, this 
   function only supports input data with coordinates representing planar 
   ('Euclidean') space (e.g. units of meters).
}
\details{
In this function, if the distance individuals moved was less than/equal to 
   the noted immobThreshold, individuals are said to be immobile ("immob"),
   and their position will not change relative to their previous one. (i.e.,
   you assume that any observed movement less than immobThreshold was due to
   errors or miniscule bodily movements (e.g., head shaking) that are not 
   indicative of actual movement.)

If distance == NULL, then function will require information (dist, dx, dy) 
   from 2 points on an individual's path to work properly. Because of this, 
   when no gyroscopic data are provided, at least the first point in each 
   individual's path will be removed (the function will report NAs for 
   adjusted locations). Also note that if the distance between an 
   individual's first point in their path and the second one is 0, the 
   function will also report NAs for the second point's adjusted 
   coordinates. The first non-NA values will only be reported for the 
   instance where dist > 0.

Note that populating the direction argument with gyroscopic accelerometer 
   data (or data collected using similar devices) collected concurrently 
   with point-locations allows us to overcome a couple of assumptions 
   associated with using point-locations alone.
   
   First, unless the direction argument is specifically given (i.e., 
   direction != NULL), new point-locations in output are subject to the 
   assumption that dt values are sufficiently small to capture individuals'
   orientations (i.e., individuals do not face unknown directions 
   inbetween observed relocations). If input was previously processed using 
   tempAggregate with resolutionLevel == "reduced," dt > secondAgg indicates
   that tracked individuals were missing in the original dataset for a 
   period of time. In this case, the assumption that individuals are facing 
   a given direction because they moved from the previous timepoint may not 
   be accurate. Consider removing these rows (rows following one with 
   dt > secondAgg; remember that dt indicates the time between recording xy 
   coordinates in row i to row i + 1) from your data set.

   Second, unless the direction argument is specifically given (i.e., 
   direction != NULL), this function assumes tracked individuals are always 
   forward-facing. This is because by observing only a single point on each 
   individual, we cannot ascertain the true positioning of individuals' 
   bodies. For example, even if we know a point-location moved x distance in
   a 90-degree direction, from this information alone we cannot determine 
   what direction said individual was facing at the time (e.g., this could
   be an example of forward, bawckward, or sideward movement). However, 
   gyroscopic data (or data collected using similar devices) can tell us 
   absolute movement directions, as opposed to relative ones.
}
\examples{
\donttest{
data("calves")
calves.dateTime<-datetime.append(calves, date = calves$date, 
   time = calves$time) #create a dataframe with dateTime identifiers for location fixes.

calves.agg<-tempAggregate(calves.dateTime, id = calves.dateTime$calftag, 
   dateTime = calves.dateTime$dateTime, point.x = calves.dateTime$x, 
   point.y = calves.dateTime$y, secondAgg = 300, extrapolate.left = FALSE, 
   extrapolate.right = FALSE, resolutionLevel = "reduced", parallel = FALSE, 
   na.rm = TRUE, smooth.type = 1) #smooth to 5-min fix intervals.

leftShoulder.point<-repositionReferencePoint(x = calves.agg, 
   id = calves.agg$id, dateTime = calves.agg$dateTime, 
   point.x = calves.agg$x, point.y = calves.agg$y, direction = NULL, 
   repositionAngle = 180, repositionDist = 0.0835, immobThreshold = 0, parallel = FALSE, 
   modelOrientation = 90)
   }
}
\references{
Farthing, T.S., Dawson, D.E., Sanderson, M.W., and Lanzas, 
   C. in Press. Accounting for space and uncertainty in real-time-location-
   system-derived contact networks. Ecology and Evolution.
}
\keyword{data-processing}
\keyword{location}
\keyword{planar}
\keyword{point}
