% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Simulations-methods.R
\docType{methods}
\name{summary,DualSimulations-method}
\alias{summary,DualSimulations-method}
\title{Summarize the dual-endpoint design simulations, relative to given true
dose-toxicity and dose-biomarker curves}
\usage{
\S4method{summary}{DualSimulations}(object, trueTox, trueBiomarker,
  target = c(0.2, 0.35), ...)
}
\arguments{
\item{object}{the \code{\linkS4class{DualSimulations}} object we want to
summarize}

\item{trueTox}{a function which takes as input a dose (vector) and returns the
true probability (vector) for toxicity.}

\item{trueBiomarker}{a function which takes as input a dose (vector) and
returns the true biomarker level (vector).}

\item{target}{the target toxicity interval (default: 20-35\%) used for the
computations}

\item{\dots}{Additional arguments can be supplied here for \code{trueTox}
and \code{trueBiomarker}}
}
\value{
an object of class \code{\linkS4class{DualSimulationsSummary}}
}
\description{
Summarize the dual-endpoint design simulations, relative to given true
dose-toxicity and dose-biomarker curves
}
\examples{

# Define the dose-grid
emptydata <- DataDual(doseGrid = c(1, 3, 5, 10, 15, 20, 25, 40, 50, 80, 100))

# Initialize the CRM model 
model <- DualEndpointRW(mu = c(0, 1),
                        Sigma = matrix(c(1, 0, 0, 1), nrow=2),
                        sigma2betaW = 0.01,
                        sigma2W = c(a=0.1, b=0.1),
                        rho = c(a=1, b=1),
                        smooth="RW1")

# Choose the rule for selecting the next dose 
myNextBest <- NextBestDualEndpoint(target=c(0.9, 1),
                                   overdose=c(0.35, 1),
                                   maxOverdoseProb=0.25)

# Choose the rule for stopping
myStopping4 <- StoppingTargetBiomarker(target=c(0.9, 1),
                                       prob=0.5)
# StoppingMinPatients will usually take a higher sample size,
# just for illustration here
myStopping <- myStopping4 | StoppingMinPatients(6)

# Choose the rule for dose increments
myIncrements <- IncrementsRelative(intervals=c(0, 20),
                                   increments=c(1, 0.33))

# Initialize the design
design <- DualDesign(model = model,
                     data = emptydata,
                     nextBest = myNextBest,
                     stopping = myStopping,
                     increments = myIncrements,
                     cohortSize = CohortSizeConst(3),
                     startingDose = 3)
  
# define scenarios for the TRUE toxicity and efficacy profiles
betaMod <- function (dose, e0, eMax, delta1, delta2, scal)
{
  maxDens <- (delta1^delta1) * (delta2^delta2)/((delta1 + delta2)^(delta1 + delta2))
  dose <- dose/scal
  e0 + eMax/maxDens * (dose^delta1) * (1 - dose)^delta2
}

trueBiomarker <- function(dose)
{
  betaMod(dose, e0=0.2, eMax=0.6, delta1=5, delta2=5 * 0.5 / 0.5, scal=100)
}

trueTox <- function(dose)
{
  pnorm((dose-60)/10)
}

# Draw the TRUE profiles
par(mfrow=c(1, 2))
curve(trueTox(x), from=0, to=80)
curve(trueBiomarker(x), from=0, to=80)

# Run the simulation on the desired design
# We only generate 1 trial outcome here for illustration, for the actual study
# Also for illustration purpose, we will use 5 burn-ins to generate 20 samples
# this should be increased of course!
mySims <- simulate(design,
                   trueTox=trueTox,
                   trueBiomarker=trueBiomarker,
                   sigma2W=0.01,
                   rho=0,
                   nsim=1,
                   parallel=FALSE,
                   seed=3,
                   startingDose=6,
                   mcmcOptions =
                     McmcOptions(burnin=5,
                                 step=1,
                                 samples=20))

# Summarize the Results of the Simulations
summary(mySims,
        trueTox = trueTox,
        trueBiomarker = trueBiomarker)



  
}
\keyword{methods}
