\name{dlm}
\alias{dlm}

\title{
Implement finite distributed lag model
}
\description{
A function that applies distributed lag models with one or multiple predictor(s).
}
\usage{
dlm(formula , data , x , y , q , remove , show.summary = TRUE)
}

\arguments{
  \item{formula}{A \code{formula} object for the model to be fitted. In the case of multiple predictor series, the model should be entered via a formula object. 
  }
  \item{data}{  A \code{data.frame} including all dependent and independent series. In the case of multiple predictor series, the data should be entered via the \code{data} argument.
  }
  \item{x}{
  A vector including the observations of predictor time series. This is not restricted to \code{ts} objects. If the series are supplied by \code{data} 
  }
  \item{y}{
  A vector including the observations of dependent time series. This is not restricted to \code{ts} objects.
  } 
  \item{q}{
  An integer representing finite lag length.
  }  
  \item{remove}{
  A matrix showing the lags to be removed from the model for each independent series. Please see the details for the construction of this argument.
  }
\item{show.summary}{If \code{TRUE}, prints standard model summary for the model of interest.
  }  
 
}

\details{
When a decision made on a variable, some of the related variables would be effected through time. For example, when income tax rate is increased, this would reduce expenditures of consumers on goods and services, which reduces profits of suppliers, which reduces the demand for productive inputs, which reduces the profits of the input suppliers, and so on (Judge and Griffiths, 2000). These effects occur over the future time periods; hence, they are distributed across the time.    

In a distributed-lag model, the effect of an independent variable \eqn{X} on a dependent variable \eqn{Y} occurs over the time. Therefore, DLMs are dynamic models. A linear finite DLM with one independent variable is written as follows:


\deqn{
      Y_{t} = \alpha +\sum_{s = 0}^{q}\beta_{s}X_{t-s}+\epsilon_{t},
      }

where \eqn{\epsilon_{t}} is a stationary error term with \eqn{E(\epsilon_{t})=0, Var(\epsilon_{t})=\sigma^{2},Cov(\epsilon_{t},\epsilon_{s})=0}.

When there is only one predictor series, both of \code{model} and \code{formula} objects can be used. But when they are supplied, both \code{x} and \code{y} arguments should be \code{NULL}.

The variable names in \code{formula} must match with the names of variables in \code{data} argument and it must be in the form of a generic formula for R functions. 

The argument \code{data} contains dependent series and independent series. Required lags of dependent series are generated by the \code{dlm} function automatically. 

The argument \code{remove} is used to specify which lags will be removed from the full model. Each row of \code{remove} includes the numbers of lags to be removed in its columns and the rest of the elements should be set to \code{NA}. Notice that it is possible to fit a model with different lag lengths for each independent series by removing the appropriate lags of independent series with \code{remove} argument.

}

\value{

\item{model}{An object of class \code{lm}.}

\item{designMatrix}{The design matrix composed of transformed z-variables.}

\item{k}{The number of independent series. This is returned if multiple independent series are entered.}

\item{q}{The lag length.}

\item{removed}{A matrix showing the removed lags from the model for each independent series. Returns \code{NULL} if the fitted model is full.}

\item{formula}{Model formula of the fitted model. This is returned if multiple independent series are entered.}

\item{data}{A \code{data.frame} including all dependent and independent series. This is returned if multiple independent series are entered.}
}

\references{
B.H. Baltagi. \emph{Econometrics}, Fifth Ed. Springer, 2011.

R.C. Hill, W.E. Griffiths, G.G. Judge. \emph{Undergraduate Econometrics}. Wiley, 2000. 
}

\author{
Haydar Demirhan

Maintainer: Haydar Demirhan <haydar.demirhan@rmit.edu.au>
}

\examples{
# Only one independent series
data(warming)
model.dlm = dlm(x = warming$NoMotorVehicles , 
y = warming$Warming , q = 2 ,  show.summary = TRUE)

# Multiple independent series
data(M1Germany)
data = M1Germany[1:144,]
model.dlm  = dlm(formula = logprice ~ interest + logm1, 
          data = data.frame(data) , q = 4 , show.summary = TRUE)

removed = matrix(c(1,3,2,NA), nrow = 2, ncol = 2, byrow = TRUE)
removed
model.dlm  = dlm(formula = logprice ~ interest + logm1, 
data = data.frame(data) , q = 4 , remove = removed , show.summary = TRUE)

removed = matrix(c(1,3) , 1 , 2)
model.dlm  = dlm(formula = logprice ~ interest , data = data.frame(data) , 
                  q = 4 , remove = removed , show.summary = TRUE)

}
