## ----echo = FALSE, message=FALSE----------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>"
)
library(dRiftDM)
set.seed(1014)

## -----------------------------------------------------------------------------
ddm <- dmc_dm()

## -----------------------------------------------------------------------------
print(ddm)

## -----------------------------------------------------------------------------
five_traces <- simulate_traces(object = ddm, k = 5)
five_traces

## -----------------------------------------------------------------------------
five_traces <- simulate_traces(object = ddm, k = 5, add_x = TRUE)
five_traces

## -----------------------------------------------------------------------------
plot(five_traces, col = c("green", "red"))

## -----------------------------------------------------------------------------
exp_behavior <- simulate_traces(object = ddm, k = 1, sigma = 0)
plot(exp_behavior, col = c("green", "red"))

## -----------------------------------------------------------------------------
sum_stats <- calc_stats(object = ddm, type = c("cafs", "quantiles"))
sum_stats

## -----------------------------------------------------------------------------
plot(sum_stats, col = c("green", "red"))

## -----------------------------------------------------------------------------
coef(ddm)

## -----------------------------------------------------------------------------
coef(ddm)["muc"] <- 5
coef(ddm)

## -----------------------------------------------------------------------------
coef(ddm, select_unique = FALSE)

## -----------------------------------------------------------------------------
prms_solve(ddm)

## -----------------------------------------------------------------------------
prms_solve(ddm)["dt"] <- .0025
prms_solve(ddm)

## -----------------------------------------------------------------------------
solver(ddm)

## -----------------------------------------------------------------------------
b_coding(ddm)

## -----------------------------------------------------------------------------
copy <- ddm # to not change the original model object
b_coding(copy)$column <- "Response"
b_coding(copy)$u_name_value <- c("left" = -1)
b_coding(copy)$l_name_value <- c("right" = 1)

b_coding(copy)

## -----------------------------------------------------------------------------
data <- dRiftDM::dmc_synth_data # some synthetic data suitable for DMC that ships with dRiftDM
# the Cond column matches with conds(ddm).
# The Error column matches b_coding(ddm)
# the RT column is in seconds ;)
head(data)

obs_data(ddm) <- data

## -----------------------------------------------------------------------------
summary(ddm)

## -----------------------------------------------------------------------------
# get some data (here we use again the synthetic data that ships with dRiftDM)
data <- dRiftDM::dmc_synth_data
head(data)

# increase the discretization steps to 0.005 and set the maximum time space to 1.5 seconds
prms_solve(ddm)["dt"] <- .005
prms_solve(ddm)["dx"] <- .005
prms_solve(ddm)["t_max"] <- 1.5
max(data$RT) # maximum time space easily covers the maximum RT

# attach the data to the model
obs_data(ddm) <- data

# now call the estimation routine
ddm <- estimate_model(
  drift_dm_obj = ddm,
  lower = c(muc = 1, b = .3, non_dec = .1, sd_non_dec = .005, tau = .03, A = .01, alpha = 2),
  upper = c(muc = 6, b = .9, non_dec = .5, sd_non_dec = .050, tau = .12, A = .15, alpha = 9),
  use_de_optim = FALSE, # overrule the default Differential Evolution setting
  use_nmkb = TRUE
)

coef(ddm)

## -----------------------------------------------------------------------------
logLik(ddm)
AIC(ddm)
BIC(ddm)

## -----------------------------------------------------------------------------
check_fit <- calc_stats(object = ddm, type = c("cafs", "quantiles"))
plot(check_fit, col = c("green", "red"))

## -----------------------------------------------------------------------------
flanker_data <- dRiftDM::ulrich_flanker_data
head(flanker_data)
flanker_data <- flanker_data[flanker_data$ID %in% c(1, 2), ]
obs_data(ddm) <- NULL # detach data (from the previous sections) to avoid a warning

estimate_model_ids(
  drift_dm_obj = ddm,
  obs_data_ids = flanker_data,
  lower = c(muc = 1, b = .3, non_dec = .1, sd_non_dec = .005, tau = .03, A = .01, alpha = 2),
  upper = c(muc = 6, b = .9, non_dec = .5, sd_non_dec = .050, tau = .12, A = .15, alpha = 9),
  fit_procedure_name = "flanker_test_run", # a label to identify the fits
  fit_path = tempdir(), # to save fits in the working directory use getwd()
  use_de_optim = FALSE, # overrule the default Differential Evolution setting
  use_nmkb = TRUE
)

## -----------------------------------------------------------------------------
all_fits <- load_fits_ids(path = tempdir(), fit_procedure_name = "flanker_test_run")
all_fits

## -----------------------------------------------------------------------------
summary(all_fits)

## -----------------------------------------------------------------------------
coef_fits <- coef(all_fits)
coef_fits

## -----------------------------------------------------------------------------
logLik(all_fits)
AIC(all_fits)
BIC(all_fits)

## -----------------------------------------------------------------------------
check_fit <- calc_stats(object = all_fits, type = c("cafs", "quantiles"))
plot(check_fit, col = c("green", "red"))

## -----------------------------------------------------------------------------
ddm <- ratcliff_dm() # a model for demonstration purpose
sim_1 <- simulate_data(object = ddm, n = 200)
head(sim_1)

## -----------------------------------------------------------------------------
sim_2 <- simulate_data(
  object = ddm, n = 200, k = 2,
  lower = c(muc = 1, b = .4, non_dec = .2),
  upper = c(muc = 6, b = .8, non_dec = .4)
)

## -----------------------------------------------------------------------------
head(sim_2$synth_data)
head(sim_2$prms)

## -----------------------------------------------------------------------------
ddm <- dmc_dm(dt = .005, dx = .005)
traces <- simulate_traces(ddm, k = 2)
# although this object is essentially a list of matrices, the class label ...
class(traces)
print(traces) # ... leads to nicely formatted output; but hides the underlying structure

raw <- unpack_obj(traces) # provides the plain list of matrices
head(t(raw$comp))

