#' Filter loci in a genlight \{adegenet\} object based on average reproducibility of alleles at a locus
#'
#' SNP datasets generated by DArT have in index, RepAvg, generated by reproducing the data independently for 30\% of loci.
#' RepAvg is the proportion of alleles that give a reproducible result, averaged over both alleles for each locus.
#'
#' @param gl -- name of the genlight object containing the SNP data [required]
#' @param t -- threshold value below which loci will be removed [default 0.95]
#' @return     Returns a genlight object retaining loci with a RepAvg less than the specified threshold deleted.
#' @export
#' @author Arthur Georges (glbugs@aerg.canberra.edu.au)
#' @examples
#' gl.report.repavg(testset.gl)
#' result <- gl.filter.repavg(testset.gl, t=0.95)

gl.filter.repavg <- function(gl, t=1) {
x <- gl

  if(class(x) == "genlight") {
    cat("Reporting for a genlight object\n")
  } else {
    cat("Fatal Error: Specify a genlight object\n")
    stop()
  }
  cat("Note: RepAvg is a DArT statistic reporting reproducibility averaged across alleles for each locus. \n\n")
  
  n0 <- nLoc(x)
  cat("Initial no. of loci =", n0, "\n")

  if(class(x)=="genlight") {
    # Remove SNP loci with RepAvg < t
    x2 <- x[, x@other$loc.metrics["RepAvg"]>=t]
    # Remove the corresponding records from the loci metadata
    x2@other$loc.metrics <- x@other$loc.metrics[x@other$loc.metrics["RepAvg"]>=t,]
    cat ("No. of loci deleted =", (n0-nLoc(x2)),"\n")
    
  } else if (class(x)=="genind") {
    x2 <- x[,(colSums(is.na(tab((x))))/nInd(x))<(1-t)]
    idx <- which((colSums(is.na(tab((x))))/nInd(x))<(1-t))
    x2@other$loc.metrics <- x@other$loc.metrics[c(idx),]
    
    # Remove SNP loci with RepAvg < t
    x2 <- x[, x@other$loc.metrics["Reproducibility"]>=t]
    # Remove the corresponding records from the loci metadata
    x2@other$loc.metrics <- x@other$loc.metrics[x@other$loc.metrics["Reproducibility"]>=t,]
    cat ("No. of loci deleted =", (n0-nLoc(x2)),"\n")
  } else {
    cat("Fatal Error: genlight or genind objects required for call rate filtering!\n"); stop()
  }
  
  # REPORT A SUMMARY
  cat("Summary of filtered dataset\n")
  cat(paste("  Reproducibility >=",t,"\n"))
  cat(paste("  No. of loci:",nLoc(x2),"\n"))
  cat(paste("  No. of individuals:", nInd(x2),"\n"))
  cat(paste("  No. of populations: ", length(levels(factor(pop(x2)))),"\n"))
  
  return(x2)
  
}
