#' Filter loci in a genlight \{adegenet\} object based on average repeatability of alleles at a locus
#'
#' SNP datasets generated by DArT have in index, RepAvg, generated by reproducing the data independently for 30\% of loci.
#' RepAvg is the proportion of alleles that give a repeatable result, averaged over both alleles for each locus.
#'
#' @param x -- name of the genlight object containing the SNP data [required]
#' @param threshold -- threshold value below which loci will be removed [default 0.99]
#' @param v -- verbosity: 0, silent or fatal errors; 1, begin and end; 2, progress log ; 3, progress and results summary; 5, full report [default 2]
#' @return     Returns a genlight object retaining loci with a RepAvg greater than the specified threshold deleted.
#' @export
#' @author Arthur Georges (Post to \url{https://groups.google.com/d/forum/dartr})
#' @examples
#' gl.report.repavg(testset.gl)
#' result <- gl.filter.repavg(testset.gl, threshold=0.95, v=3)

# Last edit:25-Apr-18

gl.filter.repavg <- function(x, threshold=0.99, v=2) {

# ERROR CHECKING
  
  if(class(x)!="genlight") {
    cat("Fatal Error: genlight object required!\n"); stop("Execution terminated\n")
  }
  
  if (v < 0 | v > 5){
    cat("    Warning: verbosity must be an integer between 0 [silent] and 5 [full report], set to 2\n")
    v <- 2
  }
  
  if (v > 0) {
    cat("Starting gl.filter.repavg: Filtering on repeatability\n")
  }
  
  if (v > 2) {cat("Note: RepAvg is a DArT statistic reporting repeatability averaged across alleles for each locus. \n\n")}
  
  n0 <- nLoc(x)
  if (v > 2) {cat("Initial no. of loci =", n0, "\n")}

  if(class(x)=="genlight") {
    # Remove SNP loci with RepAvg < threshold
    if (v > 1){cat("  Removing loci with RepAvg <",threshold,"\n")}
    x2 <- x[, x@other$loc.metrics["RepAvg"]>=threshold]
    # Remove the corresponding records from the loci metadata
    x2@other$loc.metrics <- x@other$loc.metrics[x@other$loc.metrics["RepAvg"]>=threshold,]
    if (v > 2) {cat ("No. of loci deleted =", (n0-nLoc(x2)),"\n")}
    
  } else if (class(x)=="genind") {
    x2 <- x[,(colSums(is.na(tab((x))))/nInd(x))<(1-threshold)]
    idx <- which((colSums(is.na(tab((x))))/nInd(x))<(1-threshold))
    x2@other$loc.metrics <- x@other$loc.metrics[c(idx),]
    
    # Remove SNP loci with RepAvg < threshold
    if (v > 1){cat("  Removing loci with Reproducibility <",threshold,"\n")}
    x2 <- x[, x@other$loc.metrics["Reproducibility"]>=threshold]
    # Remove the corresponding records from the loci metadata
    x2@other$loc.metrics <- x@other$loc.metrics[x@other$loc.metrics["Reproducibility"]>=threshold,]
    if (v > 2) {cat ("No. of loci deleted =", (n0-nLoc(x2)),"\n")}
  } else {
    cat("Fatal Error: genlight or genind objects required for call rate filtering!\n"); stop()
  }
  
  # REPORT A SUMMARY
  if (v > 2) {
    cat("Summary of filtered dataset\n")
    cat(paste("  Repeatability >=",threshold,"\n"))
    cat(paste("  No. of loci:",nLoc(x2),"\n"))
    cat(paste("  No. of individuals:", nInd(x2),"\n"))
    cat(paste("  No. of populations: ", length(levels(factor(pop(x2)))),"\n"))
  }  
  
  if ( v > 0) {cat("gl.filter.repavg completed\n")}
  
  return(x2)
  
}
