% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/difNLR.R
\name{difNLR}
\alias{difNLR}
\alias{print.difNLR}
\alias{fitted.difNLR}
\alias{coef.difNLR}
\alias{logLik.difNLR}
\alias{AIC.difNLR}
\alias{BIC.difNLR}
\alias{residuals.difNLR}
\alias{resid.difNLR}
\title{DIF detection using non-linear regression method.}
\usage{
difNLR(Data, group, focal.name, model, constraints, type = "all", method = "nls",
match = "zscore", anchor = NULL, purify = FALSE, nrIter = 10, test = "LR", alpha = 0.05,
p.adjust.method = "none", start, initboot = T, nrBo = 20)

\method{print}{difNLR}(x, ...)

\method{fitted}{difNLR}(object, item = "all", ...)

\method{coef}{difNLR}(object, SE = FALSE, simplify = FALSE, ...)

\method{logLik}{difNLR}(object, item = "all", ...)

\method{AIC}{difNLR}(object, item = "all", ...)

\method{BIC}{difNLR}(object, item = "all", ...)

\method{residuals}{difNLR}(object, item = "all", ...)
}
\arguments{
\item{Data}{data.frame or matrix: dataset which rows represent scored examinee answers (\code{"1"}
correct, \code{"0"} incorrect) and columns correspond to the items. In addition, \code{Data} can
hold the vector of group membership.}

\item{group}{numeric or character: a dichotomous vector of the same length as \code{nrow(Data)}
or a column identifier of \code{Data}.}

\item{focal.name}{numeric or character: indicates the level of \code{group} which corresponds to
focal group.}

\item{model}{character: generalized logistic regression model to be fitted. See \strong{Details}.}

\item{constraints}{character: which parameters should be the same for both groups. Possible values
are any combinations of parameters \code{"a"}, \code{"b"}, \code{"c"}, and \code{"d"}.
See \strong{Details}.}

\item{type}{character: type of DIF to be tested. Possible values are \code{"all"} for detecting
difference in any parameter (default), \code{"udif"} for uniform DIF only (i.e., difference in
difficulty parameter \code{"b"}), \code{"nudif"} for non-uniform DIF only (i.e., difference in
discrimination parameter \code{"a"}), \code{"both"} for uniform and non-uniform DIF (i.e.,
difference in parameters \code{"a"} and \code{"b"}), or combination of parameters \code{"a"},
\code{"b"}, \code{"c"}, and \code{"d"}. Can be specified as a single value (for all items) or as
an item-specific vector.}

\item{method}{character: method used to estimate parameters. Either \code{"nls"} for
non-linear least squares (default), or \code{"likelihood"} for maximum likelihood method.}

\item{match}{numeric or character: matching criterion to be used as an estimate of trait. Can be
either \code{"zscore"} (default, standardized total score), \code{"score"} (total test score),
or vector of the same length as number of observations in \code{Data}.}

\item{anchor}{numeric or character: specification of DIF free items. Either \code{NULL} (default),
or a vector of item names (column names of \code{Data}), or item identifiers (integers specifying
the column number) determining which items are currently considered as anchor (DIF free) items.
Argument is ignored if \code{match} is not \code{"zscore"} or \code{"score"}.}

\item{purify}{logical: should the item purification be applied? (default is \code{FALSE}).}

\item{nrIter}{numeric: the maximal number of iterations in the item purification (default is 10).}

\item{test}{character: test to be performed for DIF detection. Can be either \code{"LR"} for
likelihood ratio test of a submodel (default), or \code{"F"} for F-test of a submodel.}

\item{alpha}{numeric: significance level (default is 0.05).}

\item{p.adjust.method}{character: method for multiple comparison correction. Possible values are
\code{"holm"}, \code{"hochberg"}, \code{"hommel"}, \code{"bonferroni"}, \code{"BH"}, \code{"BY"},
\code{"fdr"}, and \code{"none"} (default). For more details see \code{\link[stats]{p.adjust}}.}

\item{start}{numeric: initial values for estimation of parameters. If not specified, starting
values are calculated with \code{\link[difNLR]{startNLR}} function. Otherwise, list with as many
elements as a number of items. Each element is a named numeric vector of length 8 representing initial
values for parameter estimation. Specifically, parameters \code{"a"}, \code{"b"}, \code{"c"}, and
\code{"d"} are initial values for discrimination, difficulty, guessing, and inattention for reference
group. Parameters \code{"aDif"}, \code{"bDif"}, \code{"cDif"}, and \code{"dDif"} are then differences
in these parameters between reference and focal group.}

\item{initboot}{logical: in case of convergence issues, should be starting values re-calculated based
on bootstraped samples? (default is \code{TRUE}; newly calculated initial values are applied only to
items/models with convergence issues).}

\item{nrBo}{numeric: the maximal number of iterations for calculation of starting values using
bootstraped samples (default is 20).}

\item{x}{an object of \code{"difNLR"} class.}

\item{...}{other generic parameters for S3 methods.}

\item{object}{an object of \code{"difNLR"} class.}

\item{item}{numeric or character: either character \code{"all"} to apply for all converged items (default),
or a vector of item names (column names of \code{Data}), or item identifiers (integers specifying
the column number).}

\item{SE}{logical: should the standard errors of estimated parameters be also returned? (default is \code{FALSE}).}

\item{simplify}{logical: should the estimated parameters be simplified to a matrix? (default is \code{FALSE}).}
}
\value{
The \code{difNLR()} function returns an object of class \code{"difNLR"}. The output
including values of the test statistics, p-values, and items detected as function differently
is displayed by the \code{print()} method.

Item characteristic curves and graphical representation of DIF statistics can be displayed
with \code{plot()} method. For more details see \code{\link[difNLR]{plot.difNLR}}.
Estimated parameters can be displayed with \code{coef()} method.

Fitted values and residuals can be extracted by the \code{fitted()} and \code{residuals()} methods
for converged item(s) specified in \code{item} argument.

Predicted values are produced by the \code{predict()} method for converged item(s)
specified in \code{item} argument. New data can be introduced with \code{match} and \code{group}
arguments. For more details see \code{\link[difNLR]{predict.difNLR}}.

Log-likelihood, Akaike's information criterion, and Schwarz's Bayesian criterion can be
extracted with methods \code{logLik()}, \code{AIC()}, \code{BIC()} for converged item(s)
specified in \code{item} argument.

Object of class \code{"difNLR"} is a list with the following components:
\describe{
  \item{\code{Sval}}{the values of the test statistics.}
  \item{\code{nlrPAR}}{the estimates of final model.}
  \item{\code{nlrSE}}{the standard errors of estimates of final model.}
  \item{\code{parM0}}{the estimates of null model.}
  \item{\code{seM0}}{the standard errors of estimates of null model.}
  \item{\code{covM0}}{the covariance matrices of estimates of null model.}
  \item{\code{llM0}}{log-likelihood of null model.}
  \item{\code{parM1}}{the estimates of alternative model.}
  \item{\code{seM1}}{the standard errors of estimates of alternative model.}
  \item{\code{covM1}}{the covariance matrices of estimates of alternative model.}
  \item{\code{llM1}}{log-likelihood of alternative model.}
  \item{\code{DIFitems}}{either the column identifiers of the items which were detected as DIF, or
  \code{"No DIF item detected"} in case no item was detected as function differently.}
  \item{\code{model}}{fitted model.}
  \item{\code{constraints}}{constraints for the \code{model}.}
  \item{\code{type}}{character: type of DIF that was tested. If parameters were specified, the value is \code{"other"}.}
  \item{\code{types}}{character: the parameters (specified by user, \code{type} has value \code{"other"}) which were
  tested for difference.}
  \item{\code{p.adjust.method}}{character: method for multiple comparison correction which was applied.}
  \item{\code{pval}}{the p-values of the test.}
  \item{\code{adj.pval}}{the adjusted p-values of the test using \code{p.adjust.method}.}
  \item{\code{df}}{the degress of freedom of the test.}
  \item{\code{test}}{used test.}
  \item{\code{purification}}{\code{purify} value.}
  \item{\code{nrPur}}{number of iterations in item purification process. Returned only if \code{purify}
  is \code{TRUE}.}
  \item{\code{difPur}}{a binary matrix with one row per iteration of item purification and one column per item.
  \code{"1"} in i-th row and j-th column means that j-th item was identified as DIF in i-th iteration. Returned only
  if \code{purify} is \code{TRUE}.}
  \item{\code{conv.puri}}{logical: indicating whether item purification process converged before the maximal number
  \code{nrIter} of iterations. Returned only if \code{purify} is \code{TRUE}.}
  \item{\code{method}}{used estimation method.}
  \item{\code{conv.fail}}{numeric: number of convergence issues.}
  \item{\code{conv.fail.which}}{the identifiers of the items which did not converge.}
  \item{\code{alpha}}{numeric: significance level.}
  \item{\code{Data}}{the data matrix.}
  \item{\code{group}}{the vector of group membership.}
  \item{\code{group.names}}{names of groups.}
  \item{\code{match}}{matching criterion.}
}
}
\description{
Performs DIF detection procedure for dichotomous data based on non-linear
regression model (generalized logistic regression) and either likelihood-ratio or F test
of a submodel.
}
\details{
DIF detection procedure based on non-linear regression is the extension of logistic regression
procedure (Swaminathan and Rogers, 1990; Drabinova and Martinkova, 2017).

The unconstrained form of 4PL generalized logistic regression model for probability of correct
answer (i.e., \eqn{y = 1}) is
\deqn{P(y = 1) = (c + cDif*g) + (d + dDif*g - c - cDif*g)/(1 + exp(-(a + aDif*g)*(x - b - bDif*g))), }
where \eqn{x} is by default standardized total score (also called Z-score) and \eqn{g} is a group membership.
Parameters \eqn{a}, \eqn{b}, \eqn{c}, and \eqn{d} are discrimination, difficulty, guessing, and inattention.
Terms \eqn{aDif}, \eqn{bDif}, \eqn{cDif}, and \eqn{dDif} then represent differences between two groups
(reference and focal) in relevant parameters.

This 4PL model can be further constrained by \code{model} and \code{constraints} arguments.
The arguments \code{model} and \code{constraints} can be also combined. Both arguments can
be specified as a single value (for all items) or as an item-specific vector (where each
element correspond to one item).

The \code{model} argument offers several predefined models. The options are as follows:
\code{Rasch} for 1PL model with discrimination parameter fixed on value 1 for both groups,
\code{1PL} for 1PL model with discrimination parameter fixed for both groups,
\code{2PL} for logistic regression model,
\code{3PLcg} for 3PL model with fixed guessing for both groups,
\code{3PLdg} for 3PL model with fixed inattention for both groups,
\code{3PLc} (alternatively also \code{3PL}) for 3PL regression model with guessing parameter,
\code{3PLd} for 3PL model with inattention parameter,
\code{4PLcgdg} for 4PL model with fixed guessing and inattention parameter for both groups,
\code{4PLcgd} (alternatively also \code{4PLd}) for 4PL model with fixed guessing for both groups,
\code{4PLcdg} (alternatively also \code{4PLc}) for 4PL model with fixed inattention for both groups,
or \code{4PL} for 4PL model.

The \code{model} can be specified in more detail with \code{constraints} argument which specifies what
parameters should be fixed for both groups. For example, choice \code{"ad"} means that discrimination
(parameter \code{"a"}) and inattention (parameter \code{"d"}) are fixed for both groups and other parameters
(\code{"b"} and \code{"c"}) are not. The \code{NA} value for \code{constraints} means no constraints.

Missing values are allowed but discarded for item estimation. They must be coded as
\code{NA} for both, \code{Data} and \code{group} arguments.

In case that model considers difference in guessing or inattention parameter, the different parameterization is
used and parameters with standard errors are re-calculated by delta method. However, covariance matrices stick with
alternative parameterization.
}
\examples{
# loading data based on GMAT
data(GMAT)

Data <- GMAT[, 1:20]
group <- GMAT[, "group"]

# Testing both DIF effects using likelihood-ratio test and
# 3PL model with fixed guessing for groups
(x <- difNLR(Data, group, focal.name = 1, model = "3PLcg"))

# Graphical devices
plot(x, item = x$DIFitems)
plot(x, item = "Item1")
plot(x, item = 1, group.names = c("Group 1", "Group 2"))
plot(x, plot.type = "stat")

# Coefficients
coef(x)
coef(x, SE = TRUE)
coef(x, SE = TRUE, simplify = TRUE)

# Fitted values
fitted(x)
fitted(x, item = 1)

# Residuals
residuals(x)
residuals(x, item = 1)

# Predicted values
predict(x)
predict(x, item = 1)

# Predicted values for new subjects
predict(x, item = 1, match = 0, group = 0)
predict(x, item = 1, match = 0, group = 1)

# AIC, BIC, log-likelihood
AIC(x)
BIC(x)
logLik(x)
# AIC, BIC, log-likelihood for the first item
AIC(x, item = 1)
BIC(x, item = 1)
logLik(x, item = 1)
\dontrun{
# Testing both DIF effects using F test and
# 3PL model with fixed guessing for groups
difNLR(Data, group, focal.name = 1, model = "3PLcg", test = "F")

# Testing both DIF effects using LR test,
# 3PL model with fixed guessing for groups
# and Benjamini-Hochberg correction
difNLR(Data, group, focal.name = 1, model = "3PLcg", p.adjust.method = "BH")

# Testing both DIF effects using LR test,
# 3PL model with fixed guessing for groups
# and item purification
difNLR(Data, group, focal.name = 1, model = "3PLcg", purify = TRUE)

# Testing both DIF effects using 3PL model with fixed guessing for groups
# and total score as matching criterion
difNLR(Data, group, focal.name = 1, model = "3PLcg", match = "score")

# Testing uniform DIF effects using 4PL model with the same
# guessing and inattention
difNLR(Data, group, focal.name = 1, model = "4PLcgdg", type = "udif")

# Testing non-uniform DIF effects using 2PL model
difNLR(Data, group, focal.name = 1, model = "2PL", type = "nudif")

# Testing difference in parameter b using 4PL model with fixed
# a and c parameters
difNLR(Data, group, focal.name = 1, model = "4PL", constraints = "ac", type = "b")

# Testing both DIF effects using LR test,
# 3PL model with fixed guessing for groups
# with maximum likelihood estimation method
difNLR(Data, group, focal.name = 1, model = "3PLcg", method = "likelihood")
}

}
\references{
Drabinova, A. & Martinkova P. (2017). Detection of Differential Item Functioning with NonLinear Regression:
Non-IRT Approach Accounting for Guessing. Journal of Educational Measurement, 54(4), 498-517,
\url{https://doi.org/10.1111/jedm.12158}.

Swaminathan, H. & Rogers, H. J. (1990). Detecting Differential Item Functioning Using Logistic Regression Procedures.
Journal of Educational Measurement, 27(4), 361-370,
\url{https://doi.org/10.1111/j.1745-3984.1990.tb00754.x}
}
\seealso{
\code{\link[stats]{nls}} \code{\link[stats]{p.adjust}} \code{\link[difNLR]{plot.difNLR}} \code{\link[difNLR]{startNLR}}
}
\author{
Adela Hladka (nee Drabinova) \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
Faculty of Mathematics and Physics, Charles University \cr
\email{hladka@cs.cas.cz} \cr

Patricia Martinkova \cr
Institute of Computer Science of the Czech Academy of Sciences \cr
\email{martinkova@cs.cas.cz} \cr

Karel Zvara \cr
Faculty of Mathematics and Physics, Charles University \cr
}
\keyword{DIF}
