% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/likelihood_RM.R
\name{LogLikRM}
\alias{LogLikRM}
\alias{LLRM}
\alias{LogLikIRM}
\alias{LogLikPCRM}
\title{Log-Likelihood functions for the independent and partially anti-correlated race models of confidence}
\usage{
LogLikRM(data, paramDf, model = "IRM", time_scaled = FALSE,
  precision = 6, data_names = list(), ...)
}
\arguments{
\item{data}{a dataframe where each row is one trial. Containing following variables:
\itemize{
\item condition    (not necessary; convertible to integer (e.g. factor); for different levels of stimulus quality),
\item rating            (convertible to integer (e.g. factor); discrete confidence judgments),
\item rt                    (numeric; giving reaction times for decision task),
\item stimulus     (values at least convertible to c(1,2), i.e. integer or factor; stimulus category (index of accumulator with higher drift))
\item response     (values at least convertible to c(1,2); direction of decision; (index of accumulator reaching the boundary first))

}}

\item{paramDf}{a list or data frame with one row. Column names should match the names of
\link{RaceModels} parameter names (only \code{mu1} and \code{mu2} are not used in this context but
replaced by the parameter \code{v}). For different stimulus quality/mean
drift rates, names should be \code{v1}, \code{v2}, \code{v3},....
Different \code{s} parameters are possible with \code{s1}, \code{s2}, \code{s3},... with equally many steps as for drift rates. Additionally, the confidence
thresholds should be given by names with \code{thetaUpper1}, \code{thetaUpper2},..., \code{thetaLower1},... or,
for symmetric thresholds only by \code{theta1}, \code{theta2},.... (see Details for the correspondence to the data)}

\item{model}{character scalar. One of "IRM" or "PCRM". ("IRMt" and "PCRMt" will also be accepted. In that case,
time_scaled is set to TRUE.)}

\item{time_scaled}{logical. Whether the confidence measure should be scaled by 1/sqrt(rt). Default: TRUE.}

\item{precision}{numerical scalar. Precision of calculation for integration over t0.}

\item{data_names}{list. Possibility of giving alternative column names for the variables in the data. By default column names are identical to the
ones given in the data argument description.}

\item{...}{Another possibility of giving alternative variable names in data frame (in the form \code{condition = "SOA"}).}
}
\value{
Numeric scalar. The summed Log-likelihood of the data given the parameters in the respective model. If one or more row-wise probabilities is <=0,
the function returns -1e+12.
}
\description{
Computes the Log-likelihood for given data and parameters in the IRM and PCRM with or without time-scaled
confidence measure. It is a wrapped version of the respective densities \code{\link{dIRM}} and \code{\link{dPCRM}},
where one can find more information about the parameters. It restricts the rates of accumulation to be the negative
of each other, though (a common assumption in perceptual decision tasks).
The function is mainly used inside \code{\link{fitRTConf}} for race models but exported
for individual usage in other contexts.
}
\details{
Note, that the requirements on the format of the columns for the likelihood functions are much stricter, than in \code{\link{fitRTConf}}.
This is because the function is very frequently called in the optimization routines of the fitting process and the preprocessing steps are
therefore included in the other function.

\strong{rating, condition}. If integer, values should range from 1
to number of possible ratings/conditions. If factor,
the number of levels should be equal to number of possible
ratings/conditions. This should be consistent with the
parameter vector. The confidence thresholds should be named as
\code{thetaUpper1}, \code{thetaLower1},.... (or \code{theta1},... for symmetric
thresholds), with the number of ratings -1 and the mean drift rates
(and possibly the standard deviation in drift rates)
should be denoted as \code{v1}, \code{v2},...
(and \code{s1}, \code{s2},...) with the number equal to the number of conditions.
If only one condition is used \code{v} will be accepted as well as \code{v1}.

\strong{stimulus, response}. stimulus and response should always
be given in numerical format with values 1 and 2.
Stimulus determines which of two accumulators has positive drift.
The other has negative drift with the same absolute
value. Response gives the index of the accumulator that reaches the
boundary first.
}
\examples{
# 1. Generate data from an artificial participants
# Get random index for accumulator with positive
# drift (i.e. stimulus category) and
# stimulus discriminability (two steps: hard, easy)
stimulus <- sample(c(1, 2), 200, replace=TRUE)
discriminability <- sample(c(1, 2), 200, replace=TRUE)
# generate data for participant 1
data <- rPCRM(200, mu1=ifelse(stimulus==1, 1, -1)*discriminability*0.5,
              mu2=ifelse(stimulus==1, -1, 1)*discriminability*0.5,
             a=2, b=1.8, t0=0.2, st0=0, wx=0.7, wint=0.3, wrt=0)
# discretize confidence ratings (only 2 steps: unsure vs. sure)
data$rating <- as.numeric(cut(data$conf, breaks = c(0, 3, Inf), include.lowest = TRUE))
data$stimulus <- stimulus
data$discriminability <- discriminability
data <- data[data$response!=0, ] # drop not finished decision processes
data <- data[,-c(3,4)] # drop xl and conf measure (unobservable variable)
head(data)

# 2. Define some parameter set in a data.frame
paramDf <- data.frame(a=2,b=2, v1=0.5, v2=1, t0=0.1,st0=0,
                      wx=0.6, wint=0.2, wrt=0.2,
                      theta1=4)

# 3. Compute log likelihood for parameter and data
LogLikRM(data, paramDf, model="PCRMt", condition="discriminability")
# same result
LogLikRM(data, paramDf, model="PCRM", time_scaled=TRUE,condition="discriminability")
# different
LogLikRM(data, paramDf, model="PCRM", condition="discriminability")

# same parameters used for IRM model
LogLikRM(data, paramDf, model="IRMt", condition="discriminability")

}
\author{
Sebastian Hellmann.
}
