% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dtm_main.r
\name{run.dtm}
\alias{run.dtm}
\title{Run Dynamic TOPMODEL against hydrometric data and a catchment discretisation}
\usage{
run.dtm(groups, weights, rain, routing, upstream_inputs = NULL, qobs = NULL,
  qt0 = 1e-04, pe = NULL, dt = NULL, ntt = 2, ichan = 1,
  Wsurf = weights, Wover = weights, i.out = ichan[1], dqds = NULL,
  sim.start = NA, sim.end = NA, disp.par = get.disp.par(), ...)
}
\arguments{
\item{groups}{Data frame of areal group definitions along with their hydrological parameters (see Metcalfe et al., 2015)}

\item{weights}{If the discretisation has n groups, this holds the n x n flux distribution (weighting) matrix defining downslope}

\item{rain}{A time series of rainfall data in m/hr. One column per gauge if multiple gauges used.}

\item{routing}{data.frame  Channel routing table comprises a two-column data.frame or matrix. Its first column should be average flow distance to the outlet in m, the second the proportions of the catchment channel network within each distance category. Can be generated by make.routing.table}

\item{upstream_inputs}{xts A list of any upstream hydrographs in addition to hillslope runoff feeding into the river network}

\item{qobs}{Optional time series of observation data}

\item{qt0}{Initial specific discharge (m/hr)}

\item{pe}{Time series of potential evapotranspiration, at the same time step as rainfall data}

\item{dt}{Time step (hours). Defaults to the interval used by the rainfall data}

\item{ntt}{Number of inner time steps used in subsurface routing algorithm}

\item{ichan}{Integer index of the "channel" group. Defaults to 1}

\item{Wsurf}{matrix  Surface routing matrix. Defines routing of overland flow downslope between units. By default identical to subsurface routing matrix by default, but can be altered to reflect modified connectivity of certain areas with the hillslope}

\item{Wover}{matrix  Optional surface overflow routing matrix. Defines routing of overland flow from a unit that has run out of surface excess storage capacity. Identical to surface routing matrix by default. Can be altered to reflect an overflow channel for a runoff storage area, for example.}

\item{i.out}{For multi-channel systems, the index of the outlet reach}

\item{dqds}{Function to supply a custom flux-storage relationship as the kinematic wave celerity. If not supplied then exponential relationship used.}

\item{sim.start}{Optional start time for simulation in any format that can be coerced into a POSIXct instance. Defaults to start of rainfall data}

\item{sim.end}{Optional end time of simulation in any format that can be coerced into a POSIXct instance. Defaults to end of rainfall data}

\item{disp.par}{List of graphical routing parameters. A set of defaults are retrieved by calling disp.par()}

\item{...}{Any further arguments will be treated as graphical parameters as documented in get.disp.par}
}
\value{
qsim: time series of specific discharges (m/hr) at the specified time interval. can be converted to absolute discharges by multiplying by catch.area

catch.area: the catchment area in m^2, calculated from the areas in the groups table

data.in: a list comprising the parameters supplied to the call

datetime sim.start Start of simulation

sim.end  datetime   End time of simulation

fluxes: a list comprising, for each response unit the specific base flows qbf, specific upslope inputs qin, drainage fluxes quz, and any overland flow qof, all in m/hr

storages: a list comprising, for each response unit, root zone and unsaturated storage, total storage deficit and surface storages (all m)
}
\description{
Run Dynamic TOPMODEL against hydrometric data and a catchment discretisation
}
\details{
The grouping (HRU) table may be generated by the discretise method and includes each indexed channel as separate group. See Metcalfe et al. (2015) for descriptions of the parameters maintained in this table.

Evapotranspiration input can be generated using the approx.pe.ts method
}
\note{
If rain, pe or observation data differ in time period, use aggregate_xts to coerce the relevant series to the desired  time interval
}
\examples{
\dontrun{
require(dynatopmodel)
data(brompton)

# Examine the November 2012 event that flooded the village (see Metcalfe et al., 2017)
sel <- "2012-11-23 12:00::2012-12-01"
# Precalculated discretisation
disc <- brompton$disc
groups <- disc$groups
rain <- brompton$rain[sel]
# to 15 minute intervals
rain <- disaggregate_xts(rain, dt = 15/60)
# Reduce PE, seems a bit on high side and resulted in a weighting factor for the rainfall
pe <- brompton$pe[sel]/2
qobs <- brompton$qobs[sel]

# Here we apply the same parameter values to all groups.
# we could also consider a discontinuity at the depth of subsurface drains (~1m)
# or in areas more remote from the channel that do not contribute fast subsurface
# flow via field drainage
groups <- disc$groups
groups$m <- 0.0044
# Simulate impermeable clay soils
groups$td <-  33
groups$ln_t0 <- 1.15
groups$srz_max <- 0.1
qobs <- brompton$qobs[sel]
qt0 <- as.numeric(qobs[1,])
# initial root zone storage - almost full due to previous event
groups$srz0 <- 0.98
# Quite slow channel flow, which might be expected with the shallow and reedy
# low bedslope reaches with very rough banks comprising the major channel
groups$vchan <- 400
groups$vof <- 50
# Rain is supplied at hourly intervals: convert to 15 minutes
rain <- disaggregate_xts(rain, dt = 15/60)
weights <- disc$weights
# Output goes to a new window
graphics.off()
x11()

# Initial discharge from the observations
qt0 <- as.numeric(qobs[1,])

# Run the model across the November 2012 storm event
# using a 15 minute interval
run <- run.dtm(groups=groups,
               weights=weights,
               rain=rain,
               pe=pe,
               qobs=qobs,
               qt0=qt0,
               routing=brompton$routing,
               graphics.show=TRUE, max.q=2.4)
}

}
\references{
Metcalfe, P., Beven, K., & Freer, J. (2015). Dynamic TOPMODEL: a new implementation in R and its sensitivity to time and space steps. Environmental Modelling & Software, 72, 155-172.
}
\seealso{
aggregate_xts

discretise
}
\author{
Peter Metcalfe
}
