#' Convert calls into quantile functions
#'
#' Calls come from ... in `simulateTrialData`. For individual level data,
#' empirical quantile will be generated using `simdata::quantile_functions_from_data`.
#' For calls specifying common distribution families and their parameters in plain text,
#' quantile functions will be generated by quantile functions in `stats`, e.g.,
#' qnorm, qunif, etc. For calls of user-customized quantile functions, it returns
#' those directly.
#'
#' @param ... calls describing the marginal distributions of endpoints. See `...`
#' in `simulateTrialData`.
#' @param var_prefix prefix to name variables that users do not give to those.
#'
#' @importFrom stringr str_glue
#' @importFrom dplyr %>%
#' @importFrom simdata simdesign_norta
#' @importFrom simdata quantile_functions_from_data
#' @importFrom ggplot2 ggplot aes geom_point geom_abline labs theme_minimal annotate
#' @importFrom gridExtra grid.arrange
#'
#' @export
#'
#' @return a list of quantile functions.

configureTrial <- function(..., var_prefix){

  calls <- as.list(substitute(list(...)))[-1L]
  attr(calls, 'env') <- new.env(parent = parent.frame())

  distr <- lapply(calls, parseCall, env = attr(calls, 'env'))

  distr_funcs <- list()
  for(d in distr){
    if(attr(d, 'type') %in% 'data'){
      for(i in seq_along(d)){
        func <- d[[i]]
        attributes(func) <-
          list(
            type = 'data',
            var_name = attr(d, 'var_names')[i]
          )
        distr_funcs <- c(distr_funcs, func)
      }
    }else{
      distr_funcs <- c(distr_funcs, d)
    }
  }

  var_idx <- 0
  var_names <- NULL
  for(func in distr_funcs){
    var_name <- attributes(func)$var_name
    if(is.null(var_name)){
      var_idx <- var_idx + 1
      var_names <- c(var_names, str_glue('{var_prefix}_{attr(func, "type")}_{var_idx}'))
    }else{
      var_names <- c(var_names, var_name)
    }
  }

  attr(distr_funcs, 'var_names') <- var_names

  distr_funcs

}
