#' Compute subject-specific or overall transition probabilities
#'
#' This function is a wrapper for \code{mstate::probtrans}. Its purpose is to 
#' preclude the computation of (co-)variances of the transition probability 
#' estimator when the fitted Cox model is empirical Bayes. 
#' This help page is an adaptation of the \code{mstate::probtrans} help page. 
#'
#' @param object An msfit object containing estimated cumulative hazards
#' for each of the transitions in the multi-state model and,
#' if standard errors are requested, (co)variances of these cumulative hazards
#' for each pair of transitions.
#' @param predt A positive number indicating the prediction time. 
#' This is either the time at which the prediction is made (if direction= "forward") 
#' or the time for which the prediction is to be made (if direction="fixedhorizon").
#' @param direction One of "forward" (default) or "fixedhorizon", indicating whether
#' prediction is forward or for a fixed horizon.
#' @param method A character string specifying the type of variances to be computed
#'  (so only needed if either variance or covariance is TRUE). Possible values are 
#'  "aalen" or "greenwood".
#' @param variance Logical value indicating whether standard errors are to be
#' calculated (default is TRUE).
#' @param covariance Logical value indicating whether covariances of transition
#'  probabilities for different states are to be calculated (default is FALSE).
#' @param ... other arguments.
#' @details \code{probtrans_mstate} computes estimates of transition probabilities
#' for an object generated by \code{msfit_generic}. It calls the method 
#' \code{probtrans_mstate.coxrfx}, if the msfit object was generated
#' by msfit_generic.coxrfx, or the method probtrans_mstate.default otherwise.
#' Both methods are identical to the function \code{mstate::probtrans}. The 
#' only reserve is that \code{probtrans_mstate.coxrfx} does not allow the computation of
#' the (co-)variances of the transition probability estimator. In fact,
#' this computation relies on asymptotic results for the \emph{fixed} effects Cox model
#' (see de Wreede et al, 2010, section 2.3.2), and \code{msfit_generic.coxrfx} produces
#' estimates of cumulative hazards under a random effects/empirical Bayes Cox model. 
#' 
#' \code{probtrans_mstate} should only be used for Markov models, as it relies
#' on product limit calculations.
#' @return An object of class \code{probtrans}. See the `value' section in the
#' the help page of \code{\link{probtrans}} for details.
#' @author Rui Costa, adapting the work of L. de Wreede,
#' M. Fiocco and H. Putter in the
#' \code{\link{mstate}} package.
#' @references 
#' de Wreede LC, Fiocco M, and Putter H (2010). The mstate package for 
#' estimation and prediction in non- and semi-parametric multi-state and 
#' competing risks models. \emph{Computer Methods and Programs in Biomedicine} 
#' \strong{99}, 261–274.
#' @seealso \code{\link[mstate:probtrans]{mstate::probtrans}}; \code{\link[mstate:msfit]{mstate::msfit}}; \code{\link{msfit_generic}}. 
#' @export
probtrans_mstate <- function(object, ...){
  UseMethod("probtrans_mstate")
}

#' @rdname probtrans_mstate 
#' @export
probtrans_mstate.default<-function(object, predt,
                                  direction=c("forward","fixedhorizon"),
                                  method=c("aalen","greenwood"), 
                                  variance=TRUE, covariance=FALSE,...){
  return(probtrans(object,predt,direction,method,variance,covariance))
}

#' @rdname probtrans_mstate
#' @export
probtrans_mstate.coxrfx<-function(object, predt, 
                                   direction=c("forward",
                                               "fixedhorizon"),...){
  return(probtrans(object,predt,direction,method="aalen",variance=FALSE
                   ,covariance=FALSE))
}
