% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/epim.R
\name{epim}
\alias{epim}
\title{Fit a Bayesian epidemiological model with epidemia}
\usage{
epim(
  rt,
  inf,
  obs,
  data,
  algorithm = c("sampling", "meanfield", "fullrank"),
  group_subset = NULL,
  prior_PD = FALSE,
  ...
)
}
\arguments{
\item{rt}{An object of class \code{epirt}. This defines
the model for the time-varying reproduction rates. See \code{\link{epirt}} for more details.}

\item{inf}{An object of class \code{epiinf}. This defines
the model for latent infections. See \code{\link{epiinf}} for more details.}

\item{obs}{Either an \code{epiobs} object, or a list of such objects. Each
element in the list defines a model for the specified observation vector. See \code{\link{epiobs}} for more details.}

\item{data}{A dataframe with all data required for fitting the model. This includes all observation variables and covariates specified in the models for the reproduction number and ascertainment rates.}

\item{algorithm}{One of \code{"sampling"}, \code{"meanfield"} or
\code{"fullrank"}. This specifies which \pkg{rstan} function to use for
fitting the model. For \code{"sampling"} this is \code{\link[rstan]{sampling}}, otherwise
this is \code{\link[rstan]{vb}}.}

\item{group_subset}{If specified, a character vector naming a subset of regions to include in the model.}

\item{prior_PD}{Same as in \code{\link[rstanarm]{stan_glm}}. If \code{TRUE},
samples all parameters from the joint prior distribution. This is useful for
prior predictive checks. Defaults to \code{FALSE}.}

\item{...}{Additional arguments to pass to the \pkg{rstan} function used to fit the model.}
}
\value{
An object of class \code{epimodel}.
}
\description{
\code{\link{epim}} is the only model fitting function in \pkg{epidemia}.
It takes a model description, a dataframe, and additional
arguments relating to the fitting algorithm, and translates this
to data that is then passed to a precompiled \pkg{Stan} program which is used to fit the model.
This allows model fitting to begin immediately as opposed to requiring compilation
each time \code{epim} is called.
}
\details{
This is similar to the workflow for fitting Bayesian regression models with \pkg{rstanarm}.
A key difference, however, is that the models fit by \pkg{epidemia} are much more complex,
and are therefore inherently more difficult to specify. \pkg{epidemia} aims to simplify this
process by modularizing the model definition into three distinct parts: transmission, infections and observations.
These components of the model are defined with the functions \code{\link{epirt}}, \code{\link{epiinf}} and \code{\link{epiobs}}
respectively.

\code{\link{epim}} has arguments
\code{rt}, \code{inf} and \code{obs} which expect a description of the
transmission model, infection model and observational models respectively.
Together, these fully define the joint distribution of data and parameters.
Each of these model components are described in terms of variables that are expected to live in a single dataframe, \code{data}.
This dataframe must be compatible with the model components, in the sense that it holds all variables defined in these models.

In addition to taking a model description and a dataframe, \code{\link{epim}} has various
additional arguments which specify how the model should be fit. If \code{algorithm = "sampling"}
then the model will be fit using \pkg{Stan}’s adaptive Hamiltonian Monte Carlo sampler.
This is done internally by calling \code{\link[rstan]{sampling}}. If
\code{algorithm = "meanfield"} or \code{algorithm = "fullrank"}, then
\pkg{Stan}’s variational Bayes algorithms are used instead, by calling \code{\link[rstan]{vb}}.
Any unnamed arguments in the call to \code{\link{epim}} are passed directly on to the \pkg{rstan} sampling function.
\code{\link{epim}} returns a fitted model object of class \code{epimodel}, which contains posterior samples from the model along with other useful objects.

In general, the adaptive Hamiltonian Monte Carlo sampler should be used for final inference.
Nonetheless, fitting these models using HMC is often computationally demanding, and variational Bayes can often be fruitful for quickly iterating models.
}
\examples{
\donttest{
library(EpiEstim)
data("Flu1918")

date <- as.Date("1918-01-01") + seq(0, along.with = c(NA, Flu1918$incidence))
data <- data.frame(
 city = "Baltimore",
 cases = c(NA, Flu1918$incidence),
 date = date,
 week = lubridate::week(date)
)

rt <- epirt(
 formula = R(city, date) ~ rw(time = week, prior_scale = 0.1),
 prior_intercept = rstanarm::normal(log(2), 0.2),
 link = 'log'
)

obs <-  epiobs(
 formula = cases ~ 1,
 prior_intercept = rstanarm::normal(location=1, scale=0.01),
 link = "identity",
 i2o = rep(.25,4)
)

args <- list(
 rt = rt,
 inf = epiinf(gen = Flu1918$si_distr),
 obs = obs,
 data = data,
 algorithm = "fullrank",
 iter = 1e4,
 seed = 12345
)

fm <- do.call(epim, args)

}
}
