% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/EigenTensorDecomposition.R
\name{EigenTensorDecomposition}
\alias{EigenTensorDecomposition}
\alias{EigenTensorDecomposition.list}
\alias{EigenTensorDecomposition.default}
\title{Eigentensor Decomposition}
\usage{
EigenTensorDecomposition(matrices, return.projection = TRUE, ...)

\method{EigenTensorDecomposition}{list}(matrices,
  return.projection = TRUE, ...)

\method{EigenTensorDecomposition}{default}(matrices,
  return.projection = TRUE, ...)
}
\arguments{
\item{matrices}{k x k x m array of m covariance matrices with k traits;}

\item{return.projection}{Should we project covariance matrices into estimated eigentensors? Defaults to TRUE}

\item{...}{aditional arguments for methods}
}
\value{
List with the following components:

mean mean covariance matrices used to center the sample (obtained from \code{\link{MeanMatrix}})

mean.sqrt square root of mean matrix (saved for use in other functions, 
such as \code{\link{ProjectMatrix}} and \code{\link{RevertMatrix}})

values vector of ordered eigenvalues associated with eigentensors;

matrices array of eigentensor in matrix form;

projection matrix of unstandardized projected covariance matrices over eigentensors.
}
\description{
This function performs eigentensor decomposition on a set of covariance matrices.
}
\details{
The number of estimated eigentensors is the minimum between the number of data points (m) and 
the number of independent variables (k(k + 1)/2) minus one, in a similar manner to the usual 
principal component analysis.
}
\examples{
\dontrun{
data(dentus)

dentus.vcv <- daply (dentus, .(species), function(x) cov(x[,-5]))

dentus.vcv <- aperm(dentus.vcv, c(2, 3, 1))

dentus.etd <- EigenTensorDecomposition(dentus.vcv, TRUE)

# Plot some results
par(mfrow = c(1, 2))
plot(dentus.etd $ values, pch = 20, type = 'b', ylab = 'Eigenvalue')
plot(dentus.etd $ projection [, 1:2], pch = 20, 
     xlab = 'Eigentensor 1', ylab = 'Eigentensor 2')
text(dentus.etd $ projection [, 1:2],
     labels = rownames (dentus.etd $ projection), pos = 2)

# we can also deal with posterior samples of covariance matrices using plyr

dentus.models <- dlply(dentus, .(species), 
                       lm, formula = cbind(humerus, ulna, femur, tibia) ~ 1)

dentus.matrices <- llply(dentus.models, BayesianCalculateMatrix, samples = 100)

dentus.post.vcv <- laply(dentus.matrices, function (L) L $ Ps)

dentus.post.vcv <- aperm(dentus.post.vcv, c(3, 4, 1, 2))

# this will perform one eigentensor decomposition for each set of posterior samples
dentus.post.etd <- alply(dentus.post.vcv, 4, EigenTensorDecomposition)

# which would allow us to observe the posterior 
# distribution of associated eigenvalues, for instance
dentus.post.eval <- laply (dentus.post.etd, function (L) L $ values)

boxplot(dentus.post.eval, xlab = 'Index', ylab = 'Value', 
        main = 'Posterior Eigenvalue Distribution')
}
}
\references{
Basser P. J., Pajevic S. 2007. Spectral decomposition of a 4th-order 
covariance tensor: Applications to diffusion tensor MRI. Signal Processing. 87:220-236.

Hine E., Chenoweth S. F., Rundle H. D., Blows M. W. 2009. Characterizing 
the evolution of genetic variance using genetic covariance tensors. Philosophical 
transactions of the Royal Society of London. Series B, Biological sciences. 364:1567-78.
}
\seealso{
\code{\link{ProjectMatrix}}, \code{\link{RevertMatrix}}
}
\author{
Guilherme Garcia, Diogo Melo
}
