% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geometry.R
\name{tricontour}
\alias{tricontour}
\alias{tricontour.inla.mesh}
\alias{tricontour.matrix}
\alias{tricontour.list}
\alias{tricontourmap}
\alias{tricontourmap.inla.mesh}
\alias{tricontourmap.matrix}
\alias{tricontourmap.list}
\title{Calculate contour curves on a triangulation}
\usage{
tricontour(
  x,
  z,
  nlevels = 10,
  levels = pretty(range(z, na.rm = TRUE), nlevels),
  ...
)

\method{tricontour}{inla.mesh}(
  x,
  z,
  nlevels = 10,
  levels = pretty(range(z, na.rm = TRUE), nlevels),
  ...
)

\method{tricontour}{matrix}(
  x,
  z,
  nlevels = 10,
  levels = pretty(range(z, na.rm = TRUE), nlevels),
  loc,
  ...
)

\method{tricontour}{list}(
  x,
  z,
  nlevels = 10,
  levels = pretty(range(z, na.rm = TRUE), nlevels),
  loc,
  type = c("+", "-"),
  tol = 1e-07,
  ...
)

tricontourmap(
  x,
  z,
  nlevels = 10,
  levels = pretty(range(z, na.rm = TRUE), nlevels),
  ...
)

\method{tricontourmap}{inla.mesh}(
  x,
  z,
  nlevels = 10,
  levels = pretty(range(z, na.rm = TRUE), nlevels),
  ...
)

\method{tricontourmap}{matrix}(
  x,
  z,
  nlevels = 10,
  levels = pretty(range(z, na.rm = TRUE), nlevels),
  loc,
  ...
)

\method{tricontourmap}{list}(
  x,
  z,
  nlevels = 10,
  levels = pretty(range(z, na.rm = TRUE), nlevels),
  loc,
  type = c("+", "-"),
  tol = 1e-07,
  output = c("sp", "inla.mesh.segment"),
  ...
)
}
\arguments{
\item{x}{An object generated by a call to \code{inla.mesh.2d} or
\code{inla.mesh.create}, a triangle-vertex index matrix, or a list
of triangulation information, \code{list(loc, graph=list(tv))}.}

\item{z}{A vector containing the values to be contoured
(\code{NA}s are allowed).}

\item{nlevels}{Number of contour levels desired, if and only if
\code{levels} is not supplied.}

\item{levels}{Numeric vector of levels at which to calculate contour lines.}

\item{...}{Additional arguments passed to the other methods.}

\item{loc}{coordinate matrix, to be supplied when \code{x} is given as a
triangle-vertex index matrix only.}

\item{type}{\code{"+"} or \code{"-"}, indicating positive or negative
association. For \code{+}, the generated contours enclose regions
where \eqn{u_1 \leq z < u_2}, for \code{-} the regions fulfil \eqn{u_1
  < z \leq u_2}.}

\item{tol}{tolerance for determining if the value at a vertex lies on a level.}

\item{output}{The format of the generated output.  Implemented options
are \code{"sp"} (default) and \code{"inla.mesh.segment"} (requires the
                                                          INLA package).}
}
\value{
For \code{tricontour}, a list some of the same fields that
  \code{inla.mesh.segment} objects have:
  \item{loc}{A coordinate matrix}
  \item{idx}{Contour segment indices, as a 2-column matrix, each row
    indexing a single segment}
  \item{grp}{A vector of group labels.  Each segment has a label, in
    \code{1,...,nlevels*2+1}, where even labels indicate interior
    on-level contour segments, and odd labels indicate boundary segments
    between levels.}
  For \code{tricontourmap}, a list:
  \item{contour}{A list of \code{sp} or \code{inla.mesh.segment} objects
    defining countour curves (level sets)}
  \item{map}{A list of \code{sp} or \code{inla.mesh.segment} objects
    enclosing regions between level sets}
}
\description{
Calculates contour curves and/or regions between them,
for functions defined on a triangulation
}
\examples{
\donttest{
  if (require.nowarnings("INLA")) {
    ## Generate mesh and SPDE model
    n.lattice <- 20 #increase for more interesting, but slower, examples
    x <- seq(from = 0, to = 10, length.out = n.lattice)
    lattice <- inla.mesh.lattice(x = x, y = x)
    mesh <- inla.mesh.create(lattice = lattice, extend = FALSE, refine = FALSE)
    spde <- inla.spde2.matern(mesh, alpha = 2)
    
    ## Generate an artificial sample
    sigma2.e <- 0.01
    n.obs <-1000
    obs.loc <- cbind(runif(n.obs) * diff(range(x)) + min(x),
                     runif(n.obs) * diff(range(x)) + min(x))
    Q <- inla.spde2.precision(spde, theta = c(log(sqrt(0.5)), log(sqrt(1))))
    x <- inla.qsample(Q = Q)
    A <- inla.spde.make.A(mesh = mesh, loc = obs.loc)
    Y <- as.vector(A \%*\% x + rnorm(n.obs) * sqrt(sigma2.e))
    
    ## Calculate posterior
    Q.post <- (Q + (t(A) \%*\% A)/sigma2.e)
    mu.post <- as.vector(solve(Q.post,(t(A) \%*\% Y)/sigma2.e))
    
    ## Calculate continuous contours
    tric <- tricontour(mesh, z = mu.post,
                       levels = as.vector(quantile(x, c(0.25, 0.75))))
   
    ## Discrete domain contours
    map <- contourmap(n.levels = 2, mu = mu.post, Q = Q.post,
                      alpha=0.1, compute = list(F = FALSE), max.threads=1)

    ## Calculate continuous contour map
    setsc <- tricontourmap(mesh, z = mu.post,
                           levels = as.vector(quantile(x, c(0.25, 0.75))))

    ## Plot the results
    reo <- mesh$idx$lattice
    idx.setsc <- setdiff(names(setsc$map), "-1")
    cols2 <- contourmap.colors(map, col=heat.colors(100, 0.5),
                               credible.col = grey(0.5, 0))
    names(cols2) <- as.character(-1:2)

    par(mfrow = c(1,2))
    image(matrix(mu.post[reo], n.lattice, n.lattice),
                 main = "mean", axes = FALSE)
    plot(setsc$map[idx.setsc], col = cols2[idx.setsc])
    par(mfrow = c(1,1))
  }
}
}
\author{
Finn Lindgren \email{finn.lindgren@gmail.com}
}
