% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pipeline-glassbox.R
\name{glassbox}
\alias{glassbox}
\title{The opinionated "glass box" \code{eyeris} pipeline}
\usage{
glassbox(
  file,
  confirm = FALSE,
  detrend_data = FALSE,
  num_previews = 3,
  preview_duration = 5,
  preview_window = NULL,
  skip_detransient = FALSE,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{file}{An SR Research EyeLink \code{.asc} file generated by the official
EyeLink \code{edf2asc} command.}

\item{confirm}{A flag to indicate whether to run the \code{glassbox} pipeline
autonomously all the way through (set to \code{FALSE} by default), or to
interactively provide a visualization after each pipeline step, where you
must also indicate "(y)es" or "(n)o" to either proceed or cancel the
current \code{glassbox} pipeline operation (set to \code{TRUE}).}

\item{detrend_data}{A flag to indicate whether to run the \code{detrend} step (set
to \code{FALSE} by default). Detrending your pupil timeseries can have unintended
consequences; we thus recommend that users understand the implications of
detrending -- in addition to whether detrending is appropriate for the
research design and question(s) -- before using this function.}

\item{num_previews}{Number of random example "epochs" to generate for
previewing the effect of each preprocessing step on the pupil timeseries.}

\item{preview_duration}{Time in seconds of each randomly selected preview.}

\item{preview_window}{The start and stop raw timestamps used to subset the
preprocessed data from each step of the \code{eyeris} workflow for visualization.
Defaults to NULL, meaning random epochs as defined by \code{num_previews} and
\code{preview_duration} will be plotted. To override the random epochs, set
\code{preview_window} here to a vector with relative start and stop times
(e.g., \code{c(5000, 6000)} to indicate the raw data from 5-6 seconds on data that
were recorded at 1000 Hz). Note, the start/stop time values indicated here
relate to the raw index position of each pupil sample from 1 to n (which
will need to be specified manually by the user depending on the sampling rate
of the recording; i.e., 5000-6000 for the epoch positioned from 5-6 seconds
after the start of the timeseries, sampled at 1000 Hz).}

\item{skip_detransient}{A flag to indicate whether to skip the \code{detransient}
step (set to \code{FALSE} by default). In most cases, this should remain \code{FALSE}.
For a more detailed description about likely edge cases that would prompt
you to set this to \code{TRUE}, see the docs for \code{\link[=detransient]{detransient()}}.}

\item{verbose}{A flag to indicate whether to print detailed logging messages.
Defaults to \code{TRUE}. Set to \code{False} to suppress messages about the current
processing step and run silently.}

\item{...}{Additional arguments to override the default, prescribed settings.}
}
\value{
Preprocessed pupil data contained within an object of class \code{eyeris}.
}
\description{
This \code{glassbox} function (in contrast to a "black box" function where you run
it and get a result but have no (or little) idea as to how you got from input
to output) has a few primary benefits over calling each exported function
from \code{eyeris} separately.
}
\details{
First, this \code{glassbox} function provides a highly opinionated prescription of
steps and starting parameters we believe any pupillometry researcher should
use as their defaults when preprocessing pupillometry data.

Second, and not mutually exclusive from the first point, using this function
should ideally reduce the probability of accidental mishaps when
"reimplementing" the steps from the preprocessing pipeline both within and
across projects. We hope to streamline the process in such a way that you
could collect a pupillometry dataset and within a few minutes assess the
quality of those data while simultaneously running a full preprocessing
pipeline in 1-ish line of code!

Third, \code{glassbox} provides an "interactive" framework where you can evaluate
the consequences of the parameters within each step on your data in real
time, facilitating a fairly easy-to-use workflow for parameter optimization
on your particular dataset. This process essentially takes each of the
opinionated steps and provides a pre-/post-plot of the timeseries data for
each step so you can adjust parameters and re-run the pipeline until you are
satisfied with the choices of your paramters and their consequences on your
pupil timeseries data.
}
\examples{
demo_data <- system.file("extdata", "memory.asc", package = "eyeris")

# (1) examples using the default prescribed parameters and pipeline recipe

## (a) run an automated pipeline with no real-time inspection of parameters
output <- eyeris::glassbox(demo_data)

plot(
  output,
  steps = c(1, 5),
  preview_window = c(0, nrow(output$timeseries$block_1)),
  seed = 0
)

## (b) run a interactive workflow (with confirmation prompts after each step)
\donttest{
output <- eyeris::glassbox(demo_data, confirm = TRUE, seed = 0)
}

# (2) examples overriding the default parameters
output <- eyeris::glassbox(
  demo_data,
  confirm = FALSE, # TRUE if you want to visualize each step in real-time
  deblink = list(extend = 40),
  lpfilt = list(plot_freqz = FALSE)
)

plot(output, seed = 0)

}
