% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/structure_plot.R
\name{structure_plot}
\alias{structure_plot}
\alias{structure_plot_default_embed_method}
\alias{plot.poisson_nmf_fit}
\alias{plot.multinom_topic_model_fit}
\alias{structure_plot_ggplot_call}
\title{Structure Plot}
\usage{
structure_plot(
  fit,
  topics,
  grouping,
  loadings_order = "embed",
  n = 2000,
  colors,
  gap = 1,
  embed_method = structure_plot_default_embed_method,
  ggplot_call = structure_plot_ggplot_call,
  ...
)

structure_plot_default_embed_method(fit, ...)

\method{plot}{poisson_nmf_fit}(x, ...)

\method{plot}{multinom_topic_model_fit}(x, ...)

structure_plot_ggplot_call(dat, colors, ticks = NULL, font.size = 9)
}
\arguments{
\item{fit}{An object of class \dQuote{poisson_nmf_fit} or
\dQuote{multinom_topic_model_fit}, or an n x k matrix of topic
proportions, where k is the number of topics. (The elements in each
row of this matrix should sum to 1.) If a Poisson NMF fit is
provided as input, the corresponding multinomial topic model fit is
automatically recovered using \code{\link{poisson2multinom}}.}

\item{topics}{Top-to-bottom ordering of the topics in the Structure
plot; \code{topics[1]} is shown on the top, \code{topics[2]} is
shown next, and so on. If the ordering of the topics is not
specified, the topics are automatically ordered so that the topics
with the greatest total \dQuote{mass} are at shown at the bottom of
the plot. The topics may be specified by number or by name. Note
that not all of the topics need to be included, so one may also use
this argument to plot a subset of the topics.}

\item{grouping}{Optional categorical variable (a factor) with one
entry for each row of the loadings matrix \code{fit$L} defining a
grouping of the samples (rows). The samples (rows) are arranged
along the horizontal axis according to this grouping, then within
each group according to \code{loadings_order}. If
\code{grouping} is not a factor, an attempt is made to convert it
to a factor using \code{as.factor}. Note that if
\code{loadings_order} is specified manually, \code{grouping} should
be the groups for the rows of \code{fit$L} \emph{before} reordering.}

\item{loadings_order}{Ordering of the rows of the loadings matrix
\code{fit$L} along the horizontal axis the Structure plot (after
they have been grouped). If \code{loadings_order = "embed"}, the
ordering is generated automatically from a 1-d embedding,
separately for each group. The rows may be specified by number or
by name. Note that \code{loadings_order} may include all the rows
of \code{fit$L}, or a subset.}

\item{n}{The maximum number of samples (rows of the loadings matrix
\code{fit$L}) to include in the plot. Typically there is little to
no benefit in including large number of samples in the Structure
plot due to screen resolution limits. Ignored if
\code{loadings_order} is provided.}

\item{colors}{Colors used to draw topics in Structure plot.}

\item{gap}{The horizontal spacing between groups. Ignored if
\code{grouping} is not provided.}

\item{embed_method}{The function used to compute an 1-d embedding
from a loadings matrix \code{fit$L}; only used if
\code{loadings_order = "embed"}. The function must accept the
multinomial topic model fit as its first input (\dQuote{fit}) and
additional arguments may be passed (\dots). The output should be a
named numeric vector with one entry per row of \code{fit$L}, and
the names of the entries should be the same as the row names of
\code{fit$L}.}

\item{ggplot_call}{The function used to create the plot. Replace
\code{structure_plot_ggplot_call} with your own function to
customize the appearance of the plot.}

\item{\dots}{Additional arguments passed to \code{structure_plot}
(for the \code{plot} method) or \code{embed_method} (for
function \code{structure_plot}).}

\item{x}{An object of class \dQuote{poisson_nmf_fit} or
\dQuote{multinom_topic_model_fit}. If a Poisson NMF fit is provided
as input, the corresponding multinomial topic model fit is
automatically recovered using \code{\link{poisson2multinom}}.}

\item{dat}{A data frame passed as input to
\code{\link[ggplot2]{ggplot}}, containing, at a minimum, columns
\dQuote{sample}, \dQuote{topic} and \dQuote{prop}: the
\dQuote{sample} column contains the positions of the samples (rows
of the L matrix) along the horizontal axis; the \dQuote{topic}
column is a topic (a column of L); and the \dQuote{prop} column is
the topic proportion for the respective sample.}

\item{ticks}{The placement of the group labels along the horizontal
axis, and their names. For data that are not grouped, use
\code{ticks = NULL}.}

\item{font.size}{Font size used in plot.}
}
\value{
A \code{ggplot} object.
}
\description{
Create a \dQuote{Structure plot} from a multinomial
  topic model fit or other model with \dQuote{loadings} or
  \dQuote{weights}. The Structure plot represents the estimated topic
  proportions of each sample in a stacked bar chart, with bars of
  different colors representing different topics. Consequently,
  samples that have similar topic proportions have similar amounts of
  each color.
}
\details{
The name \dQuote{Structure plot} comes from its widespread
use in population genetics to visualize the results of the
Structure software (Rosenberg \emph{et al}, 2002).

For most uses of the Structure plot in population genetics, there
is usually some grouping of the samples (e.g., assignment to
pre-defined populations) that guides arrangement of the samples
along the horizontal axis in the bar chart. In other applications,
such as analysis of gene expression data, a pre-defined grouping
may not always be available. Therefore, a \dQuote{smart}
arrangement of the samples is, by default, generated automatically
by performing a 1-d embedding of the samples.
}
\examples{
\donttest{
set.seed(1)
data(pbmc_facs)

# Get the multinomial topic model fitted to the
# PBMC data.
fit <- pbmc_facs$fit

# Create a Structure plot without labels. The samples (rows of L) are
# automatically arranged along the x-axis using t-SNE to highlight the
# structure in the data.
p1a <- structure_plot(fit)

# The first argument to structure_plot may also be an "L" matrix.
# This call to structure_plot should produce the exact same plot as
# the previous call.
set.seed(1)
p1b <- structure_plot(fit$L)

# There is no requirement than the rows of L sum up to 1. To
# illustrate, in this next example we have removed topic 5 from the a
# structure plot.
p2a <- structure_plot(fit$L[,-5])

# This is perhaps a more elegant way to remove topic 5 from the
# structure plot:
p2b <- structure_plot(fit,topics = c(1:4,6))

# Create a Structure plot with the FACS cell-type labels. Within each
# group (cell-type), the cells (rows of L) are automatically arranged
# using t-SNE.
subpop <- pbmc_facs$samples$subpop
p3 <- structure_plot(fit,grouping = subpop)

# Next, we apply some customizations to improve the plot: (1) use the
# "topics" argument to specify the order in which the topic
# proportions are stacked on top of each other; (2) use the "gap"
# argrument to increase the whitespace between the groups; (3) use "n"
# to decrease the number of rows of L included in the Structure plot;
# and (4) use "colors" to change the colors used to draw the topic
# proportions.
topic_colors <- c("skyblue","forestgreen","darkmagenta",
                  "dodgerblue","gold","darkorange")
p4 <- structure_plot(fit,grouping = pbmc_facs$samples$subpop,gap = 20,
                     n = 1500,topics = c(5,6,1,4,2,3),colors = topic_colors)

# In this example, we use UMAP instead of t-SNE to arrange the
# cells in the Structure plot. Note that this can be accomplished in
# a different way by overriding the default setting of
# "embed_method".
y <- drop(umap_from_topics(fit,dims = 1))
p5 <- structure_plot(fit,loadings_order = order(y),grouping = subpop,
                     gap = 40,colors = topic_colors)

# We can also use PCA to arrange the cells.
y <- drop(pca_from_topics(fit,dims = 1))
p6 <- structure_plot(fit,loadings_order = order(y),grouping = subpop,
                     gap = 40,colors = topic_colors)

# In this final example, we plot a random subset of 400 cells, and
# arrange the cells randomly along the horizontal axis of the
# Structure plot.
p7 <- structure_plot(fit,loadings_order = sample(3744,400),gap = 10,
                     grouping = subpop,colors = topic_colors)
}

}
\references{
Dey, K. K., Hsiao, C. J. and Stephens, M. (2017). Visualizing the
structure of RNA-seq expression data using grade of membership
models. \emph{PLoS Genetics} \bold{13}, e1006599.

Rosenberg, N. A., Pritchard, J. K., Weber, J. L., Cann, H. M.,
Kidd, K. K., Zhivotovsky, L. A. and Feldman, M. W. (2002). Genetic
structure of human populations. \emph{Science} \bold{298},
2381–2385.
}
