\name{fmatch}
\alias{fmatch}
\alias{fastmatch}
\title{
Fast match() replacement
}
\description{
\code{fmatch} is a faster version of the built-in \code{\link{match}}()
function. It is slightly faster that the built-in version because it
uses more specialized code, but in addition it retains the hash table
with teh table obejct such that it can be re-used, dramatically reducing
the look-up time especially for large tables.

Although \code{fmatch} can be used separately, in general it is also
safe to use:\code{match <- fmatch} since it is a drop-in replacement.
}
\usage{
fmatch(x, table, nomatch = NA_integer_, incomparables = NULL)
}
\arguments{
  \item{x}{values to be matched}
  \item{table}{values to be matched against}
  \item{nomatch}{the value to be returned in the case when no match is
    found. It is coerced to \code{integer}.}
  \item{incomparables}{a vector of values that cannot be matched. Any
    value other than \code{NULL} will result in a fall-back to
    \code{match} without any speed gains.}
}
\details{
  See \code{\link{match}} for the purpose and details of the
  \code{match} function. \code{fmatch} is a drop-in replacement for
  the \code{match} function with the focus on
  performance. \code{incomparables} are not supported by \code{fmatch}
  and will be passed down to \code{match}.

  The first match against a table results in a hash table to be computed
  from the table. This table is then attached as the `.match.hash`
  attribute of the table so that it can be re-used on subsequent calls
  to \code{fmatch} with the same table.

  The hashing algorithm used is the same as the \code{match} function in
  R, but it is re-implemented in a slight different way to improve its
  performance at the cost of supporting only a subset of types (integer,
  real and character). For any other types \code{fmatch} falls back to
  \code{match} (with a warning).
}
\value{
  A vector of the same length as \code{x} - see \code{\link{match}} for
  details.
}
%\references{
%}
%\author{
%%  ~~who you are~~
%}
\note{
\code{fmatch} modifies the \code{table} by attaching an attribute to
  it. It is expected that the values will not change unless that
  attribute is dropped. Under normal circumstances this should not have
  any effect from user's point of view, but there is a theoretical
  chance of the cache being out of sync with the table in case the table
  is modified directly (e.g. by some C code) without removing
  attributes.

  Also \code{fmatch} does not convert to a common encoding so strings
  with different representation in two encodings don't match.
}

%% ~Make other sections like Warning with \section{Warning }{....} ~
\seealso{
\code{\link{match}}
}
\examples{
# some random speed comparison examples:
# first use integer matching
x = as.integer(rnorm(1e6) * 1000000)
s = 1:100
# the first call to fmatch is comparable to match
system.time(fmatch(s,x))
# but the subsequent calls take no time!
system.time(fmatch(s,x))
system.time(fmatch(-50:50,x))
system.time(fmatch(-5000:5000,x))
# here is the speed of match for comparison
system.time(base::match(s, x))
# the results should be identical
identical(base::match(s, x), fmatch(s, x))

# next, match a factor against the table
# this will require both x and the factor
# to be cast to strings
s=factor(c("1","1","2","foo","3",NA))
# because the casting will have to allocate a string
# cache in R, we run a dummy conversion to take
# that out of the equation
dummy = as.character(x)
# now we can run the speed tests
system.time(fmatch(s, x))
system.time(fmatch(s, x))
# the cache is still valid for string matches as well
system.time(fmatch(c("foo","bar","1","2"),x))
# now back to match
system.time(base::match(s, x))
identical(base::match(s, x), fmatch(s, x))

# finally, some reals to match
y = rnorm(1e6)
s = c(y[sample(length(y), 100)], 123.567, NA, NaN)
system.time(fmatch(s, y))
system.time(fmatch(s, y))
system.time(fmatch(s, y))
system.time(base::match(s, y))
identical(base::match(s, y), fmatch(s, y))
}
\keyword{manip}
