% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fic.coxph.R
\name{fic.coxph}
\alias{fic.coxph}
\title{Focused information criteria for Cox proportional hazard regression models}
\usage{
\method{fic}{coxph}(wide, inds, inds0 = NULL, gamma0 = 0, focus, X = NULL,
  t = NULL, sub = "auto", tidy = TRUE, ...)
}
\arguments{
\item{wide}{Fitted model object containing the wide model.}

\item{inds}{Matrix or vector of indicators for which parameters are included in the submodel or submodels to be assessed.

A matrix should be supplied if there are multiple submodels.  This should have number of rows equal to the number of submodels, and number of columns equal to the total number of parameters in the wide model.  It contains 1s in the positions where the parameter is included in the submodel, and 0s in positions where the parameter is excluded.  This should always be 1 in the positions defining the narrow model, as specified in \code{inds0}.}

\item{inds0}{Vector of indicators specifying the narrow model, in the same format as \code{inds}.  If this is omitted, the narrow model is assumed to be defined by the first row of \code{inds} (if \code{inds} is a matrix), or \code{inds} itself if this is a vector.}

\item{gamma0}{Vector of special values taken by the parameters \eqn{gamma} which define the narrow model.

This defaults to 0, as in covariate selection, where "excluded" coefficients are fixed to 0. 

This should either be a scalar, assumed to be the same for all parameters fixed in the narrow model, or a vector of length equal to the number of parameters from the wide model which are fixed in the narrow model, that is, the number of entries of \code{inds0} which are zero.}

\item{focus}{Three built-in focus quantities are supported:

\code{"hr"} for the hazard ratio between individuals with covariate values of \code{X} and 0. 

\code{"survival"} for the survival probability at time or times given in \code{t}.

\code{"cumhaz"} for the cumulative hazard at time or times given in \code{t}.

Alternatively, a list of three R functions can be supplied, with components named \code{"focus"}, \code{"deriv"} and \code{"dH"} respectively giving the focus, derivative with respect to the log hazard ratios, and derivative with respect to the cumulative hazards.   Each function should have arguments \code{par}, \code{H0}, \code{X} and \code{t}, giving the log hazard ratios, baseline cumulative hazard, covariate values and time points at which the focus function should be evaluated.  See the section "User-defined focuses" below for a little more information.}

\item{X}{Covariate values to evaluate the focus at.  See \code{\link{fic}}, argument \code{...}, for the required format.}

\item{t}{times to evaluate the focus at.  Only relevant for survival and cumulative hazard focuses, as the hazard ratio is constant through time.}

\item{sub}{If \code{"auto"} (the default) then the submodels are fitted automatically within this function.   If \code{NULL} they are not fitted, and focus estimates are not returned with the results.}

\item{tidy}{If \code{TRUE} the results are returned as a data frame with variables to indicate the submodels, focuses and corresponding result statistics.  If \code{FALSE}, the results are returned as a three-dimensional array, with dimensions indexed by the submodels, result statistics and focuses respectively.}

\item{...}{Other arguments to the focus function can be supplied here.

The built-in focus functions \code{\link{prob_logistic}} and \code{\link{mean_normal}} take an argument \code{X} giving covariate values defining the focus. This can either be a matrix or a vector, or a list or data frame that can be coerced into a matrix. 

If just one focus is needed, then \code{X} can be a vector of length equal to the number of parameters in the wide model.

To compute focused model comparison statistics for multiple focuses defined by the same focus function evaluated at multiple covariate values, \code{X} should be a matrix, with number of columns equal to the number of parameters in the wide model, and number of rows equal to the number of alternative focuses.

For a typical regression model, the first parameter will denote an intercept, so the first value of \code{X} should be 1, and the remaining values should correspond to covariates whose coefficients form parameters of the wide model.  See the examples in the vignette.

Arguments to the focus function other than \code{X} can also be supplied as a matrix, vector, list or data frame in the same way. An exception is when the argument is supplied as a vector, this is assumed to refer to multiple focuses.   For example, suppose the focus function defines the quantile of a distribution, and takes an argument \code{focus_p}, then calling \code{fic(...,focus_p=c(0.1, 0.9))} indicates two alternative focuses defined by the 0.1 and 0.9 quantiles.}
}
\description{
Focused model comparison for Cox models fitted with \code{coxph} from the \code{survival} package.  Built-in focuses include the hazard ratio, survival and cumulative hazard.
}
\details{
Stratified Cox models are not currently supported.
}
\section{User-defined focuses}{


Each function should have four arguments:

\code{par} Vector of estimated coefficients, the log hazard ratios in the Cox model.

\code{H0}  Cumulative hazard estimate at a set of times, in the form of the output from \code{\link[survival]{basehaz}}.   The function \code{\link{get_H0}} can be used on this estimate to obtain the estimate at any other times by interpolation.

\code{X}  Matrix of covariates, with \code{ncov} rows and \code{npar} columns, where \code{ncov} is the number of alternative covariate values definining alternative focuses we want to compare models for, and \code{npar} is the number of coefficients in the model.

\code{t}  Vector of times defining alternative focus quantities (such as the survival)

For examples, examine the source for the built-in functions

\code{fic:::cox_hr,fic:::cox_hr_deriv,fic:::cox_hr_dH} for the hazard ratio between \code{X} and \code{0}

\code{fic:::cox_cumhaz,fic:::cox_cumhaz_deriv,fic:::cox_cumhaz_dH} for the cumulative hazard

\code{fic:::cox_survival,fic:::cox_survival_deriv,fic:::cox_survival_dH} for the survival.
}

\examples{

## Example of focused covariate selection in a Cox model
## For more information see the main package vignette.

library(survival)
wide <- coxph(Surv(years, death==1) ~ sex + thick_centred + infilt +
              epith + ulcer + depth + age, data=melanoma)

## Define models to be selected from
## Sex included in all models
## Select between models including all combinations of other covariates
inds0 <- expand_inds(c(1,0,0,0,0,0,0), wide)
combs <- all_inds(wide, inds0)

## Covariate values defining focus 
newdata <- with(melanoma,
                data.frame(sex = c("female","male"),
                           thick_centred = tapply(thick_centred, sex, mean),
                           infilt=4, epith=1, ulcer=1, depth=2,
                           age = tapply(age, sex, mean)))
X <- newdata_to_X(newdata, wide, intercept=FALSE)

## Focus is 5-year survival for these covariate values
ficall <- fic(wide, inds=combs, inds0=inds0, focus="survival", X=X, t=5)
ggplot_fic(ficall)
summary(ficall)

}
