% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kernel.R
\name{kernel_new}
\alias{kernel_new}
\alias{fmcmc_kernel}
\alias{kernels}
\title{Transition Kernels for MCMC}
\usage{
kernel_new(proposal, ..., logratio = NULL, kernel_env = new.env(hash = TRUE))
}
\arguments{
\item{proposal, logratio}{Functions. Both receive a single argument, an environment.
This functions are called later within \link{MCMC} (see details).}

\item{...}{In the case of \code{kernel_new}, further arguments to be stored with
the kernel.}

\item{kernel_env}{Environment. This will be used as the main container of the
kernel's components. It is returned as an object of class \code{c("environment", "fmcmc_kernel")}.}
}
\value{
An environment of class \code{fmcmc_kernel} which contains the following:
\itemize{
\item \code{proposal} A function that receives a single argument, an environment. This
is the proposal function used within \code{\link[=MCMC]{MCMC()}}.
\item \code{logratio} A function to compute log ratios of the current vs the proposed
step of the chain. Also used within \code{\link[=MCMC]{MCMC()}}.
\item \code{...} Further arguments passed to \code{kernel_new}.
}
}
\description{
The function \code{kernel_new} is a helper function that allows creating
\code{fmcmc_kernel} objects which are passed to the \code{\link[=MCMC]{MCMC()}} function.
}
\details{
The objects \code{fmcmc_kernels} are environments that in general contain the
following objects:
\itemize{
\item \code{proposal}: The function used to propose changes in the chain based
on the current state. The function must return a vector of length equal
to the number of parameters in the model.
\item \code{logratio}: This function is called after a new state has been proposed,
and is used to compute the log of the Hastings ratio.

In the case that the \code{logratio} function is not specified, then it is assumed
that the transition kernel is symmetric, this is, log-ratio is then implemented
as \code{function(env) {env$f1 - env$f0}}
\item \code{...}: Further objects that are used within those functions.
}

Both functions, \code{proposal} and \code{logratio}, receive a single argument, an
environment, which is passed by the \code{\link[=MCMC]{MCMC()}} function during each step using
the function \code{\link[=environment]{environment()}}.

The passed environment is actually the environment in which the \code{MCMC}
function is running, in particular, this environment contains the following
objects:

\tabular{lcl}{
\strong{Object} \tab \tab \strong{Description} \cr
\code{i} \tab \tab Integer. The current iteration. \cr
\code{theta1} \tab \tab Numeric vector. The last proposed state. \cr
\code{theta0} \tab \tab Numeric vector. The current state \cr
\code{f}\tab \tab The log-unnormalized posterior function (a wrapper of \code{fun} passed
to \link{MCMC}). \cr
\code{f1} \tab \tab The last value of \code{f(theta1)} \cr
\code{f0} \tab \tab The last value of \code{f(theta0)} \cr
\code{kernel} \tab \tab The actual \code{fmcmc_kernel} object. \cr
\code{ans} \tab \tab The matrix of samples defined up to \code{i - 1}.
}

These are the core component of the \code{MCMC} function. The following block
of code is how this is actually implemented in the package:\preformatted{for (i in 1L:nsteps) \{
  # Step 1. Propose
  theta1[] <- kernel$proposal(environment())
  f1       <- f(theta1)
  
  # Checking f(theta1) (it must be a number, can be Inf)
  if (is.nan(f1) | is.na(f1) | is.null(f1)) 
    stop(
      "fun(par) is undefined (", f1, ")",
      "Check either -fun- or the -lb- and -ub- parameters.",
      call. = FALSE
    )
  
  # Step 2. Hastings ratio
  if (R[i] < kernel$logratio(environment())) \{
    theta0 <- theta1
    f0     <- f1
  \}
  
  # Step 3. Saving the state
  ans[i,] <- theta0
  
\}
}

For an extensive example on how to create new kernel objects see the vignette
\code{vignette("user-defined-kernels", "fmcmc")}.
}
\section{Behavior}{


In some cases, calls to the \code{proposal()} and \code{logratio()} functions in
\code{fmcmc_kernels} can trigger changes or updates of variables stored within them.
A concrete example is with adaptive kernels.

Adaptive Metropolis and Robust Adaptive Metropolis implemented in the functions
\code{\link[=kernel_adapt]{kernel_adapt()}} and \code{\link[=kernel_ram]{kernel_ram()}} both update a covariance matrix used
during the proposal stage, and furthermore, have a \code{warmup} stage that sets
the point at which both will start adapting. Because of this, both kernels
have internal counters of the absolute step count which allows them activating,
scaling, etc. the proposals correctly.
\enumerate{
\item When running multiple chains, \code{MCMC} will create independent copies of a
baseline passed \code{fmcmc_kernel} object. These are managed together in a
\code{fmcmc_kernel_list} object.
\item Even if the chains are run in parallel, if a predefined kernel object is
passed it will be updated to reflect the last state of the kernels
before the \code{MCMC} call returns.
}
}

\examples{

# Example creating a multivariate normal kernel using the mvtnorm R package
# for a bivariate normal distribution
library(mvtnorm)

# Define your Sigma
sigma <- matrix(c(1, .2, .2, 1), ncol = 2)

# How does it looks like?
sigma
#      [,1] [,2]
# [1,]  1.0  0.2
# [2,]  0.2  1.0

# Create the kernel
kernel_mvn <- kernel_new(
  proposal = function(env) {
  env$theta0 + as.vector(mvtnorm::rmvnorm(1, mean = 0, sigma = sigma.))
  },
  sigma. = sigma
)

# As you can see, in the previous call we passed sigma as it will be used by
# the proposal function
# The logaratio function was not necesary to be passed since this kernel is
# symmetric.

}
\references{
Brooks, S., Gelman, A., Jones, G. L., & Meng, X. L. (2011). Handbook of
Markov Chain Monte Carlo. Handbook of Markov Chain Monte Carlo.
}
\seealso{
Other kernels: 
\code{\link{kernel_adapt}()},
\code{\link{kernel_mirror}},
\code{\link{kernel_normal}()},
\code{\link{kernel_ram}()},
\code{\link{kernel_unif}()}
}
\concept{kernels}
