% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{running_sd3}
\alias{running_sd3}
\alias{running_skew4}
\alias{running_kurt5}
\alias{running_sd}
\alias{running_skew}
\alias{running_kurt}
\alias{running_cent_moments}
\alias{running_std_moments}
\alias{running_cumulants}
\title{Compute first K moments over a sliding window}
\usage{
running_sd3(
  v,
  window = NULL,
  wts = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_skew4(
  v,
  window = NULL,
  wts = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_kurt5(
  v,
  window = NULL,
  wts = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_sd(
  v,
  window = NULL,
  wts = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_skew(
  v,
  window = NULL,
  wts = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_kurt(
  v,
  window = NULL,
  wts = NULL,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 1,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_cent_moments(
  v,
  window = NULL,
  wts = NULL,
  max_order = 5L,
  na_rm = FALSE,
  max_order_only = FALSE,
  min_df = 0L,
  used_df = 0,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_std_moments(
  v,
  window = NULL,
  wts = NULL,
  max_order = 5L,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 0,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)

running_cumulants(
  v,
  window = NULL,
  wts = NULL,
  max_order = 5L,
  na_rm = FALSE,
  min_df = 0L,
  used_df = 0,
  restart_period = 100L,
  check_wts = FALSE,
  normalize_wts = TRUE,
  check_negative_moments = TRUE
)
}
\arguments{
\item{v}{a vector}

\item{window}{the window size. if given as finite integer or double, passed through.
If \code{NULL}, \code{NA_integer_}, \code{NA_real_} or \code{Inf} are given, equivalent
to an infinite window size. If negative, an error will be thrown.}

\item{wts}{an optional vector of weights. Weights are \sQuote{replication}
weights, meaning a value of 2 is shorthand for having two observations
with the corresponding \code{v} value. If \code{NULL}, corresponds to
equal unit weights, the default. Note that weights are typically only meaningfully defined
up to a multiplicative constant, meaning the units of weights are
immaterial, with the exception that methods which check for minimum df will,
in the weighted case, check against the sum of weights. For this reason,
weights less than 1 could cause \code{NA} to be returned unexpectedly due
to the minimum condition. When weights are \code{NA}, the same rules for checking \code{v}
are applied. That is, the observation will not contribute to the moment
if the weight is \code{NA} when \code{na_rm} is true. When there is no
checking, an \code{NA} value will cause the output to be \code{NA}.}

\item{na_rm}{whether to remove NA, false by default.}

\item{min_df}{the minimum df to return a value, otherwise \code{NaN} is returned.
This can be used to prevent moments from being computed on too few observations.
Defaults to zero, meaning no restriction.}

\item{used_df}{the number of degrees of freedom consumed, used in the denominator
of the centered moments computation. These are subtracted from the number of
observations.}

\item{restart_period}{the recompute period. because subtraction of elements can cause
loss of precision, the computation of moments is restarted periodically based on 
this parameter. Larger values mean fewer restarts and faster, though less accurate
results.}

\item{check_wts}{a boolean for whether the code shall check for negative
weights, and throw an error when they are found. Default false for speed.}

\item{normalize_wts}{a boolean for whether the weights should be
renormalized to have a mean value of 1. This mean is computed over elements
which contribute to the moments, so if \code{na_rm} is set, that means non-NA
elements of \code{wts} that correspond to non-NA elements of the data
vector.}

\item{check_negative_moments}{a boolean flag. Normal computation of running
moments can result in negative estimates of even order moments due to loss of
numerical precision. With this flag active, the computation checks for negative
even order moments and restarts the computation when one is detected. This
should eliminate the possibility of negative even order moments. The
downside is the speed hit of checking on every output step. Note also the
code checks for negative moments of every even order tracked, even if they
are not output; that is if the kurtosis, say, is being computed, and a
negative variance is detected, then the computation is restarted.
Defaults to \code{TRUE} to avoid negative even moments. Set to \code{FALSE}
only if you know what you are doing.}

\item{max_order}{the maximum order of the centered moment to be computed.}

\item{max_order_only}{for \code{running_cent_moments}, if this flag is set, only compute
the maximum order centered moment, and return in a vector.}
}
\value{
Typically a matrix, where the first columns are the kth, k-1th through 2nd standardized, 
centered moments, then a column of the mean, then a column of the number of (non-nan) elements in the input,
with the following exceptions:
\describe{
\item{running_cent_moments}{Computes arbitrary order centered moments. When \code{max_order_only} is set,
only a column of the maximum order centered moment is returned.}
\item{running_std_moments}{Computes arbitrary order standardized moments, then the standard deviation, the mean,
and the count. There is not yet an option for \code{max_order_only}, but probably should be.}
\item{running_cumulants}{Computes arbitrary order cumulants, and returns the kth, k-1th, through the second 
(which is the variance) cumulant, then the mean, and the count.}
}
}
\description{
Compute the (standardized) 2nd through kth moments, the mean, and the number of elements over
an infinite or finite sliding window, returning a matrix.
}
\details{
Computes the number of elements, the mean, and the 2nd through kth
centered (and typically standardized) moments, for \eqn{k=2,3,4}{k=2,3,4}. These
are computed via the numerically robust one-pass method of Bennett \emph{et. al.}

Given the length \eqn{n} vector \eqn{x}, we output matrix \eqn{M} where
\eqn{M_{i,j}}{M_i,j} is the \eqn{order - j + 1} moment (\emph{i.e.}
excess kurtosis, skewness, standard deviation, mean or number of elements)
of \eqn{x_{i-window+1},x_{i-window+2},...,x_{i}}{x_(i-window+1),x_(i-window+2),...,x_i}.
Barring \code{NA} or \code{NaN}, this is over a window of size \code{window}.
During the 'burn-in' phase, we take fewer elements.
}
\note{
the kurtosis is \emph{excess kurtosis}, with a 3 subtracted, and should be nearly zero
for Gaussian input.

The moment computations provided by fromo are 
numerically robust, but will often \emph{not} provide the
same results as the 'standard' implementations,
due to differences in roundoff. We make every attempt to balance
speed and robustness. User assumes all risk from using
the fromo package.

Note that when weights are given, they are treated as replication weights.
This can have subtle effects on computations which require minimum
degrees of freedom, since the sum of weights will be compared to
that minimum, not the number of data points. Weight values
(much) less than 1 can cause computations to return \code{NA}
somewhat unexpectedly due to this condition, while values greater
than one might cause the computation to spuriously return a value
with little precision.

As this code may add and remove observations, numerical imprecision
may result in negative estimates of squared quantities, like the
second or fourth moments.  By default we check for this condition
in running computations. It may also be mitigated somewhat by setting 
a smaller \code{restart_period}. Post an issue if you experience this bug.
}
\examples{
x <- rnorm(1e5)
xs3 <- running_sd3(x,10)
xs4 <- running_skew4(x,10)

if (require(moments)) {
    set.seed(123)
    x <- rnorm(5e1)
    window <- 10L
    kt5 <- running_kurt5(x,window=window)
    rm1 <- t(sapply(seq_len(length(x)),function(iii) { 
                xrang <- x[max(1,iii-window+1):iii]
                c(moments::kurtosis(xrang)-3.0,moments::skewness(xrang),
                sd(xrang),mean(xrang),length(xrang)) },
             simplify=TRUE))
    stopifnot(max(abs(kt5 - rm1),na.rm=TRUE) < 1e-12)
}

xc6 <- running_cent_moments(x,window=100L,max_order=6L)

}
\references{
Terriberry, T. "Computing Higher-Order Moments Online."
\url{https://web.archive.org/web/20140423031833/http://people.xiph.org/~tterribe/notes/homs.html}

J. Bennett, et. al., "Numerically Stable, Single-Pass, 
Parallel Statistics Algorithms," Proceedings of IEEE
International Conference on Cluster Computing, 2009.
\doi{10.1109/CLUSTR.2009.5289161}

Cook, J. D. "Accurately computing running variance."
\url{https://www.johndcook.com/standard_deviation/}

Cook, J. D. "Comparing three methods of computing 
standard deviation."
\url{https://www.johndcook.com/blog/2008/09/26/comparing-three-methods-of-computing-standard-deviation/}
}
\author{
Steven E. Pav \email{shabbychef@gmail.com}
}
