% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/galah_geolocate.R, R/galah_bbox.R,
%   R/galah_polygon.R, R/galah_radius.R
\name{galah_geolocate}
\alias{galah_geolocate}
\alias{galah_bbox}
\alias{galah_polygon}
\alias{galah_radius}
\alias{st_crop.data_request}
\title{Narrow a query to within a specified area}
\usage{
galah_geolocate(..., type = c("polygon", "bbox", "radius"))

galah_bbox(...)

galah_polygon(...)

galah_radius(...)

\method{st_crop}{data_request}(x, y, ...)
}
\arguments{
\item{...}{a single \code{sf} object, WKT string or shapefile. Bounding boxes can
be supplied as a \code{tibble}/\code{data.frame} or a \code{bbox}}

\item{type}{\code{string}: one of \code{c("polygon", "bbox")}. Defaults to
\code{"polygon"}. If \code{type = "polygon"}, a multipolygon will be built via
\code{\link[=galah_polygon]{galah_polygon()}}. If \code{type = "bbox"}, a multipolygon will be built via
\code{\link[=galah_bbox]{galah_bbox()}}. The multipolygon is used to narrow a query to the ALA.}

\item{x}{An object of class \code{data_request}, created using \code{\link[=galah_call]{galah_call()}}}

\item{y}{A valid Well-Known Text string (wkt), a \code{POLYGON} or a \code{MULTIPOLYGON}}
}
\value{
If \code{type = "polygon"} or \code{type = "bbox"},
length-1 string (class \code{character}) containing a multipolygon WKT
string representing the area provided.
If \code{type = "radius"},
\code{list} of \code{lat}, \code{long} and \code{radius} values.
}
\description{
Restrict results to those from a specified area using \code{galah_geolocate()}.
Areas can be specified as either polygons or bounding boxes, depending on
\code{type}. Alternatively, users can call the underlying functions directly via
\code{galah_polygon()}, \code{galah_bbox()} or \code{galah_radius()}. It is possible to use \code{sf}
syntax by calling \code{st_crop()}, which is synonymous with \code{galah_polygon()}.

\strong{Use a polygon}
If calling \code{galah_geolocate()}, the default \code{type} is \code{"polygon"}, which
narrows queries to within an area supplied as a \code{POLYGON} or \code{MULTIPOLYGON}.
Polygons must be
specified as either an \code{sf} object, a 'well-known text' (WKT) string, or a
shapefile. Shapefiles must be simple to be accepted by the ALA.
}
\details{
\strong{Use a bounding box}
Alternatively, set \code{type = "bbox"} to narrow queries to within a bounding
box. Bounding boxes can be extracted from a supplied \code{sf} object or
a shapefile. A bounding box can also be supplied as a \code{bbox} object
(via \code{sf::st_bbox()}) or a \code{tibble}/\code{data.frame}.

\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#experimental}{\figure{lifecycle-experimental.svg}{options: alt='[Experimental]'}}}{\strong{[Experimental]}}
\strong{Use a point radius}
Alternatively, set \code{type = "radius"} to narrow queries to within a circular
area around a specific point location. Point coordinates can be supplied as
latitude/longitude coordinate numbers or as an \code{sf} object (\code{sfc_POINT}).
Area is supplied as a \code{radius} in kilometres. Default radius is 10 km.

If \code{type = "polygon"}, WKT strings longer than 10000 characters and
\code{sf} objects with more than 500 vertices will not be
accepted by the ALA. Some polygons  may need to be simplified.
If \code{type = "bbox"}, sf objects and shapefiles will be converted to a bounding
box to query the ALA.
If \verb{type = "radius}, \code{sfc_POINT} objects will be converted to lon/lat
coordinate numbers to query the ALA. Default radius is 10 km.
}
\examples{
\dontrun{
# Search for records within a polygon using a shapefile
location <- sf::st_read("path/to/shapefile.shp")
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location) |>
  atlas_counts()
  
# Search for records within the bounding box of a shapefile
location <- sf::st_read("path/to/shapefile.shp")
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location, type = "bbox") |>
  atlas_counts()

# Search for records within a polygon using an `sf` object
location <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))" |>
 sf::st_as_sfc()
galah_call() |>
  galah_identify("reptilia") |>
  galah_polygon(location) |>
  atlas_counts()
  
# Alternatively, we can use `st_crop()` as a synonym for `galah_polygon()`. 
# Hence the above example can be rewritten as:
request_data() |>
  identify("reptilia") |>
  st_crop(location) |>
  count() |>
  collect()
   
# Search for records using a Well-known Text string (WKT)
wkt <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))"
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(wkt) |>
  atlas_counts()

# Search for records within the bounding box extracted from an `sf` object
location <- "POLYGON((142.3 -29.0,142.7 -29.1,142.7 -29.4,142.3 -29.0))" |>
  sf::st_as_sfc()
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(location, type = "bbox") |>
  atlas_counts()

# Search for records using a bounding box of coordinates
b_box <- sf::st_bbox(c(xmin = 143, xmax = 148, ymin = -29, ymax = -28), 
                     crs = sf::st_crs("WGS84"))
galah_call() |>
  galah_identify("reptilia") |>
  galah_geolocate(b_box, type = "bbox") |>
  atlas_counts()

# Search for records using a bounding box in a `tibble` or `data.frame`
b_box <- tibble::tibble(xmin = 148, ymin = -29, xmax = 143, ymax = -21)
galah_call() |>
  galah_identify("vulpes") |>
  galah_geolocate(b_box, type = "bbox") |>
  atlas_counts()

# Search for records within a radius around a point's coordinates
galah_call() |>
  galah_identify("manorina melanocephala") |>
  galah_geolocate(lat = -33.7,
                  lon = 151.3,
                  radius = 5,
                  type = "radius") |>
  atlas_counts()

# Search for records with a radius around an `sf_POINT` object
point <- sf::st_sfc(sf::st_point(c(-33.66741, 151.3174)), crs = 4326)
galah_call() |>
  galah_identify("manorina melanocephala") |>
  galah_geolocate(point,
                  radius = 5,
                  type = "radius") |>
  atlas_counts()
}
}
