% $Id: reorder.Rd,v 1.11 2005/06/09 14:20:26 nj7w Exp $


\name{reorder.factor}
\alias{reorder.factor}
\title{Reorder the Levels of a Factor}
\description{
  Reorder the levels of a factor
}
\usage{
\method{reorder}{factor}(x,
                         order,
                         X,
                         FUN,
                         sort=mixedsort,
                         make.ordered = is.ordered(x),
                         ... )
}
\arguments{
  \item{x}{factor.}
  \item{order}{A vector of indexes or a vector of label names giving the
    order of the new factor levels.}
  \item{X}{auxillary data vector}
  \item{FUN}{function to be applied to subsets of \code{X} determined by
    \code{x}, to determine factor order.}
  \item{sort}{function to use to sort the factor level names}
  \item{make.ordered}{logical value indicating whether the returned
    object should be an \code{'ordered'} factor.}
  \item{...}{Optional parameters to FUN.}
}
\details{
  This function changes the order of the levels of a factor.  It can do
  so via three different mechanisms, depending on whether \code{order},
  \code{X} \emph{and} \code{FUN}, or \code{sort} are provided.

  If \code{order} is provided:  For a numeric vector, the new factor level names
  are constructed by reordering the factor levels according to the
  numeric values. For vectors, \code{order} gives the list of new factor
  level names. In either case levels omitted from \code{order} will
  become missing values.

  If \code{X} \emph{and} \code{Fun} are provided: The data in \code{X}
  is grouped by the levels of \code{data} and \code{FUN} is applied.
  The groups are then sorted by this value, and the resulting order is
  used for the new factor level names.

  If \code{sort} is provided (as it is by default): The new
  factor level names are generated by applying the supplied function
  to the existing factor level names.  With \code{order="mixedsort"} the
  factor levels are sorted so that combined numeric and character
  strings are sorted in according to character rules on the character
  sections (including ignoring case), and be numeric rules for the
  numeric sections.  See \code{mixedsort} for details.

}
\value{
  A new factor with the levels ordered as specified.
}

\author{Gregory R. Warnes \email{Gregory\_R\_Warnes@groton.pfizer.com}}

\seealso{ \code{\link{factor}}, \code{\link[stats]{reorder}} }

\examples{
   # Create a 4 level example factor
   trt <- factor( sample( c("PLACEBO","300 MG", "600 MG", "1200 MG"),
                  100, replace=TRUE ) )
   summary(trt)
   # Note that the levels are not in a meaningful order.

   library(gtools)
   # Change the order to something useful
   # default "mixedsort" ordering
   trt2 <- reorder(trt)
   summary(trt2)
   # using indexes:
   trt3 <- reorder(trt, c(4,2,3,1))
   summary(trt3)
   # using label names:
   trt4 <- reorder(trt, c("PLACEBO","300 MG", "600 MG", "1200 MG") )
   summary(trt4)
   # using frequency
   trt5 <- reorder(trt, X=as.numeric(trt), FUN=length)
   summary(trt5)


   # drop out the '300 MG' level
   trt6 <- reorder(trt, c("PLACEBO", "600 MG", "1200 MG") )
   summary(trt6)
}
\keyword{ manip }
