#' Download spatial data of urbanized areas in Brazil
#'
#' @description
#' This function reads the official data on the urban footprint of Brazilian cities
#' in the years 2005 and 2015. Original data were generated by the Institute of Geography
#' and Statistics (IBGE)  For more information about the methodology, see details at
#' \url{https://biblioteca.ibge.gov.br/visualizacao/livros/liv100639.pdf}
#'
#' @param year Numeric. Year of the data in YYYY format. Defaults to `2015`.
#' @template simplified
#' @template showProgress
#'
#'
#' @return An `"sf" "data.frame"` object
#'
#' @export
#' @family area functions
#'
#' @examplesIf identical(tolower(Sys.getenv("NOT_CRAN")), "true")
#' # Read urban footprint of Brazilian cities in an specific year
#' d <- read_urban_area(year=2005)
#'
read_urban_area <- function(year=2015, simplified=TRUE, showProgress=TRUE){

  # Get metadata with data url addresses
  temp_meta <- select_metadata(geography="urban_area", year=year, simplified=simplified)

  # list paths of files to download
  file_url <- as.character(temp_meta$download_path)

  # download files
  temp_sf <- download_gpkg(file_url, progress_bar = showProgress)

  # check if download failed
  if (is.null(temp_sf)) { return(invisible(NULL)) }

  return(temp_sf)
}
