% Generated by roxygen2 (4.0.2): do not edit by hand
\name{advanced.procD.lm}
\alias{advanced.procD.lm}
\title{Procrustes ANOVA and pairwise tests for shape data, using complex linear models}
\usage{
advanced.procD.lm(f1, f2, groups = NULL, slope = NULL, angle.type = c("r",
  "deg", "rad"), iter = 999, verbose = FALSE)
}
\arguments{
\item{f1}{A formula for a linear model, containing the response matrix (e.g., y ~ x1 + x2)}

\item{f2}{A formula for another linear model (e.g., ~ x1 + x2 + x3 + a*b) (f1 and f2 should be nested)}

\item{groups}{A formula for grouping factors (e.g., ~ a, or ~ a*b)}

\item{slope}{A formula with one covariate (e.g., ~ x3)}

\item{angle.type}{A value specifying whether differences between slopes should be represented by vector
correlations (r), radians (rad) or degrees (deg)}

\item{iter}{Number of iterations for significance testing}

\item{verbose}{A logical value specifying whether additional output should be displayed}
}
\value{
Function returns an ANOVA table of statistical results for model comparison: error df (for each model), SS, MS,
F ratio, Z, and Prand.
}
\description{
The function quantifies the relative amount of shape variation explained by  a suite of factors
and covariates in a "full" model, after accounting for variation in a "reduced" model.  Inputs are
formulae for full and reduced models (order is not important), plus indication if means or slopes
are to be comapred among groups, with appropriate formulae to define how they should be compared.
}
\details{
The response matrix 'y' must be in the form of a two-dimensional data
  matrix of dimension (n x [p x k]), rather than a 3D array.  It is assumed that the landmarks have previously
  been aligned using Generalized Procrustes Analysis (GPA) [e.g., with \code{\link{gpagen}}]. The function
  \code{\link{two.d.array}} can be used to obtain a two-dimensional data matrix from a 3D array of landmark
  coordinates. The names specified for the independent (x) variables in the formula represent one or more
  vectors containing continuous data or factors. It is assumed that the order of the specimens in the
  shape matrix matches the order of values in the independent variables.

The function performs statistical assessment of the terms in the model using Procrustes distances among
  specimens, rather than explained covariance matrices among variables. With this approach, the sum-of-squared
  Procrustes distances are used as a measure of SS (see Goodall 1991). The SS betwen models is evaluated through
  permutation. In morphometrics this approach is known as a Procrustes ANOVA (Goodall 1991), which is equivalent
  to distance-based anova designs (Anderson 2001). Unlike procD.lm, this function is strictly for comparison
  of two nested models.  The function will readily accept non-nested models, but the results will not be meaningful.
  (Use of procD.lm will be more suitable in most cases.)  A residual randomization permutation procedure (RRPP) is utilized
  for reduced model residuals to evalute the SS between models (Collyer et al. 2014).  Effect-sizes (Z-scores) are
  computed as standard deviates of the SS sampling
  distributions generated, which might be more intuitive for P-values than F-values (see Collyer et al. 2014).

Pairwise tests are only performed if formulae are provided to compute such results.
}
\examples{
### Example of comparison of allometries between two populations of pupfish
### After accounting for sexual dimorphism, Method 1
data(pupfish)
shape <-two.d.array(pupfish$coords)   # GPA-alignment previously performed
CS <- pupfish$CS
Sex <- pupfish$Sex
Pop <- pupfish$Pop
f1 <-  shape ~ log(CS) + Sex + Pop
f2 <- ~ log(CS)*Sex*Pop
advanced.procD.lm(f1, f2, groups = ~Pop, slope = ~ log(CS), angle.type = "r", iter=24)

### Method 2
f1 <-  shape ~ log(CS)*Sex
f2 <- ~ log(CS)*Sex*Pop
advanced.procD.lm(f1, f2, groups = ~Pop, slope = ~ log(CS), angle.type = "r", iter=24)
}
\author{
Michael Collyer
}
\references{
Collyer, M.L., D.J. Sekora, and D.C. Adams. 2014. A method for analysis of phenotypic change for phenotypes described
by high-dimensional data. Heredity. 113: doi:10.1038/hdy.2014.75.
}
\keyword{analysis}

