% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/predict_response.R
\name{predict_response}
\alias{predict_response}
\title{Adjusted predictions and estimated marginal means from regression models}
\usage{
predict_response(
  model,
  terms,
  margin = "mean_reference",
  ci_level = 0.95,
  type = "fixed",
  condition = NULL,
  back_transform = TRUE,
  ppd = FALSE,
  vcov_fun = NULL,
  vcov_type = NULL,
  vcov_args = NULL,
  weights = NULL,
  interval,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{model}{A model object.}

\item{terms}{Names of those terms from \code{model}, for which predictions should
be displayed (so called \emph{focal terms}). Can be:
\itemize{
\item A character vector, specifying the names of the focal terms. This is the
preferred and probably most flexible way to specify focal terms, e.g.
\code{terms = "x [40:60]"}, to calculate predictions for the values 40 to 60.
\item A list, where each element is a named vector, specifying the focal terms
and their values. This is the "classical" R way to specify focal terms,
e.g. \code{list(x = 40:60)}.
\item A formula, e.g. \code{terms = ~ x + z}, which is internally converted to a
character vector. This is probably the least flexible way, as you cannot
specify representative values for the focal terms.
\item A data frame representig a "data grid" or "reference grid". Predictions
are then made for all combinations of the variables in the data frame.
}

\code{term} at least requires one variable name. The maximum length is four terms,
where the second to fourth term indicate the groups, i.e. predictions of first
term are grouped at meaningful values or levels of the remaining terms (see
\code{\link[=values_at]{values_at()}}). It is also possible to define specific values for focal
terms, at which adjusted predictions should be calculated (see details below).
All remaining covariates that are not specified in \code{terms} are "marginalized",
see the \code{margin} argument. See also argument \code{condition} to fix non-focal
terms to specific values.}

\item{margin}{Character string, indicating how to marginalize over the
\emph{non-focal} predictors, i.e. those variables that are \emph{not} specified in
\code{terms}. Possible values are \code{"mean_reference"}, \code{"mean_mode"},
\code{"marginalmeans"} and \code{"empirical"} (or \code{"counterfactual"}, aka average
"counterfactual" predictions). You can set a default-option for the \code{margin}
argument via \code{options()}, e.g. \code{options(ggeffects_margin = "empirical")},
so you don't have to specify your preferred marginalization method each time
you call \code{predict_response()}. See details in the documentation below.}

\item{ci_level}{Numeric, the level of the confidence intervals. Use
\code{ci_level = NA} if confidence intervals should not be calculated
(for instance, due to computation time). Typically, confidence intervals are
based on the returned standard errors for the predictions, assuming a t- or
normal distribution (based on the model and the available degrees of freedom,
i.e. roughly \verb{+/- 1.96 * SE}). See introduction of
\href{https://strengejacke.github.io/ggeffects/articles/ggeffects.html}{this vignette}
for more details.}

\item{type}{Character, indicating whether predictions should be conditioned
on specific model components or not. Consequently, most options only apply
for survival models, mixed effects models and/or models with zero-inflation
(and their Bayesian counter-parts); only exeption is \code{type = "simulate"},
which is available for some other model classes as well (which respond to
\code{simulate()}).

\strong{Note 1:} For \code{brmsfit}-models with zero-inflation component, there is no
\code{type = "zero_inflated"} nor \code{type = "zi_random"}; predicted values for these
models \emph{always} condition on the zero-inflation part of the model. The same
is true for \code{MixMod}-models from \strong{GLMMadaptive} with zero-inflation
component (see 'Details').

\strong{Note 2:} If \code{margin = "empirical"} (i.e. counterfactual predictions), the
\code{type} argument is handled differently. It is set to \code{"response"} by default,
and usually accepts all values from the \code{type}-argument of the model's respective
\code{predict()} method. E.g., passing a \code{glm} object would allow the options
\code{"response"}, \code{"link"}, and \code{"terms"}. Thus, the following options apply to
\code{predict_response()} when \code{margin} is \emph{not} \code{"empirical"}, and are passed to
\code{ggpredict()} or \code{ggemmeans()}, respectively (depending on the value of \code{margin}):
\itemize{
\item \code{"fixed"} (or \code{"fe"} or \code{"count"})

Predicted values are conditioned on the fixed effects or conditional
model only (for mixed models: predicted values are on the population-level
and \emph{confidence intervals} are returned, i.e. \code{re.form = NA} when calling
\code{predict()}). For instance, for models fitted with \code{zeroinfl} from \strong{pscl},
this would return the predicted mean from the count component (without
zero-inflation). For models with zero-inflation component, this type calls
\code{predict(..., type = "link")} (however, predicted values are
back-transformed to the response scale).
\item \code{"fixed_ppd"}

Only applies to \code{margin = "mean_reference"}, and only for Bayesian
models of class \code{stanreg} or \code{brmsfit}. Computes the posterior predictive
distribution. It is the same as setting \code{type = "fixed"} in combination with
\code{ppd = TRUE}.
\item \code{"random"} (or \code{"re"})

This only applies to mixed models, and \code{type = "random"} does not condition
on the zero-inflation component of the model. \code{type = "random"} still
returns population-level predictions, however, conditioned on random effects
and considering individual level predictions, i.e. \code{re.form = NULL} when
calling \code{predict()}. This may affect the returned predicted values, depending
on whether \code{REML = TRUE} or \code{REML = FALSE} was used for model fitting.
Furthermore, unlike \code{type = "fixed"}, intervals also consider the uncertainty
in the variance parameters (the mean random effect variance, see \emph{Johnson
et al. 2014} for details) and hence can be considered as \emph{prediction intervals}.
For models with zero-inflation component, this type calls
\code{predict(..., type = "link")} (however, predicted values are back-transformed
to the response scale).

To get predicted values for each level of the random effects groups, add the
name of the related random effect term to the \code{terms}-argument
(for more details, see
\href{https://strengejacke.github.io/ggeffects/articles/introduction_effectsatvalues.html}{this vignette}).
\item \code{"random_ppd"}

Only applies to \code{margin = "mean_reference"},, and only for Bayesian
models of class \code{stanreg} or \code{brmsfit}. Computes the posterior predictive
distribution. It is the same as setting \code{type = "random"} in combination with
\code{ppd = TRUE}.
\item \code{"zero_inflated"} (or \code{"fe.zi"} or \code{"zi"})

Predicted values are conditioned on the fixed effects and the zero-inflation
component. For instance, for models fitted with \code{zeroinfl}
from \strong{pscl}, this would return the predicted response (\code{mu*(1-p)})
and for \strong{glmmTMB}, this would return the expected value \code{mu*(1-p)}
\emph{without} conditioning on random effects (i.e. random effect variances
are not taken into account for the confidence intervals). For models with
zero-inflation component, this type calls \code{predict(..., type = "response")}.
See 'Details'.
\item \code{"zi_random"} (or \code{"re.zi"} or \code{"zero_inflated_random"})

Predicted values are conditioned on the zero-inflation component and
take the random effects uncertainty into account. For models fitted with
\code{glmmTMB()}, \code{hurdle()} or \code{zeroinfl()}, this would return the
expected value \code{mu*(1-p)}. For \strong{glmmTMB}, prediction intervals
also consider the uncertainty in the random effects variances. This
type calls \code{predict(..., type = "response")}. See 'Details'.
\item \code{"zi_prob"} (or \code{"zi.prob"})

Predicted zero-inflation probability. For \strong{glmmTMB} models with
zero-inflation component, this type calls \code{predict(..., type = "zlink")};
models from \strong{pscl} call \code{predict(..., type = "zero")} and for
\strong{GLMMadaptive}, \code{predict(..., type = "zero_part")} is called.
\item \code{"simulate"} (or \code{"sim"})

Predicted values and confidence resp. prediction intervals are
based on simulations, i.e. calls to \code{simulate()}. This type
of prediction takes all model uncertainty into account, including
random effects variances. Currently supported models are objects of
class \code{lm}, \code{glm}, \code{glmmTMB}, \code{wbm}, \code{MixMod}
and \code{merMod}. See \code{...} for details on number of simulations.
\item \code{"survival"} and \code{"cumulative_hazard"} (or \code{"surv"} and \code{"cumhaz"})

Applies only to \code{coxph}-objects from the \strong{survial}-package and
calculates the survival probability or the cumulative hazard of an event.
}

When \code{margin = "empirical"}, the \code{type} argument accepts all values from
the \code{type}-argument of the model's respective \code{predict()}-method.}

\item{condition}{Named character vector, which indicates covariates that
should be held constant at specific values. Unlike \code{typical}, which
applies a function to the covariates to determine the value that is used
to hold these covariates constant, \code{condition} can be used to define
exact values, for instance \code{condition = c(covariate1 = 20, covariate2 = 5)}.
See 'Examples'.}

\item{back_transform}{Logical, if \code{TRUE} (the default), predicted values
for log- or log-log transformed responses will be back-transformed to
original response-scale.}

\item{ppd}{Logical, if \code{TRUE}, predictions for Stan-models are based on the
posterior predictive distribution \code{\link[rstantools:posterior_predict]{rstantools::posterior_predict()}}. If
\code{FALSE} (the default), predictions are based on posterior draws of the linear
predictor \code{\link[rstantools:posterior_linpred]{rstantools::posterior_linpred()}}.}

\item{vcov_fun}{Variance-covariance matrix used to compute uncertainty
estimates (e.g., for confidence intervals based on robust standard errors).
This argument accepts a covariance matrix, a function which returns a
covariance matrix, or a string which identifies the function to be used to
compute the covariance matrix.
\itemize{
\item A (variance-covariance) matrix
\item A function which returns a covariance matrix (e.g., \code{stats::vcov()})
\item A string which indicates the estimation type for the heteroscedasticity-consistent
variance-covariance matrix, e.g. \code{vcov_fun = "HC0"}. Possible values are
\code{"HC0"}, \code{"HC1"}, \code{"HC2"}, \code{"HC3"}, \code{"HC4"}, \code{"HC4m"}, and \code{"HC5"}, which
will then call the \code{vcovHC()}-function from the \strong{sandwich} package, using
the specified type. Further possible values are \code{"CR0"}, \code{"CR1"}, \code{"CR1p"},
\code{"CR1S"}, \code{"CR2"}, and \code{"CR3"}, which will call the \code{vcovCR()}-function from
the \strong{clubSandwich} package.
\item A string which indicates the name of the \verb{vcov*()}-function from the
\strong{sandwich} or \strong{clubSandwich} packages, e.g. \code{vcov_fun = "vcovCL"},
which is used to compute (cluster) robust standard errors for predictions.
}

If \code{NULL}, standard errors (and confidence intervals) for predictions are
based on the standard errors as returned by the \code{predict()}-function.
\strong{Note} that probably not all model objects that work with \code{ggpredict()}
are also supported by the \strong{sandwich} or \strong{clubSandwich} packages.

See details in \href{https://strengejacke.github.io/ggeffects/articles/practical_robustestimation.html}{this vignette}.}

\item{vcov_type}{Character vector, specifying the estimation type for the
robust covariance matrix estimation (see \code{?sandwich::vcovHC}
or \code{?clubSandwich::vcovCR} for details). Only used when \code{vcov_fun} is a
character string indicating one of the functions from those packages.}

\item{vcov_args}{List of named vectors, used as additional arguments that
are passed down to \code{vcov_fun}.}

\item{weights}{Character vector, naming the weigthing variable in the data,
or a vector of weights (of same length as the number of observations in the
data). Only applies to \code{margin = "empirical"}.}

\item{interval}{Type of interval calculation, can either be \code{"confidence"}
(default) or \code{"prediction"}. May be abbreviated. Unlike \emph{confidence intervals},
\emph{prediction intervals} include the residual variance (sigma^2) to account for
the uncertainty of predicted values. For mixed models, \code{interval = "prediction"}
is the default for \code{type = "random"}. When \code{type = "fixed"}, the default is
\code{interval = "confidence"}. Note that prediction intervals are not available
for all models, but only for models that work with \code{\link[insight:get_sigma]{insight::get_sigma()}}.}

\item{verbose}{Toggle messages or warnings.}

\item{...}{If \code{margin} is set to \code{"mean_reference"} or \code{"mean_mode"}, arguments
are passed down to \code{ggpredict()} (further down to \code{predict()}); for
\code{margin = "marginalmeans"}, further arguments passed down to \code{ggemmeans()} and
thereby to \code{emmeans::emmeans()}; if \code{margin = "empirical"}, further arguments are
passed down to \code{marginaleffects::avg_predictions()}. If \code{type = "simulate"},
\code{...} may also be used to set the number of simulation, e.g. \code{nsim = 500}.
When calling \code{ggeffect()}, further arguments passed down to \code{effects::Effect()}.}
}
\value{
A data frame (with \code{ggeffects} class attribute) with consistent data columns:
\itemize{
\item \code{"x"}: the values of the first term in \code{terms}, used as x-position in plots.
\item \code{"predicted"}: the predicted values of the response, used as y-position in plots.
\item \code{"std.error"}: the standard error of the predictions. \emph{Note that the standard
errors are always on the link-scale, and not back-transformed for non-Gaussian
models!}
\item \code{"conf.low"}: the lower bound of the confidence interval for the predicted values.
\item \code{"conf.high"}: the upper bound of the confidence interval for the predicted values.
\item \code{"group"}: the grouping level from the second term in \code{terms}, used as
grouping-aesthetics in plots.
\item \code{"facet"}: the grouping level from the third term in \code{terms}, used to indicate
facets in plots.

The estimated marginal means (or predicted values) are always on the
response scale!

For proportional odds logistic regression (see \code{?MASS::polr})
resp. cumulative link models (e.g., see \code{?ordinal::clm}),
an additional column \code{"response.level"} is returned, which indicates
the grouping of predictions based on the level of the model's response.

Note that for convenience reasons, the columns for the intervals
are always named \code{"conf.low"} and \code{"conf.high"}, even though
for Bayesian models credible or highest posterior density intervals
are returned.

There is an \code{\link[=as.data.frame]{as.data.frame()}} method for objects of class \code{ggeffects},
which has an \code{terms_to_colnames} argument, to use the term names as column
names instead of the standardized names \code{"x"} etc.
}
}
\description{
The \strong{ggeffects} package computes marginal means and adjusted predicted
values for the response, at the margin of specific values or levels from
certain model terms. The package is built around three core functions:
\code{predict_response()} (understanding results), \code{test_predictions()} (importance
of results) and \code{plot()} (communicate results).

By default, adjusted predictions or marginal means are returned on the
\emph{response} scale, which is the easiest and most intuitive scale to interpret
the results. There are other options for specific models as well, e.g. with
zero-inflation component (see documentation of the \code{type}-argument). The
result is returned as structured data frame, which is nicely printed by
default. \code{plot()} can be used to easily create figures.

The main function to calculate marginal means and adjusted predictions is
\code{predict_response()}, which returns adjusted predictions, marginal means
or averaged counterfactual predictions depending on value of the
\code{margin}-argument.

In previous versions of \strong{ggeffects}, the functions \code{ggpredict()}, \code{ggemmeans()},
\code{ggeffect()} and \code{ggaverage()} were used to calculate marginal means and
adjusted predictions. These functions are still available, but \code{predict_response()}
as a "wrapper" around these functions is the preferred way to calculate marginal
means and adjusted predictions now.
}
\note{
\strong{Printing Results}

The \code{print()} method gives a clean output (especially for predictions by
groups), and indicates at which values covariates were held constant.
Furthermore, the \code{print()} method has several arguments to customize the
output. See \href{https://strengejacke.github.io/ggeffects/articles/introduction_print.html}{this vignette}
for details.

\strong{Limitations}

The support for some models, for example from package \strong{MCMCglmm}, is
rather experimental and may fail for certain models. If you encounter
any errors, please file an issue \href{https://github.com/strengejacke/ggeffects/issues}{at Github}.
}
\section{Supported Models}{


A list of supported models can be found at \href{https://github.com/strengejacke/ggeffects}{the package website}.
Support for models varies by marginalization method (the \code{margin} argument),
i.e. although \code{predict_response()} supports most models, some models are only
supported exclusively by one of the four downstream functions (\code{ggpredict()},
\code{ggemmeans()}, \code{ggeffect()} or \code{ggaverage()}). This means that not all models
work for every \code{margin} option of \code{predict_response()}.
}

\section{Holding covariates at constant values, or how to marginalize over the \emph{non-focal} predictors}{


\code{predict_response()} is a wrapper around \code{ggpredict()}, \code{ggemmeans()} and
\code{ggaverage()}. Depending on the value of the \code{margin} argument,
\code{predict_response()} calls one of those functions. The \code{margin} argument
indicates how to marginalize over the \emph{non-focal} predictors, i.e. those
variables that are \emph{not} specified in \code{terms}. Possible values are:
\itemize{
\item \code{"mean_reference"} and \code{"mean_mode"}: For \code{"mean_reference"}, non-focal
predictors are set to their mean (numeric variables), reference level
(factors), or "most common" value (mode) in case of character vectors.
For \code{"mean_mode"}, non-focal predictors are set to their mean (numeric
variables) or mode (factors, or "most common" value in case of character
vectors).

These predictons represent a rather "theoretical" view on your data,
which does not necessarily exactly reflect the characteristics of your
sample. It helps answer the question, "What is the predicted value of the
response at meaningful values or levels of my focal terms for a 'typical'
observation in my data?", where 'typical' refers to certain characteristics
of the remaining predictors.
\item \code{"marginalmeans"}: non-focal predictors are set to their mean (numeric
variables) or averaged over the levels or "values" for factors and
character vectors. Averaging over the factor levels of non-focal terms
computes a kind of "weighted average" for the values at which these terms
are hold constant. Thus, non-focal categorical terms are conditioned on
"weighted averages" of their levels.

These predictions come closer to the sample, because all possible values
and levels of the non-focal predictors are taken  into account. It would
answer the question, "What is the predicted value of the response at
meaningful values or levels of my focal terms for an 'average' observation
in my data?". It refers to randomly picking a subject of your sample and
the result you get on average.
\item \code{"empirical"} (or \code{"counterfactual"}): non-focal predictors are averaged
over the observations in the sample. The response is predicted for each
subject in the data and predicted values are then averaged across all
subjects, aggregated/grouped by the focal terms. In particular, averaging
is applied to \emph{counterfactual predictions} (Dickerman and Hernan 2020).
There is a more detailed description in
\href{https://strengejacke.github.io/ggeffects/articles/technical_differencepredictemmeans.html}{this vignette}.

Counterfactual predictions are useful, insofar as the results can also
be transferred to other contexts. It answers the question, "What is the
predicted value of the response at meaningful values or levels of my focal
terms for the 'average' observation in the population?". It does not only
refer to the actual data in your sample, but also "what would be if" we had
more data, or if we had data from a different population. This is where
"counterfactual" refers to.
}

You can set a default-option for the \code{margin} argument via \code{options()}, e.g.
\code{options(ggeffects_margin = "empirical")}, so you don't have to specify your
"default" marginalization method each time you call \code{predict_response()}.
Use \code{options(ggeffects_margin = NULL)} to remove that setting.

The \code{condition} argument can be used to fix non-focal terms to specific
values.
}

\section{Marginal Means and Adjusted Predictions at Specific Values}{


Meaningful values of focal terms can be specified via the \code{terms} argument.
Specifying meaningful or representative values as string pattern is the
preferred way in the \strong{ggeffects} package. However, it is also possible to
use a \code{list()} for the focal terms if prefer the "classical" R way, which is
described in \href{https://strengejacke.github.io/ggeffects/articles/introduction_effectsatvalues.html}{this vignette}.

Indicating levels in square brackets allows for selecting only certain
groups or values resp. value ranges. The term name and the start of the
levels in brackets must be separated by a whitespace character, e.g.
\code{terms = c("age", "education [1,3]")}. Numeric ranges, separated with colon,
are also allowed: \code{terms = c("education", "age [30:60]")}. The stepsize for
ranges can be adjusted using \code{by}, e.g. \code{terms = "age [30:60 by=5]"}.

The \code{terms} argument also supports the same shortcuts as the \code{values} argument
in \code{values_at()}. So \code{terms = "age [meansd]"} would return predictions for
the values one standard deviation below the mean age, the mean age and one SD
above the mean age. \code{terms = "age [quart2]"} would calculate predictions at
the value of the lower, median and upper quartile of age.

Furthermore, it is possible to specify a function name. Values for predictions
will then be transformed, e.g. \code{terms = "income [exp]"}. This is useful when
model predictors were transformed for fitting the model and should be
back-transformed to the original scale for predictions. It is also possible
to define own functions (see
\href{https://strengejacke.github.io/ggeffects/articles/introduction_effectsatvalues.html}{this vignette}).

Instead of a function, it is also possible to define the name of a variable
with specific values, e.g. to define a vector \code{v = c(1000, 2000, 3000)} and
then use \code{terms = "income [v]"}.

You can take a random sample of any size with \code{sample=n}, e.g
\code{terms = "income [sample=8]"}, which will sample eight values from
all possible values of the variable \code{income}. This option is especially
useful for plotting predictions at certain levels of random effects
group levels, where the group factor has many levels that can be completely
plotted. For more details, see
\href{https://strengejacke.github.io/ggeffects/articles/introduction_effectsatvalues.html}{this vignette}.

Finally, numeric vectors for which no specific values are given, a "pretty range"
is calculated (see \code{\link[=pretty_range]{pretty_range()}}), to avoid memory allocation problems
for vectors with many unique values. If a numeric vector is specified as
second or third term (i.e. if this vector represents a grouping structure),
representative values (see \code{\link[=values_at]{values_at()}}) are chosen (unless other values
are specified). If all values for a numeric vector should be used to compute
predictions, you may use e.g. \code{terms = "age [all]"}. See also package vignettes.

To create a pretty range that should be smaller or larger than the default
range (i.e. if no specific values would be given), use the \code{n} tag, e.g.
\code{terms="age [n=5]"} or \code{terms="age [n=12]"}. Larger values for \code{n} return a
larger range of predicted values.
}

\section{Bayesian Regression Models}{


\code{predict_response()} also works with \strong{Stan}-models from the \strong{rstanarm} or
\strong{brms}-packages. The predicted values are the median value of all drawn
posterior samples. The confidence intervals for Stan-models are Bayesian
predictive intervals. By default (i.e. \code{ppd = FALSE}), the predictions are
based on \code{\link[rstantools:posterior_linpred]{rstantools::posterior_linpred()}} and hence have some limitations:
the uncertainty of the error term is not taken into account. The recommendation
is to use the posterior predictive distribution (\code{\link[rstantools:posterior_predict]{rstantools::posterior_predict()}}).
}

\section{Zero-Inflated and Zero-Inflated Mixed Models with brms}{


Models of class \code{brmsfit} always condition on the zero-inflation component,
if the model has such a component. Hence, there is no \code{type = "zero_inflated"}
nor \code{type = "zi_random"} for \code{brmsfit}-models, because predictions are based
on draws of the posterior distribution, which already account for the
zero-inflation part of the model.

\strong{Zero-Inflated and Zero-Inflated Mixed Models with glmmTMB}

If \code{model} is of class \code{glmmTMB}, \code{hurdle}, \code{zeroinfl} or \code{zerotrunc}, and
\code{margin} is \emph{not} set to \verb{"empirical},  simulations from a multivariate
normal distribution (see \code{?MASS::mvrnorm}) are drawn to calculate \code{mu*(1-p)}.
Confidence intervals are then based on quantiles of these results.
For \code{type = "zi_random"}, prediction intervals also take the uncertainty in
the random-effect paramters into account (see also \emph{Brooks et al. 2017},
pp.391-392 for details).

An alternative for models fitted with \strong{glmmTMB} that take all model
uncertainties into account are simulations based on \code{simulate()}, which
is used when \code{type = "simulate"} (see \emph{Brooks et al. 2017}, pp.392-393 for
details).

Finally, if \code{margin = "empirical"}, the returned predictions are already
conditioned on the zero-inflation part (and possible random effects) of the
model, thus these are most comparable to the \code{type = "simulate"} option. In
other words, if all model components should be taken into account for
predictions, you should consider using \code{margin = "empirical"}.
}

\section{MixMod-models from GLMMadaptive}{


Predicted values for the fixed effects component (\code{type = "fixed"} or
\code{type = "zero_inflated"}) are based on \code{predict(..., type = "mean_subject")},
while predicted values for random effects components (\code{type = "random"} or
\code{type = "zi_random"}) are calculated with \code{predict(..., type = "subject_specific")}
(see \code{?GLMMadaptive::predict.MixMod} for details). The latter option
requires the response variable to be defined in the \code{newdata}-argument
of \code{predict()}, which will be set to its typical value (see
\code{\link[=values_at]{values_at()}}).
}

\section{Multinomial Models}{


\code{polr}, \code{clm} models, or more generally speaking, models with ordinal or
multinominal outcomes, have an additional column \code{response.level}, which
indicates with which level of the response variable the predicted values are
associated.
}

\examples{
\dontshow{if (requireNamespace("sjlabelled") && requireNamespace("ggplot2")) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
library(sjlabelled)
data(efc)
fit <- lm(barthtot ~ c12hour + neg_c_7 + c161sex + c172code, data = efc)

predict_response(fit, terms = "c12hour")
predict_response(fit, terms = c("c12hour", "c172code"))
# more compact table layout for printing
out <- predict_response(fit, terms = c("c12hour", "c172code", "c161sex"))
print(out, collapse_table = TRUE)

# specified as formula
predict_response(fit, terms = ~ c12hour + c172code + c161sex)

# only range of 40 to 60 for variable 'c12hour'
predict_response(fit, terms = "c12hour [40:60]")

# terms as named list
predict_response(fit, terms = list(c12hour = 40:60))

# covariate "neg_c_7" is held constant at a value of 11.84 (its mean value).
# To use a different value, use "condition"
predict_response(fit, terms = "c12hour [40:60]", condition = c(neg_c_7 = 20))

# to plot ggeffects-objects, you can use the 'plot()'-function.
# the following examples show how to build your ggplot by hand.

\donttest{
# plot predicted values, remaining covariates held constant
library(ggplot2)
mydf <- predict_response(fit, terms = "c12hour")
ggplot(mydf, aes(x, predicted)) +
  geom_line() +
  geom_ribbon(aes(ymin = conf.low, ymax = conf.high), alpha = 0.1)

# three variables, so we can use facets and groups
mydf <- predict_response(fit, terms = c("c12hour", "c161sex", "c172code"))
ggplot(mydf, aes(x = x, y = predicted, colour = group)) +
  stat_smooth(method = "lm", se = FALSE) +
  facet_wrap(~facet, ncol = 2)

# select specific levels for grouping terms
mydf <- predict_response(fit, terms = c("c12hour", "c172code [1,3]", "c161sex"))
ggplot(mydf, aes(x = x, y = predicted, colour = group)) +
  stat_smooth(method = "lm", se = FALSE) +
  facet_wrap(~facet) +
  labs(
    y = get_y_title(mydf),
    x = get_x_title(mydf),
    colour = get_legend_title(mydf)
  )

# level indication also works for factors with non-numeric levels
# and in combination with numeric levels for other variables
data(efc)
efc$c172code <- sjlabelled::as_label(efc$c172code)
fit <- lm(barthtot ~ c12hour + neg_c_7 + c161sex + c172code, data = efc)
predict_response(fit, terms = c("c12hour",
  "c172code [low level of education, high level of education]",
  "c161sex [1]"))

# when "terms" is a named list
predict_response(fit, terms = list(
  c12hour = seq(0, 170, 30),
  c172code = c("low level of education", "high level of education"),
  c161sex = 1)
)

# use categorical value on x-axis, use axis-labels, add error bars
dat <- predict_response(fit, terms = c("c172code", "c161sex"))
ggplot(dat, aes(x, predicted, colour = group)) +
  geom_point(position = position_dodge(0.1)) +
  geom_errorbar(
    aes(ymin = conf.low, ymax = conf.high),
    position = position_dodge(0.1)
  ) +
  scale_x_discrete(breaks = 1:3, labels = get_x_labels(dat))

# 3-way-interaction with 2 continuous variables
data(efc)
# make categorical
efc$c161sex <- as_factor(efc$c161sex)
fit <- lm(neg_c_7 ~ c12hour * barthtot * c161sex, data = efc)
# select only levels 30, 50 and 70 from continuous variable Barthel-Index
dat <- predict_response(fit, terms = c("c12hour", "barthtot [30,50,70]", "c161sex"))
ggplot(dat, aes(x = x, y = predicted, colour = group)) +
  stat_smooth(method = "lm", se = FALSE, fullrange = TRUE) +
  facet_wrap(~facet) +
  labs(
    colour = get_legend_title(dat),
    x = get_x_title(dat),
    y = get_y_title(dat),
    title = get_title(dat)
  )

# or with ggeffects' plot-method
plot(dat, ci = FALSE)
}

# predictions for polynomial terms
data(efc)
fit <- glm(
  tot_sc_e ~ c12hour + e42dep + e17age + I(e17age^2) + I(e17age^3),
  data = efc,
  family = poisson()
)
predict_response(fit, terms = "e17age")
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Brooks ME, Kristensen K, Benthem KJ van, Magnusson A, Berg CW, Nielsen A,
et al. glmmTMB Balances Speed and Flexibility Among Packages for Zero-inflated
Generalized Linear Mixed Modeling. The R Journal. 2017;9: 378-400.
\item Johnson PC, O'Hara RB. 2014. Extension of Nakagawa & Schielzeth's R2GLMM
to random slopes models. Methods Ecol Evol, 5: 944-946.
\item Dickerman BA, Hernan, MA. Counterfactual prediction is not only for causal
inference. Eur J Epidemiol 35, 615–617 (2020).
}
}
