% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggetho.R
\name{ggetho}
\alias{ggetho}
\title{Prepare a ggplot object to represent behavioural data}
\usage{
ggetho(data, mapping, summary_FUN = mean, summary_time_window = mins(30),
  time_wrap = NULL, time_offset = 0, multiplot = NULL,
  multiplot_period = hours(24), ...)
}
\arguments{
\item{data}{\link[behavr:behavr]{behavr::behavr} table containing the data and metadata}

\item{mapping}{default list of aesthetic mappings to use for plot}

\item{summary_FUN}{method (function) used to summarise \code{variable} over time (typically, the mean)}

\item{summary_time_window}{width (in seconds) of the time window to compute a summary on}

\item{time_wrap}{time (in seconds) used to wrap the data (see details)}

\item{time_offset}{time offset (i.e. phase, in seconds) when using \code{time_wrap}}

\item{multiplot}{integer, greater than two, or NULL, the default (see details)}

\item{multiplot_period}{the duration of the period when mutiplotting (see details)}

\item{...}{additional arguments to be passed to \code{\link[ggplot2:ggplot]{ggplot2::ggplot()}}}
}
\value{
An initial plot object that can be further edited.
}
\description{
This function summarises a variable of interest (y or z axis)
in order to subsequently represent it over time (x axis)
(using layers provided either by \code{ggplot2} or \code{ggetho}).
}
\details{
\code{time_wrap} is typically used to express time relatively to the start of the the day.
In other words, it can help be used to pull all days together in one representative day.
In this case, \code{time_wrap = hours(24)}.
Instead of representing data from the start of the day, it can be done from any offset, using \code{time_offset}.
For instance,  \code{time_offset = hours(12)} puts the circadian reference (ZT0) in the middle of the plot.

Multiplotting is a generalisation of double-plotting, triple-plotting...
This type or representation is useful to understand periodic behaviours.
When \code{multiplot} is \emph{not} NULL, data is repeated as
many time as its value, along the x axis.
The y axis is then the period (typically the day) onset.
It is possible to set duration of the period, which is typically 24 h to arbitrary values using the
\code{multiplot_period} argument.
}
\examples{
# We start by making a dataset with 20 animals
metadata <- data.table(id = sprintf("toy_experiment|\%02d", 1:20),
                   condition = c("A", "B"))
dt <- toy_activity_data(metadata, 3)
# We build a plot object with **nothing inside** (just the axis)
# we want to show proportion of time sleeping  on the y axis:
pl <- ggetho(dt, aes(y = asleep))
pl
# Sometimes, the variable of interest in not on the y axis, but on z axis (colour scale).
# When we do not provide a y axis,
# ggetho will make an ID fo each animal and display them on separate rows
pl <- ggetho(dt, aes(z = asleep))
pl
# this one is the same type, but it groups the animals by condition
pl <- ggetho(dt, aes(z = asleep, y = condition))
pl
# sorting with paste
pl <- ggetho(dt, aes(z = asleep,y = paste(condition, id)))
pl

# we want to summarise (wrap) data along a circadian day:
pl <- ggetho(dt, aes(y = asleep), time_wrap = hours(24))
pl

# double-plotted actogram:
pl <- ggetho(dt,
              aes(z = moving),
              multiplot = 2,
              multiplot_period = hours(24))
pl
# then use `+ stat_tile_etho()` , or `+ stat_bar_tile_etho()`
}
\references{
\itemize{
\item The relevant \href{https://rethomics.github.io/ggetho.html#the-ggetho-function}{rethomic tutorial section}
}
}
\seealso{
\itemize{
\item \link{stat_pop_etho} to show population trend by aggregating individuals over time
\item \link{stat_tile_etho} to show variable of interest as colour intensity
\item \link{stat_ld_annotations} to show light and dark phases on the plot
}
}
