% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-plot.R
\name{plot.gkwreg}
\alias{plot.gkwreg}
\title{Diagnostic Plots for Generalized Kumaraswamy Regression Models}
\usage{
\method{plot}{gkwreg}(
  x,
  which = 1:6,
  type = c("quantile", "pearson", "deviance"),
  family = NULL,
  caption = NULL,
  main = "",
  sub.caption = "",
  ask = NULL,
  use_ggplot = FALSE,
  arrange_plots = FALSE,
  nsim = 100,
  level = 0.9,
  sample_size = NULL,
  theme_fn = NULL,
  save_diagnostics = FALSE,
  ...
)
}
\arguments{
\item{x}{An object of class \code{"gkwreg"}, typically the result of a call
to \code{\link{gkwreg}}.}

\item{which}{Integer vector specifying which diagnostic plots to produce.
If a subset of the plots is required, specify a subset of the numbers 1:6.
Defaults to \code{1:6} (all plots). The plots correspond to:
\enumerate{
\item \strong{Residuals vs. Observation Indices}: Checks for temporal
patterns, trends, or autocorrelation in residuals across observation order.
\item \strong{Cook's Distance Plot}: Identifies influential observations
that have disproportionate impact on model estimates. Points exceeding
the 4/n threshold warrant investigation.
\item \strong{Generalized Leverage vs. Fitted Values}: Identifies high
leverage points with unusual predictor combinations. Points exceeding
2p/n threshold may be influential.
\item \strong{Residuals vs. Linear Predictor}: Checks for non-linearity
in the predictor-response relationship and heteroscedasticity
(non-constant variance).
\item \strong{Half-Normal Plot with Simulated Envelope}: Assesses
normality of residuals (particularly useful for quantile residuals)
by comparing observed residuals against simulated quantiles. Points
outside the envelope indicate potential model misspecification.
\item \strong{Predicted vs. Observed Values}: Overall goodness-of-fit
check showing model prediction accuracy and systematic bias.
}}

\item{type}{Character string indicating the type of residuals to be used for
plotting. Defaults to \code{"quantile"}. Valid options are:
\itemize{
\item \code{"quantile"}: Randomized quantile residuals (Dunn & Smyth, 1996).
\strong{Recommended for bounded responses} as they should be approximately
N(0,1) if the model is correctly specified. Most interpretable with
standard diagnostic tools.
\item \code{"pearson"}: Pearson residuals (response residual standardized
by estimated standard deviation). Useful for checking the variance function
and identifying heteroscedasticity patterns.
\item \code{"deviance"}: Deviance residuals. Related to the log-likelihood
contribution of each observation. Sum of squared deviance residuals equals
the model deviance.
}}

\item{family}{Character string specifying the distribution family assumptions
to use when calculating residuals and other diagnostics. If \code{NULL}
(default), the family stored within the fitted \code{object} is used.
Specifying a different family can be useful for diagnostic comparisons
across competing model specifications. Available options match those in
\code{\link{gkwreg}}: \code{"gkw"}, \code{"bkw"}, \code{"kkw"},
\code{"ekw"}, \code{"mc"}, \code{"kw"}, \code{"beta"}.}

\item{caption}{Titles for the diagnostic plots. Can be specified in three ways:
\itemize{
\item \code{NULL} (default): Uses standard default captions for all plots.
\item \strong{Character vector} (backward compatibility): A vector of 6
strings corresponding to plots 1-6. Must provide all 6 titles even if
only customizing some.
\item \strong{Named list} (recommended): A list with plot numbers as
names (e.g., \code{list("3" = "My Custom Title")}). Only specified
plots are customized; others use defaults. This allows partial
customization without repeating all titles.
}
Default captions are:
\enumerate{
\item "Residuals vs. Observation Indices"
\item "Cook's Distance Plot"
\item "Generalized Leverage vs. Fitted Values"
\item "Residuals vs. Linear Predictor"
\item "Half-Normal Plot of Residuals"
\item "Predicted vs. Observed Values"
}}

\item{main}{Character string to be prepended to individual plot captions
(from the \code{caption} argument). Useful for adding a common prefix
to all plot titles. Defaults to \code{""} (no prefix).}

\item{sub.caption}{Character string used as a common subtitle positioned
above all plots (especially when multiple plots are arranged). If \code{NULL}
(default), automatically generates a subtitle from the model call
(\code{deparse(x$call)}). Set to \code{""} to suppress the subtitle entirely.}

\item{ask}{Logical. If \code{TRUE} (and using base R graphics with multiple
plots on an interactive device), the user is prompted before displaying
each plot. If \code{NULL} (default), automatically determined: \code{TRUE}
if more plots are requested than fit on the current screen layout and the
session is interactive; \code{FALSE} otherwise. Explicitly set to
\code{FALSE} to disable prompting or \code{TRUE} to force prompting.}

\item{use_ggplot}{Logical. If \code{TRUE}, plots are generated using the
\code{ggplot2} package, providing modern, publication-quality graphics
with extensive theming capabilities. If \code{FALSE} (default), uses base
R graphics, which are faster and require no additional dependencies.
Requires the \code{ggplot2} package to be installed if set to \code{TRUE}.}

\item{arrange_plots}{Logical. Only relevant if \code{use_ggplot = TRUE} and
multiple plots are requested (\code{length(which) > 1}). If \code{TRUE},
attempts to arrange the generated \code{ggplot} objects into a grid layout
using either the \code{gridExtra} or \code{ggpubr} package (requires one
of them to be installed). If \code{FALSE} (default), plots are displayed
individually in sequence. Ignored when using base R graphics.}

\item{nsim}{Integer. Number of simulations used to generate the confidence
envelope in the half-normal plot (\code{which = 5}). Higher values provide
more accurate envelopes but increase computation time. Defaults to 100,
which typically provides adequate precision. Must be a positive integer.
Typical range: 50-500.}

\item{level}{Numeric. The confidence level (between 0 and 1) for the simulated
envelope in the half-normal plot (\code{which = 5}). Defaults to 0.90
(90\\% confidence envelope). Common choices are 0.90, 0.95, or 0.99. Points
falling outside this envelope suggest potential model inadequacy or outliers.}

\item{sample_size}{Integer or \code{NULL}. If specified as an integer less
than the total number of observations (\code{x$nobs}), a random sample of
this size is used for calculating diagnostics and plotting. This can
significantly speed up plot generation for very large datasets (n > 10,000)
with minimal impact on diagnostic interpretation. Defaults to \code{NULL}
(use all observations). Recommended values: 1000-5000 for large datasets.}

\item{theme_fn}{A function. Only relevant if \code{use_ggplot = TRUE}.
Specifies a \code{ggplot2} theme function to apply to all plots for
consistent styling (e.g., \code{ggplot2::theme_bw},
\code{ggplot2::theme_classic}, \code{ggplot2::theme_minimal}). If
\code{NULL} (default), automatically uses \code{ggplot2::theme_minimal}
when \code{use_ggplot = TRUE}. Can also be a custom theme function.
Ignored when using base R graphics.}

\item{save_diagnostics}{Logical. If \code{TRUE}, the function invisibly
returns a list containing all calculated diagnostic measures (residuals,
leverage, Cook's distance, fitted values, etc.) instead of the model object.
Useful for programmatic access to diagnostic values for custom analysis or
reporting. If \code{FALSE} (default), the function invisibly returns the
original model object \code{x}. The function is primarily called for its
side effect of generating plots.}

\item{...}{Additional graphical parameters passed to the underlying plotting
functions. For base R graphics, these are standard \code{par()} parameters
such as \code{col}, \code{pch}, \code{cex}, \code{lwd}, etc. For ggplot2,
these are typically ignored but can be used for specific geom customizations
in advanced usage. Always specified last to follow R best practices.}
}
\value{
Invisibly returns either:
\itemize{
\item The original fitted model object \code{x} (if
\code{save_diagnostics = FALSE}, the default). This allows piping or
chaining operations.
\item A list containing diagnostic measures (if \code{save_diagnostics = TRUE}),
including:
\itemize{
\item \code{data}: Data frame with observation indices, observed values,
fitted values, residuals, Cook's distance, leverage, and linear predictors
\item \code{model_info}: List with model metadata (n, p, thresholds,
family, type, etc.)
\item \code{half_normal}: Data frame with half-normal plot data and
envelope (if \code{which} includes 5)
}
}

The function is primarily called for its side effect of generating diagnostic
plots. The invisible return allows:
\preformatted{
# Silent plotting
plot(model)

# Or capture for further use
diag <- plot(model, save_diagnostics = TRUE)
head(diag$data)
}
}
\description{
Produces a comprehensive set of diagnostic plots for assessing the adequacy
of a fitted Generalized Kumaraswamy (GKw) regression model (objects of class
\code{"gkwreg"}). The function offers flexible plot selection, multiple
residual types, and support for both base R graphics and \code{ggplot2} with
extensive customization options. Designed for thorough model evaluation
including residual analysis, influence diagnostics, and goodness-of-fit
assessment.
}
\details{
Diagnostic plots are essential for evaluating the assumptions and adequacy of
fitted regression models. This function provides a comprehensive suite of
standard diagnostic tools adapted specifically for \code{gkwreg} objects,
which model bounded responses in the (0,1) interval.

\subsection{Residual Types and Interpretation}{
The choice of residual type (\code{type}) is important and depends on the
diagnostic goal:

\itemize{
\item \strong{Quantile Residuals} (\code{type = "quantile"}):
\strong{Recommended as default} for bounded response models. These residuals
are constructed to be approximately N(0,1) under a correctly specified model,
making standard diagnostic tools (QQ-plots, hypothesis tests) directly
applicable. They are particularly effective for detecting model
misspecification in the distributional family or systematic bias.

\item \strong{Pearson Residuals} (\code{type = "pearson"}):
Standardized residuals that account for the mean-variance relationship.
Useful for assessing whether the assumed variance function is appropriate.
If plots show patterns or non-constant spread, this suggests the variance
model may be misspecified.

\item \strong{Deviance Residuals} (\code{type = "deviance"}):
Based on the contribution of each observation to the model deviance.
Often have more symmetric distributions than Pearson residuals and are
useful for identifying observations that fit poorly according to the
likelihood criterion.
}
}

\subsection{Individual Plot Interpretations}{

\strong{Plot 1 - Residuals vs. Observation Indices}:
\itemize{
\item \emph{Purpose}: Detect temporal patterns or autocorrelation
\item \emph{What to look for}: Random scatter around zero. Any systematic
patterns (trends, cycles, clusters) suggest autocorrelation or omitted
time-varying predictors.
\item \emph{Action}: If patterns are detected, consider adding time-related
predictors or modeling autocorrelation structure.
}

\strong{Plot 2 - Cook's Distance}:
\itemize{
\item \emph{Purpose}: Identify influential observations affecting coefficient estimates
\item \emph{What to look for}: Points exceeding the 4/n reference line have
high influence. These observations, if removed, would substantially change
model estimates.
\item \emph{Action}: Investigate high-influence points for data entry errors,
outliers, or legitimately unusual cases. Consider sensitivity analysis.
}

\strong{Plot 3 - Leverage vs. Fitted Values}:
\itemize{
\item \emph{Purpose}: Identify observations with unusual predictor combinations
\item \emph{What to look for}: Points exceeding the 2p/n reference line have
high leverage. These are unusual in predictor space but may or may not be
influential.
\item \emph{Action}: High leverage points deserve scrutiny but are only
problematic if they also have large residuals (check Plots 1, 4).
}

\strong{Plot 4 - Residuals vs. Linear Predictor}:
\itemize{
\item \emph{Purpose}: Detect non-linearity and heteroscedasticity
\item \emph{What to look for}: Random scatter around zero with constant spread.
Curved patterns suggest non-linear relationships. Funnel shapes indicate
heteroscedasticity (non-constant variance).
\item \emph{Action}: For non-linearity, add polynomial terms or use splines.
For heteroscedasticity, consider alternative link functions or variance models.
}

\strong{Plot 5 - Half-Normal Plot with Envelope}:
\itemize{
\item \emph{Purpose}: Assess overall distributional adequacy
\item \emph{What to look for}: Points should follow the reference line and
stay within the simulated envelope. Systematic deviations indicate
distributional misspecification. Isolated points outside the envelope
suggest outliers.
\item \emph{Action}: If many points fall outside the envelope, try a different
distributional family or check for outliers and data quality issues.
}

\strong{Plot 6 - Predicted vs. Observed}:
\itemize{
\item \emph{Purpose}: Overall model fit and prediction accuracy
\item \emph{What to look for}: Points should cluster around the 45-degree line.
Systematic deviations above or below indicate over- or under-prediction.
Large scatter indicates poor predictive performance.
\item \emph{Action}: Poor fit suggests missing predictors, incorrect functional
form, or inappropriate distributional family.
}
}

\subsection{Using Caption Customization}{
The new \strong{named list interface} for \code{caption} allows elegant partial
customization:

\preformatted{
# OLD WAY (still supported): Must repeat all 6 titles
plot(model, caption = c(
  "Residuals vs. Observation Indices",
  "Cook's Distance Plot",
  "MY CUSTOM TITLE FOR PLOT 3",  # Only want to change this
  "Residuals vs. Linear Predictor",
  "Half-Normal Plot of Residuals",
  "Predicted vs. Observed Values"
))

# NEW WAY: Specify only what changes
plot(model, caption = list(
  "3" = "MY CUSTOM TITLE FOR PLOT 3"
))
# Plots 1,2,4,5,6 automatically use defaults

# Customize multiple plots
plot(model, caption = list(
  "1" = "Time Series of Residuals",
  "5" = "Distributional Assessment"
))
}

The vector interface remains fully supported for backward compatibility.
}

\subsection{NULL Defaults and Intelligent Behavior}{
Several arguments default to \code{NULL}, triggering intelligent automatic behavior:

\itemize{
\item \code{sub.caption = NULL}: Automatically generates subtitle from model call
\item \code{ask = NULL}: Automatically prompts only when needed (multiple plots
on interactive device)
\item \code{theme_fn = NULL}: Automatically uses \code{theme_minimal} when
\code{use_ggplot = TRUE}
}

You can override these by explicitly setting values:
\preformatted{
plot(model, sub.caption = "")           # Disable subtitle
plot(model, ask = FALSE)                # Never prompt
plot(model, theme_fn = theme_classic)   # Custom theme
}
}

\subsection{Performance Considerations}{
For large datasets (n > 10,000):
\itemize{
\item Use \code{sample_size} to work with a random subset (e.g.,
\code{sample_size = 2000})
\item Reduce \code{nsim} for half-normal plot (e.g., \code{nsim = 50})
\item Use base R graphics (\code{use_ggplot = FALSE}) for faster rendering
\item Skip computationally intensive plots: \code{which = c(1,2,4,6)}
(excludes half-normal plot)
}
}

\subsection{Graphics Systems}{
\strong{Base R Graphics} (\code{use_ggplot = FALSE}):
\itemize{
\item Faster rendering, especially for large datasets
\item No external dependencies beyond base R
\item Traditional R look and feel
\item Interactive \code{ask} prompting supported
\item Customize via \code{...} parameters (standard \code{par()} settings)
}

\strong{ggplot2 Graphics} (\code{use_ggplot = TRUE}):
\itemize{
\item Modern, publication-quality aesthetics
\item Consistent theming via \code{theme_fn}
\item Grid arrangement support via \code{arrange_plots}
\item Requires \code{ggplot2} package (and optionally \code{gridExtra} or
\code{ggpubr} for arrangements)
\item No interactive \code{ask} prompting (ggplot limitation)
}
}
}
\examples{
\donttest{
# EXAMPLE 1: Basic Usage with Default Settings

# Simulate data
library(gkwdist)

set.seed(123)
n <- 200
x1 <- runif(n, -2, 2)
x2 <- rnorm(n)

# True model parameters
alpha_true <- exp(0.7 + 0.3 * x1)
beta_true <- exp(1.2 - 0.2 * x2)

# Generate response
y <- rkw(n, alpha = alpha_true, beta = beta_true)
df <- data.frame(y = y, x1 = x1, x2 = x2)

# Fit model
model <- gkwreg(y ~ x1 | x2, data = df, family = "kw")

# Generate all diagnostic plots with defaults
par(mfrow = c(3, 2))
plot(model, ask = FALSE)

# EXAMPLE 2: Selective Plots with Custom Residual Type

# Focus on key diagnostic plots only
par(mfrow = c(3, 1))
plot(model,
  which = c(2, 4, 5), # Cook's distance, Resid vs LinPred, Half-normal
  type = "pearson"
) # Use Pearson residuals

# Check for influential points (plot 2) and non-linearity (plot 4)
par(mfrow = c(2, 1))
plot(model,
  which = c(2, 4),
  type = "deviance"
)

# EXAMPLE 3: Caption Customization - New Named List Interface

# Customize only specific plot titles (RECOMMENDED NEW WAY)
par(mfrow = c(3, 1))
plot(model,
  which = c(1, 4, 6),
  caption = list(
    "1" = "Time Pattern Check",
    "4" = "Linearity Assessment",
    "6" = "Predictive Accuracy"
  )
)

# Customize subtitle and main title
par(mfrow = c(2, 1))
plot(model,
  which = c(1, 5),
  main = "Model Diagnostics",
  sub.caption = "Kumaraswamy Regression - Training Data",
  caption = list("5" = "Normality Check with 95\% Envelope")
)

# Suppress subtitle entirely
par(mfrow = c(3, 2))
plot(model, sub.caption = "")

# EXAMPLE 4: Backward Compatible Caption (Vector Interface)

# OLD WAY - still fully supported
par(mfrow = c(3, 2))
plot(model,
  which = 1:6,
  caption = c(
    "Residual Pattern Analysis",
    "Influence Diagnostics",
    "Leverage Assessment",
    "Linearity Check",
    "Distributional Fit",
    "Prediction Quality"
  )
)

# EXAMPLE 5: ggplot2 Graphics with Theming

# Modern publication-quality plots
plot(model,
  use_ggplot = TRUE,
  arrange_plots = TRUE
)

# With custom theme
plot(model,
  use_ggplot = TRUE,
  theme_fn = ggplot2::theme_bw,
  arrange_plots = TRUE
)

# With classic theme and custom colors (via ...)
plot(model,
  use_ggplot = TRUE,
  theme_fn = ggplot2::theme_classic,
  arrange_plots = TRUE
)

# EXAMPLE 6: Arranged Multi-Panel ggplot2 Display

# Requires gridExtra or ggpubr package
plot(model,
  which = 1:4,
  use_ggplot = TRUE,
  arrange_plots = TRUE, # Arrange in grid
  theme_fn = ggplot2::theme_minimal
)

# Focus plots in 2x2 grid
plot(model,
  which = c(2, 3, 4, 6),
  use_ggplot = TRUE,
  arrange_plots = TRUE,
  caption = list(
    "2" = "Influential Cases",
    "3" = "High Leverage Points"
  )
)

# EXAMPLE 7: Half-Normal Plot Customization

# Higher precision envelope (more simulations)
par(mfrow = c(1, 2))
plot(model,
  which = 5,
  nsim = 500, # More accurate envelope
  level = 0.95
) # 95\% confidence level

# Quick envelope for large datasets
plot(model,
  which = 5,
  nsim = 500, # Faster computation
  level = 0.90
)

# EXAMPLE 8: Different Residual Types Comparison

# Compare different residual types
par(mfrow = c(2, 2))
plot(model, which = 4, type = "quantile", main = "Quantile")
plot(model, which = 4, type = "pearson", main = "Pearson")
plot(model, which = 4, type = "deviance", main = "Deviance")
par(mfrow = c(1, 1))

# Quantile residuals for half-normal plot (recommended)
plot(model, which = 5, type = "quantile")

# EXAMPLE 9: Family Comparison Diagnostics

# Compare diagnostics under different distributional assumptions
# Helps assess if alternative family would fit better
par(mfrow = c(2, 2))
plot(model,
  which = c(5, 6),
  family = "kw", # Original family
  main = "Kumaraswamy"
)

plot(model,
  which = c(5, 6),
  family = "beta", # Alternative family
  main = "Beta"
)
par(mfrow = c(1, 1))

# EXAMPLE 10: Large Dataset - Performance Optimization

# Simulate large dataset
set.seed(456)
n_large <- 50000
x1_large <- runif(n_large, -2, 2)
x2_large <- rnorm(n_large)
alpha_large <- exp(0.5 + 0.2 * x1_large)
beta_large <- exp(1.0 - 0.1 * x2_large)
y_large <- rkw(n_large, alpha = alpha_large, beta = beta_large)
df_large <- data.frame(y = y_large, x1 = x1_large, x2 = x2_large)

model_large <- gkwreg(y ~ x1 | x2, data = df_large, family = "kw")

# Optimized plotting for large dataset
par(mfrow = c(2, 2), mar = c(3, 3, 2, 2))
plot(model_large,
  which = c(1, 2, 4, 6), # Skip computationally intensive plot 5
  sample_size = 2000, # Use random sample of 2000 observations
  ask = FALSE
) # Don't prompt

# If half-normal plot needed, reduce simulations
par(mfrow = c(1, 1))
plot(model_large,
  which = 5,
  sample_size = 1000, # Smaller sample
  nsim = 50
) # Fewer simulations

# EXAMPLE 11: Saving Diagnostic Data for Custom Analysis

# Extract diagnostic measures without plotting
par(mfrow = c(1, 1))
diag_data <- plot(model_large,
  which = 1:6,
  save_diagnostics = TRUE
)

# Examine structure
str(diag_data)

# Access diagnostic measures
head(diag_data$data) # Residuals, Cook's distance, leverage, etc.

# Identify influential observations
influential <- which(diag_data$data$cook_dist > diag_data$model_info$cook_threshold)
cat("Influential observations:", head(influential), "\n")

# High leverage points
high_lev <- which(diag_data$data$leverage > diag_data$model_info$leverage_threshold)
cat("High leverage points:", head(high_lev), "\n")

# Custom diagnostic plot using saved data
plot(diag_data$data$fitted, diag_data$data$resid,
  xlab = "Fitted Values", ylab = "Residuals",
  main = "Custom Diagnostic Plot",
  col = ifelse(diag_data$data$cook_dist >
    diag_data$model_info$cook_threshold, "red", "black"),
  pch = 16
)
abline(h = 0, col = "gray", lty = 2)
legend("topright", legend = "Influential", col = "red", pch = 16)

# EXAMPLE 12: Interactive Plotting Control

# ask = TRUE Force prompting between plots (useful for presentations)
# Disable prompting (batch processing)
par(mfrow = c(3, 2))
plot(model,
  which = 1:6,
  ask = FALSE
) # Never prompts

# EXAMPLE 13: Base R Graphics Customization via ...

# Customize point appearance
par(mfrow = c(2, 2))
plot(model,
  which = c(1, 4, 6),
  pch = 16, # Filled circles
  col = "steelblue", # Blue points
  cex = 0.8
) # Smaller points

# Multiple customizations
plot(model,
  which = 2,
  pch = 21, # Circles with border
  col = "black", # Border color
  bg = "lightblue", # Fill color
  cex = 1.2, # Larger points
  lwd = 2
) # Thicker lines

# EXAMPLE 14: Comparing Models

# Fit competing models
model_kw <- gkwreg(y ~ x1 | x2, data = df, family = "kw")
model_beta <- gkwreg(y ~ x1 | x2, data = df, family = "beta")

# Compare diagnostics side-by-side
par(mfrow = c(2, 2))

# Kumaraswamy model
plot(model_kw, which = 5, main = "Kumaraswamy - Half-Normal")
plot(model_kw, which = 6, main = "Kumaraswamy - Pred vs Obs")

# Beta model
plot(model_beta, which = 5, main = "Beta - Half-Normal")
plot(model_beta, which = 6, main = "Beta - Pred vs Obs")

par(mfrow = c(1, 1))
}

}
\references{
Dunn, P. K., & Smyth, G. K. (1996). Randomized Quantile Residuals.
\emph{Journal of Computational and Graphical Statistics}, \strong{5}(3), 236-244.
\doi{10.1080/10618600.1996.10474708}

Cook, R. D. (1977). Detection of Influential Observation in Linear Regression.
\emph{Technometrics}, \strong{19}(1), 15-18.
\doi{10.1080/00401706.1977.10489493}

Atkinson, A. C. (1985). \emph{Plots, Transformations and Regression}.
Oxford University Press.
}
\seealso{
\itemize{
\item \code{\link{gkwreg}} for fitting Generalized Kumaraswamy regression models
\item \code{\link{residuals.gkwreg}} for extracting different types of residuals
\item \code{\link{fitted.gkwreg}} for extracting fitted values
\item \code{\link{summary.gkwreg}} for model summaries
\item \code{\link[stats]{plot.lm}} for analogous diagnostics in linear models
\item \code{\link[ggplot2]{ggplot}} for ggplot2 graphics system
\item \code{\link[gridExtra]{grid.arrange}} for arranging ggplot2 plots
}
}
\author{
Lopes, J. E.
}
\keyword{diagnostics}
\keyword{hplot}
\keyword{methods}
\keyword{plot}
\keyword{regression}
