% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data.R
\name{envelope.gnm}
\alias{envelope.gnm}
\title{Normal QQ-plot with simulated envelope of residuals in Generalized Nonlinear Models}
\usage{
\method{envelope}{gnm}(
  object,
  rep = 25,
  conf = 0.95,
  type = c("quantile", "deviance", "pearson"),
  standardized = FALSE,
  plot.it = TRUE,
  identify,
  ...
)
}
\arguments{
\item{object}{an object of the class \emph{gnm}.}

\item{rep}{an (optional) positive integer which allows to specify the number of replicates which should be used to build the simulated envelope. As default, \code{rep} is set to 25.}

\item{conf}{an (optional) value in the interval (0,1) indicating the confidence level which should be used to build the pointwise confidence intervals, which form the envelope. As default, \code{conf} is set to 0.95.}

\item{type}{a character string indicating the type of residuals which should be used. The available options are: randomized quantile ("quantile"), deviance ("deviance") and pearson ("pearson") residuals. As default, \code{type} is set to "quantile".}

\item{standardized}{an (optional) logical switch indicating if the residuals should be standardized by dividing by the square root of \eqn{(1-h)}, where \eqn{h} is a measure of leverage. As default, \code{standardized} is set to FALSE.}

\item{plot.it}{an (optional) logical switch indicating if the normal QQ-plot with simulated envelope of residuals is required or just the data matrix in which it is based. As default, \code{plot.it} is set to TRUE.}

\item{identify}{an (optional) positive integer indicating the number of individuals to identify on the QQ-plot with simulated envelope of residuals. This is only appropriate if \code{plot.it=TRUE}.}

\item{...}{further arguments passed to or from other methods. If \code{plot.it=TRUE} then \code{...} may be used to include graphical parameters to customize the plot. For example,  \code{col}, \code{pch}, \code{cex}, \code{main}, \code{sub}, \code{xlab}, \code{ylab}.}
}
\value{
A matrix with the following four columns:
\tabular{ll}{
\code{Lower limit} \tab the quantile (1 - \code{conf})/2 of the random sample of size \code{rep} of the \eqn{i}-th order\cr
                   \tab  statistic of the \code{type}-type residuals for \eqn{i=1,2,...,n},\cr
\tab \cr
\code{Median} \tab the quantile 0.5 of the random sample of size \code{rep} of the \eqn{i}-th order\cr
              \tab  statistic of the \code{type}-type residuals for \eqn{i=1,2,...,n},\cr
\tab \cr
\code{Upper limit} \tab the quantile (1 + \code{conf})/2 of the random sample of size \code{rep} of the \eqn{i}-th order\cr
                   \tab  statistic of the \code{type}-type residuals for \eqn{i=1,2,...,n},\cr
\tab \cr
\code{Residuals} \tab the observed \code{type}-type residuals,\cr
}
}
\description{
Produces a normal QQ-plot with simulated envelope of residuals for generalized nonlinear models.
}
\details{
In order to construct the simulated envelope, \code{rep} independent realizations of the response variable for each individual are simulated, which is
done by considering (1) the model assumption about the distribution of the response variable; (2) the estimation of the "linear" predictor parameters; and (3)
the estimation of the dispersion parameter. Each time, the vector of observed responses is replaced with one of the simulated samples, re-fitting the interest
model \code{rep} times. For each \eqn{i=1,2,...,n}, where \eqn{n} is the number of individuals in the sample, the \eqn{i}-th order statistic of the
\code{type}-type residuals is computed and then sorted for each replicate, giving a random sample of size \code{rep} of the \eqn{i}-th order statistic. In
other words, the simulated envelope is comprised of the quantiles (1 - \code{conf})/2 and (1 + \code{conf})/2 of the random sample of size \code{rep} of the
\eqn{i}-th order statistic of the \code{type}-type residuals for \eqn{i=1,2,...,n}.
}
\examples{
###### Example 1: The effects of fertilizers on coastal Bermuda grass
data(Grass)
fit1 <- gnm(Yield ~ b0 + b1/(Nitrogen + a1) + b2/(Phosphorus + a2) + b3/(Potassium + a3),
            family=gaussian(inverse), start=c(b0=0.1,b1=13,b2=1,b3=1,a1=45,a2=15,a3=30), data=Grass)

#envelope(fit1, rep=50, conf=0.95, type="quantile", col="red", pch=20, col.lab="blue",
#         col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 2: Assay of an Insecticide with a Synergist
data(Melanopus)
fit2 <- gnm(Killed/Exposed ~ b0 + b1*log(Insecticide-a1) + b2*Synergist/(a2 + Synergist),
            family=binomial(logit), weights=Exposed, start=c(b0=-3,b1=1.2,a1=1.7,b2=1.7,a2=2),
		   data=Melanopus)

#envelope(fit2, rep=50, conf=0.95, type="pearson", col="red", pch=20, col.lab="blue",
#         col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 3: Developmental rate of Drosophila melanogaster
data(Drosophila)
fit3 <- gnm(Duration ~ b0 + b1*Temp + b2/(Temp-a), family=Gamma(log),
            start=c(b0=3,b1=-0.25,b2=-210,a=55), weights=Size, data=Drosophila)

#envelope(fit3, rep=50, conf=0.95, type="quantile", col="red", pch=20, col.lab="blue",
#         col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 4: Radioimmunological Assay of Cortisol
data(Cortisol)
fit4 <- gnm(Y ~ b0 + (b1-b0)/(1 + exp(b2+ b3*lDose))^b4, family=Gamma(identity),
            start=c(b0=130,b1=2800,b2=3,b3=3,b4=0.5), data=Cortisol)

#envelope(fit4, rep=50, conf=0.95, type="quantile", col="red", pch=20, col.lab="blue",
#         col.axis="blue", col.main="black", family="mono", cex=0.8)

}
\references{
Atkinson A.C. (1985) \emph{Plots, Transformations and Regression}. Oxford University Press, Oxford.

Davison A.C., Gigli A. (1989) Deviance Residuals and Normal Scores Plots. \emph{Biometrika} 76, 211-221.

Dunn P.K., Smyth G.K. (1996) Randomized Quantile Residuals. \emph{Journal of Computational and Graphical Statistics} 5, 236-244.

Pierce D.A., Schafer D.W. (1986) Residuals in Generalized Linear Models. \emph{Journal of the American Statistical Association} 81, 977-986.

Wei B.C. (1998). \emph{Exponential Family Nonlinear Models}. Springer, Singapore.
}
\seealso{
\link{envelope.lm}, \link{envelope.overglm}
}
