% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/MAINest.R
\name{fitGSMVAR}
\alias{fitGSMVAR}
\title{Two-phase maximum likelihood estimation of a GMVAR, StMVAR, or G-StMVAR model}
\usage{
fitGSMVAR(
  data,
  p,
  M,
  model = c("GMVAR", "StMVAR", "G-StMVAR"),
  conditional = TRUE,
  parametrization = c("intercept", "mean"),
  constraints = NULL,
  same_means = NULL,
  weight_constraints = NULL,
  structural_pars = NULL,
  ncalls = (M + 1)^5,
  ncores = 2,
  maxit = 1000,
  seeds = NULL,
  print_res = TRUE,
  use_parallel = TRUE,
  filter_estimates = TRUE,
  calc_std_errors = TRUE,
  ...
)
}
\arguments{
\item{data}{a matrix or class \code{'ts'} object with \code{d>1} columns. Each column is taken to represent
a univariate time series. \code{NA} values are not supported.}

\item{p}{a positive integer specifying the autoregressive order of the model.}

\item{M}{\describe{
  \item{For \strong{GMVAR} and \strong{StMVAR} models:}{a positive integer specifying the number of mixture components.}
  \item{For \strong{G-StMVAR} models:}{a size \eqn{(2\times 1)} integer vector specifying the number of \emph{GMVAR type} components \code{M1}
   in the first element and \emph{StMVAR type} components \code{M2} in the second element. The total number of mixture components
   is \code{M=M1+M2}.}
}}

\item{model}{is "GMVAR", "StMVAR", or "G-StMVAR" model considered? In the G-StMVAR model, the first \code{M1} components
are GMVAR type and the rest \code{M2} components are StMVAR type.}

\item{conditional}{a logical argument specifying whether the conditional or exact log-likelihood function}

\item{parametrization}{\code{"intercept"} or \code{"mean"} determining whether the model is parametrized with intercept
parameters \eqn{\phi_{m,0}} or regime means \eqn{\mu_{m}}, m=1,...,M.}

\item{constraints}{a size \eqn{(Mpd^2 \times  q)} constraint matrix \strong{\eqn{C}} specifying general linear constraints
to the autoregressive parameters. We consider constraints of form
(\strong{\eqn{\phi}}\eqn{_{1}}\eqn{,...,}\strong{\eqn{\phi}}\eqn{_{M}) = }\strong{\eqn{C \psi}},
where \strong{\eqn{\phi}}\eqn{_{m}}\eqn{ = (vec(A_{m,1}),...,vec(A_{m,p}) (pd^2 x 1), m=1,...,M},
contains the coefficient matrices and \strong{\eqn{\psi}} \eqn{(q x 1)} contains the related parameters.
For example, to restrict the AR-parameters to be the same for all regimes, set \strong{\eqn{C}}=
[\code{I:...:I}]\strong{'} \eqn{(Mpd^2 \times  pd^2)} where \code{I = diag(p*d^2)}.
Ignore (or set to \code{NULL}) if linear constraints should \strong{not} be employed.}

\item{same_means}{Restrict the mean parameters of some regimes to be the same? Provide a list of numeric vectors
such that each numeric vector contains the regimes that should share the common mean parameters. For instance, if
\code{M=3}, the argument \code{list(1, 2:3)} restricts the mean parameters of the second and third regime to be
the same but the first regime has freely estimated (unconditional) mean. Ignore or set to \code{NULL} if mean parameters
should not be restricted to be the same among any regimes. \strong{This constraint is available only for mean parametrized models;
that is, when \code{parametrization="mean"}.}}

\item{weight_constraints}{a numeric vector of length \eqn{M-1} specifying fixed parameter values for the mixing weight parameters
\eqn{\alpha_m, \ m=1,...,M-1}. Each element should be strictly between zero and one, and the sum of all the elements should
be strictly less than one.}

\item{structural_pars}{If \code{NULL} a reduced form model is considered. Reduced models can be used directly as recursively
identified structural models. For a structural model identified by conditional heteroskedasticity, should be a list containing
at least the first one of the following elements:
\itemize{
  \item \code{W} - a \eqn{(dxd)} matrix with its entries imposing constraints on \eqn{W}: \code{NA} indicating that the element is
    unconstrained, a positive value indicating strict positive sign constraint, a negative value indicating strict
    negative sign constraint, and zero indicating that the element is constrained to zero.
  \item \code{C_lambda} - a \eqn{(d(M-1) \times  r)} constraint matrix that satisfies (\strong{\eqn{\lambda}}\eqn{_{2}}\eqn{,...,}
    \strong{\eqn{\lambda}}\eqn{_{M}) =} \strong{\eqn{C_{\lambda} \gamma}} where \strong{\eqn{\gamma}} is the new \eqn{(r \times 1)}
    parameter subject to which the model is estimated (similarly to AR parameter constraints). The entries of \code{C_lambda}
    must be either \strong{positive} or \strong{zero}. Ignore (or set to \code{NULL}) if the eigenvalues \eqn{\lambda_{mi}}
    should not be constrained.
  \item \code{fixed_lambdas} - a length \eqn{d(M-1)} numeric vector (\strong{\eqn{\lambda}}\eqn{_{2}}\eqn{,...,}
    \strong{\eqn{\lambda}}\eqn{_{M})} with elements strictly larger than zero specifying the fixed parameter values for the
    parameters \eqn{\lambda_{mi}} should be constrained to. This constraint is alternative \code{C_lambda}.
    Ignore (or set to \code{NULL}) if the eigenvalues \eqn{\lambda_{mi}} should not be constrained.
}
See Virolainen (forthcoming) for the conditions required to identify the shocks and for the B-matrix as well (it is \eqn{W} times
a time-varying diagonal matrix with positive diagonal entries).}

\item{ncalls}{the number of estimation rounds that should be performed.}

\item{ncores}{the number CPU cores to be used in parallel computing.}

\item{maxit}{the maximum number of iterations in the variable metric algorithm.}

\item{seeds}{a length \code{ncalls} vector containing the random number generator seed for each call to the genetic algorithm,
or \code{NULL} for not initializing the seed. Exists for creating reproducible results.}

\item{print_res}{should summaries of estimation results be printed?}

\item{use_parallel}{employ parallel computing? If \code{FALSE}, no progression bar etc will be generated.}

\item{filter_estimates}{should the likely inappropriate estimates be filtered? See details.}

\item{calc_std_errors}{calculate approximate standard errors for the estimates?}

\item{...}{additional settings passed to the function \code{GAfit} employing the genetic algorithm.}
}
\value{
Returns an object of class \code{'gsmvar'} defining the estimated (reduced form or structural) GMVAR, StMVAR, or G-StMVAR model.
  Multivariate quantile residuals (Kalliovirta and Saikkonen 2010) are also computed and included in the returned object.
  In addition, the returned object contains the estimates and log-likelihood values from all the estimation rounds performed.
  The estimated parameter vector can be obtained at \code{gsmvar$params} (and corresponding approximate standard errors
  at \code{gsmvar$std_errors}). See \code{?GSMVAR} for the form of the parameter vector, if needed.

  Remark that the first autocovariance/correlation matrix in \code{$uncond_moments} is for the lag zero,
  the second one for the lag one, etc.
}
\description{
\code{fitGSMVAR} estimates a GMVAR, StMVAR, or G-StMVAR model model in two phases:
  in the first phase it uses a genetic algorithm to find starting values for a gradient based
  variable metric algorithm, which it then uses to finalize the estimation in the second phase.
  Parallel computing is utilized to perform multiple rounds of estimations in parallel.
}
\details{
If you wish to estimate a structural model without overidentifying constraints that is identified statistically,
 specify your W matrix is \code{structural_pars} to be such that it contains the same sign constraints in a single row
 (e.g. a row of ones) and leave the other elements as \code{NA}. In this way, the genetic algorithm works the best.
 The ordering and signs of the columns of the W matrix can be changed afterwards with the functions
 \code{reorder_W_columns} and \code{swap_W_signs}.

 Because of complexity and high multimodality of the log-likelihood function, it's \strong{not certain} that the estimation
 algorithms will end up in the global maximum point. It's expected that most of the estimation rounds will end up in
 some local maximum or saddle point instead. Therefore, a (sometimes large) number of estimation rounds is required
 for reliable results. Because of the nature of the model, the estimation may fail especially in the cases where the
 number of mixture components is chosen too large. \strong{With two regimes and couple hundred observations in a two-dimensional
 time series, 50 rounds is usually enough. Several hundred estimation rounds often suffices for reliably fitting two-regimes
 models to 3 or 4 dimensional time series. With more than two regimes and more than couple hundred
 observations, thousands of estimation rounds (or more) are often required to obtain reliable results.}

 The estimation process is computationally heavy and it might take considerably long time for large models with
 large number of observations. If the iteration limit \code{maxit} in the variable metric algorithm is reached,
 one can continue the estimation by iterating more with the function \code{iterate_more}. Alternatively, one may
 use the found estimates as starting values for the genetic algorithm and and employ another round of estimation
 (see \code{?GAfit} how to set up an initial population with the dot parameters).

 \strong{If the estimation algorithm fails to create an initial population for the genetic algorithm},
 it usually helps to scale the individual series so that the AR coefficients (of a VAR model) will be
 relative small, preferably less than one. Even if one is able to create an initial population, it should
 be preferred to scale the series so that most of the AR coefficients will not be very large, as the
 estimation algorithm works better with relatively small AR coefficients. If needed, another package can be used
 to fit linear VARs to the series to see which scaling of the series results in relatively small AR coefficients.

 The code of the genetic algorithm is mostly based on the description by \emph{Dorsey and Mayer (1995)} but it
 includes some extra features that were found useful for this particular estimation problem. For instance,
 the genetic algorithm uses a slightly modified version of the individually adaptive crossover and mutation
 rates described by \emph{Patnaik and Srinivas (1994)} and employs (50\%) fitness inheritance discussed
 by \emph{Smith, Dike and Stegmann (1995)}.

 The gradient based variable metric algorithm used in the second phase is implemented with function \code{optim}
 from the package \code{stats}.

 Note that the structural models are even more difficult to estimate than the reduced form models due to
 the different parametrization of the covariance matrices, so larger number of estimation rounds should be considered.
 Also, be aware that if the lambda parameters are constrained in any other way than by restricting some of them to be
 identical, the parameter "lambda_scale" of the genetic algorithm (see \code{?GAfit}) needs to be carefully adjusted accordingly.
 \strong{When estimating a structural model that imposes overidentifiying constraints to a time series with \eqn{d>3},
 it is highly recommended to create an initial population based on the estimates of a statistically identified model
 (when \eqn{M=2}). This is because currently obtaining the ML estimate reliably to such a structural model seems
 difficult in many application.}

 Finally, the function fails to calculate approximate standard errors and the parameter estimates are near the border
 of the parameter space, it might help to use smaller numerical tolerance for the stationarity and positive
 definiteness conditions. The numerical tolerance of an existing model can be changed with the function
 \code{update_numtols}.

 \strong{Filtering inappropriate estimates:} If \code{filter_estimates == TRUE}, the function will automatically filter
 out estimates that it deems "inappropriate". That is, estimates that are not likely solutions of interest.
 Specifically, solutions that incorporate a near-singular error term covariance matrix (any eigenvalue less than \eqn{0.002}),
 mixing weights such that they are close to zero for almost all \eqn{t} for at least one regime, or mixing weight parameter
 estimate close to zero (or one). It also filters out estimates with any modulus "bold A" eigenvalues larger than 0.9985,
as the solution is near the boundary of the stationarity region and likely not a local maximum. You can also set
 \code{filter_estimates=FALSE} and find the solutions of interest yourself by using the
 function \code{alt_gsmvar}.
}
\section{S3 methods}{

  The following S3 methods are supported for class \code{'gsmvar'}: \code{logLik}, \code{residuals}, \code{print}, \code{summary},
   \code{predict}, \code{simulate}, and \code{plot}.
}

\examples{
\donttest{
## These are long running examples that use parallel computing!
# Running all the below examples will take approximately 3-4 minutes.

# GMVAR(1,2) model: 10 estimation rounds with seeds set
# for reproducibility
fit12 <- fitGSMVAR(gdpdef, p=1, M=2, ncalls=10, seeds=1:10)
fit12
plot(fit12)
summary(fit12)
print_std_errors(fit12)
profile_logliks(fit12)

# The rest of the examples only use a single estimation round with a given
# seed that produces the MLE to reduce running time of the examples. When
# estimating models for empirical applications, a large number of estimation
# rounds (ncalls = a large number) should be performed to ensure reliability
# of the estimates (see the section details).

# StMVAR(2, 2) model
fit22t <- fitGSMVAR(gdpdef, p=2, M=2, model="StMVAR", ncalls=1, seeds=1)
fit22t # Overly large degrees of freedom estimate in the 2nd regime!
fit22gs <- stmvar_to_gstmvar(fit22t) # So switch it to GMVAR type!
fit22gs # This is the appropriate G-StMVAR model based on the above StMVAR model.
fit22gss <- gsmvar_to_sgsmvar(fit22gs) # Switch to structural model
fit22gss # This is the implied statistically identified structural model.

# Structural GMVAR(1,2) model identified with sign
# constraints.
W_122 <- matrix(c(1, 1, -1, 1), nrow=2)
fit12s <- fitGSMVAR(gdpdef, p=1, M=2, structural_pars=list(W=W_122),
  ncalls=1, seeds=1)
fit12s
# A statistically identified structural model can also be obtained with
# gsmvar_to_sgsmvar(fit12)


# GMVAR(2,2) model with autoregressive parameters restricted
# to be the same for both regimes
C_mat <- rbind(diag(2*2^2), diag(2*2^2))
fit22c <- fitGSMVAR(gdpdef, p=2, M=2, constraints=C_mat, ncalls=1, seeds=1)
fit22c

# G-StMVAR(2, 1, 1) model with autoregressive parameters and unconditional means restricted
# to be the same for both regimes:
fit22gscm <- fitGSMVAR(gdpdef, p=2, M=c(1, 1), model="G-StMVAR", constraints=C_mat,
   parametrization="mean", same_means=list(1:2), ncalls=1, seeds=1)

# GMVAR(2,2) model with autoregressive parameters restricted
# to be the same for both regimes and non-diagonal elements
# the coefficient matrices constrained to zero.
tmp <- matrix(c(1, rep(0, 10), 1, rep(0, 8), 1, rep(0, 10), 1),
 nrow=2*2^2, byrow=FALSE)
C_mat2 <- rbind(tmp, tmp)
fit22c2 <- fitGSMVAR(gdpdef, p=2, M=2, constraints=C_mat2, ncalls=1,
  seeds=1)
fit22c2
}
}
\references{
\itemize{
   \item Dorsey R. E. and Mayer W. J. 1995. Genetic algorithms for estimation problems with multiple optima,
         nondifferentiability, and other irregular features. \emph{Journal of Business & Economic Statistics},
        \strong{13}, 53-66.
   \item Kalliovirta L., Meitz M. and Saikkonen P. 2016. Gaussian mixture vector autoregression.
         \emph{Journal of Econometrics}, \strong{192}, 485-498.
   \item Patnaik L.M. and Srinivas M. 1994. Adaptive Probabilities of Crossover and Mutation in Genetic Algorithms.
         \emph{Transactions on Systems, Man and Cybernetics} \strong{24}, 656-667.
   \item Smith R.E., Dike B.A., Stegmann S.A. 1995. Fitness inheritance in genetic algorithms.
         \emph{Proceedings of the 1995 ACM Symposium on Applied Computing}, 345-350.
   \item Virolainen S. 2025. A statistically identified structural vector autoregression with endogenously
         switching volatility regime. \emph{Journal of Business & Economic Statistics}, \strong{43}, 1, 44-54.
   \item Virolainen S. 2022. Gaussian and Student's t mixture vector autoregressive model with application to the
     asymmetric effects of monetary policy shocks in the Euro area. Unpublished working
     paper, available as arXiv:2109.13648.
 }
}
\seealso{
\code{\link{GSMVAR}}, \code{\link{iterate_more}}, \code{\link{stmvar_to_gstmvar}}, \code{\link{predict.gsmvar}},
  \code{\link{profile_logliks}}, \code{\link{simulate.gsmvar}}, \code{\link{quantile_residual_tests}}, \code{\link{print_std_errors}},
  \code{\link{swap_parametrization}}, \code{\link{get_gradient}}, \code{\link{GIRF}}, \code{\link{GFEVD}}, \code{\link{LR_test}},
  \code{\link{Wald_test}}, \code{\link{gsmvar_to_sgsmvar}}, \code{\link{stmvar_to_gstmvar}}, \code{\link{reorder_W_columns}},
   \code{\link{swap_W_signs}}, \code{\link{cond_moment_plot}}, \code{\link{update_numtols}}
}
