% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/GMWM.R
\name{gmwm}
\alias{gmwm}
\title{GMWM for Sensors, ARMA, SSM, and Robust}
\usage{
gmwm(model, data, model.type = "ssm", compute.v = "auto", robust = FALSE,
  eff = 0.6, alpha = 0.05, seed = 1337, G = NULL, K = 1, H = 100,
  freq = NULL)
}
\arguments{
\item{model}{A \code{ts.model} object containing one of the allowed models.}

\item{data}{A \code{matrix} or \code{data.frame} object with only column (e.g. \eqn{N \times 1}{ N x 1 }), or a \code{lts} object, or a \code{gts} object.}

\item{model.type}{A \code{string} containing the type of GMWM needed e.g. sensor or SSM}

\item{compute.v}{A \code{string} indicating the type of covariance matrix solver. "fast", "bootstrap", "asymp.diag", "asymp.comp", "fft"}

\item{robust}{A \code{boolean} indicating whether to use the robust computation (TRUE) or not (FALSE).}

\item{eff}{A \code{double} between 0 and 1 that indicates the efficiency.}

\item{alpha}{A \code{double} between 0 and 1 that correspondings to the \eqn{\frac{\alpha}{2}}{alpha/2} value for the wavelet confidence intervals.}

\item{seed}{A \code{integer} that controls the reproducibility of the auto model selection phase.}

\item{G}{An \code{integer} to sample the space for sensor and SSM models to ensure optimal identitability.}

\item{K}{An \code{integer} that controls how many times the bootstrapping procedure will be initiated.}

\item{H}{An \code{integer} that indicates how many different samples the bootstrap will be collect.}

\item{freq}{A \code{double} that indicates the time between frequency.}
}
\value{
A \code{gmwm} object with the structure: 
\describe{
 \item{estimate}{Estimated Parameters Values from the GMWM Procedure}
 \item{init.guess}{Initial Starting Values given to the Optimization Algorithm}
 \item{wv.empir}{The data's empirical wavelet variance}
 \item{ci.low}{Lower Confidence Interval}
 \item{ci.high}{Upper Confidence Interval}
 \item{orgV}{Original V matrix}
 \item{V}{Updated V matrix (if bootstrapped)}
 \item{omega}{The V matrix inversed}
 \item{obj.fun}{Value of the objective function at Estimated Parameter Values}
 \item{theo}{Summed Theoretical Wavelet Variance}
 \item{decomp.theo}{Decomposed Theoretical Wavelet Variance by Process}
 \item{scales}{Scales of the GMWM Object}
 \item{robust}{Indicates if parameter estimation was done under robust or classical}
 \item{eff}{Level of efficiency of robust estimation}
 \item{model.type}{Models being guessed}
 \item{compute.v}{Type of V matrix computation}
 \item{augmented}{Indicates moments have been augmented}
 \item{alpha}{Alpha level used to generate confidence intervals}
 \item{expect.diff}{Mean of the First Difference of the Signal}
 \item{N}{Length of the Signal}
 \item{G}{Number of Guesses Performed}
 \item{H}{Number of Bootstrap replications}
 \item{K}{Number of V matrix bootstraps}
 \item{model}{\code{ts.model} supplied to gmwm}
 \item{model.hat}{A new value of \code{ts.model} object supplied to gmwm}
 \item{starting}{Indicates whether the procedure used the initial guessing approach}
 \item{seed}{Randomization seed used to generate the guessing values}
 \item{freq}{Frequency of data}
}
}
\description{
GMM object
}
\details{
This function is under work. Some of the features are active. Others... Not so much. 

The V matrix is calculated by:
\eqn{diag\left[ {{{\left( {Hi - Lo} \right)}^2}} \right]}{diag[(Hi-Lo)^2]}.

The function is implemented in the following manner:
1. Calculate MODWT of data with levels = floor(log2(data))
2. Apply the brick.wall of the MODWT (e.g. remove boundary values)
3. Compute the empirical wavelet variance (WV Empirical).
4. Obtain the V matrix by squaring the difference of the WV Empirical's Chi-squared confidence interval (hi - lo)^2
5. Optimize the values to obtain \eqn{\hat{\theta}}{theta^hat}
6. If FAST = TRUE, return these results. Else, continue.

Loop  k = 1 to K
Loop h = 1 to H
7. Simulate xt under \eqn{F_{\hat{\theta}}}{F_theta^hat}
8. Compute WV Empirical
END
9. Calculate the covariance matrix
10. Optimize the values to obtain \eqn{\hat{\theta}}{theta^hat}
END
11. Return optimized values.


The function estimates a variety of time series models. If type = "imu" or "ssm", then
parameter vector should indicate the characters of the models that compose the latent or state-space model. The model
options are:
\describe{
  \item{"AR1"}{a first order autoregressive process with parameters \eqn{(\phi,\sigma^2)}{phi, sigma^2}}
  \item{"GM"}{a guass-markov process \eqn{(\beta,\sigma_{gm}^2)}{beta, sigma[gm]^2}}
  \item{"ARMA"}{an autoregressive moving average process with parameters \eqn{(\phi _p, \theta _q, \sigma^2)}{phi[p], theta[q], sigma^2}}
  \item{"DR"}{a drift with parameter \eqn{\omega}{omega}}
  \item{"QN"}{a quantization noise process with parameter \eqn{Q}}
  \item{"RW"}{a random walk process with parameter \eqn{\sigma^2}{sigma^2}}
  \item{"WN"}{a white noise process with parameter \eqn{\sigma^2}{sigma^2}}
}
If only an ARMA() term is supplied, then the function takes conditional least squares as starting values
If robust = TRUE the function takes the robust estimate of the wavelet variance to be used in the GMWM estimation procedure.
}
\examples{
# AR
set.seed(1336)
n = 200
data = gen.gts(AR1(phi = .99, sigma2 = 0.01) + WN(sigma2 = 1), n)

# Models can contain specific parameters e.g.
adv.model = gmwm(AR1(phi = .99, sigma2 = 0.01) + WN(sigma2 = 0.01),
                            data)

# Or we can guess the parameters:
guided.model = gmwm(AR1() + WN(), data) 

# Want to try different models? 
guided.ar1 = gmwm(AR1(), data)

# Faster:
guided.ar1.wn.prev = update(guided.ar1, AR1()+WN())

# OR 

# Create new GMWM object. 
# Note this is SLOWER since the Covariance Matrix is recalculated.
guided.ar1.wn.new = gmwm(AR1()+WN(), data)
 
# ARMA case
set.seed(1336)
data = gen.gts(ARMA(ar = c(0.8897, -0.4858), ma = c(-0.2279, 0.2488),
              sigma2 = 0.1796), 200)
#guided.arma = gmwm(ARMA(2,2), data, model.type="ssm")
adv.arma = gmwm(ARMA(ar=c(0.8897, -0.4858), ma = c(-0.2279, 0.2488), sigma2=0.1796),
                data, model.type="ssm")
}

