% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gorica_methods.R
\name{gorica}
\alias{gorica}
\alias{gorica.lavaan}
\alias{gorica.table}
\title{Evaluate informative hypotheses using the GORICA}
\usage{
gorica(x, hypothesis, comparison = "unconstrained", iterations = 1e+05, ...)

\method{gorica}{lavaan}(
  x,
  hypothesis,
  comparison = "unconstrained",
  iterations = 1e+05,
  ...,
  standardize = FALSE
)

\method{gorica}{table}(x, hypothesis, comparison = "unconstrained", ...)
}
\arguments{
\item{x}{An R object containing the outcome of a statistical analysis.
Currently, the following objects can be processed:
\itemize{
\item \code{lm()} objects (anova, ancova, multiple regression).
\item \code{t_test()} objects.
\item \code{lavaan} objects.
\item \code{lmerMod} objects.
\item A named vector containing the estimates resulting from a statistical
analysis, when the argument \code{Sigma} is also specified.
Note that, named means that each estimate has to be labeled such that it can
be referred to in \code{hypotheses}.
}}

\item{hypothesis}{A character string containing the informative hypotheses to
evaluate (see Details).}

\item{comparison}{A character string indicating what the \code{hypothesis}
should be compared to. Defaults to \code{comparison = "unconstrained"};
options include \code{c("unconstrained", "complement", "none")}.}

\item{iterations}{Integer. Number of samples to draw from the parameter space
when computing the \code{gorica} penalty.}

\item{...}{Additional arguments passed to the internal function
\code{compare_hypotheses}.}

\item{standardize}{Logical. For \code{lavaan} objects, whether or not to
extract the standardized model coefficients. Defaults to \code{FALSE}.}
}
\value{
An object of class \code{gorica}, containing the following elements:
\itemize{
\item \code{fit}  A \code{data.frame} containing the loglikelihood, penalty
(for complexity), the GORICA value, and the GORICA weights. The GORICA
weights are calculated by taking into account the misfits and complexities of
the hypotheses under evaluation. These weights are used to quantify the
support in the data for each hypothesis under evaluation. By looking at the
pairwise ratios between the GORICA weights, one can determine the relative
importance of one hypothesis over another hypothesis.
\item \code{call}  The original function call.
\item \code{model}  The original model object (\code{x}).
\item \code{estimates}  The parameters extracted from the \code{model}.
\item \code{Sigma}  The asymptotic covariance matrix of the
\code{estimates}.
\item \code{comparison}  Which alternative hypothesis was used.
\item \code{hypotheses}  The hypotheses evaluated in \code{fit}.
\item \code{relative_weights}  The relative weights of each hypothesis (rows)
versus each other hypothesis in the set (cols). The diagonal is equal to one,
as each hypothesis is equally likely as itself. A value of, e.g., 6, means
that the hypothesis in the row is 6 times more likely than the hypothesis in
the column.
}
}
\description{
GORICA is an acronym for "generalized order-restricted information criterion
approximation". It can be utilized to evaluate informative hypotheses, which
specify directional relationships between model parameters in terms of
(in)equality constraints.
}
\details{
The GORICA is applicable to not only normal linear models, but also applicable to generalized linear models (GLMs) (McCullagh & Nelder, 1989), generalized linear
mixed models (GLMMs) (McCullogh & Searle, 2001), and structural equation
models (SEMs) (Bollen, 1989). In addition, the GORICA can be utilized in the context of contingency tables for which (in)equality constrained hypotheses do not necessarily contain linear restrictions on cell probabilities, but instead often contain non-linear restrictions on cell probabilities.

\code{hypotheses} is a character string that specifies which informative
hypotheses have to be evaluated. A simple example is \code{hypotheses <- "a >
b > c; a = b = c;"} which specifies two hypotheses using three estimates with
names "a", "b", and "c", respectively.

The hypotheses specified have to adhere to the following rules:
\enumerate{
\item Parameters are referred to using the names specified in \code{names()}.
\item Linear combinations of parameters must be specified adhering to the
following rules:
        \enumerate{ \item Each parameter name is used at most once.
                    \item Each parameter name may or may not be
                    pre-multiplied with a number.
                    \item A constant may be added or subtracted from each
                    parameter name.
                    \item A linear combination can also be a single number.}
Examples are: \code{3 * a + 5}; \code{a + 2 * b + 3 * c - 2}; \code{a - b};
and \code{5}.
\item (Linear combinations of) parameters can be constrained using <, >, and
=. For example, \code{a > 0} or
\code{a > b = 0} or \code{2 * a < b + c > 5}.
\item The ampersand & can be used to combine different parts of a hypothesis.
For example, \code{a > b & b > c} which is equivalent to \code{a > b > c} or
\code{a > 0 & b > 0 & c > 0}.
\item Sets of (linear combinations of) parameters subjected to the same
constraints can be specified using (). For
example, \code{a > (b,c)} which is equivalent to \code{a > b & a > c}.
\item The specification of a hypothesis is completed by typing ; For example,
\code{hypotheses <- "a > b > c; a = b = c;"}, specifies two hypotheses.
\item Hypotheses have to be compatible, non-redundant and possible. What
these terms mean will be elaborated below.
}

\emph{The set of hypotheses has to be compatible}. For the statistical
background of this requirement see Gu, Mulder, Hoijtink (2018). Usually the
sets of hypotheses specified by researchers are compatible, and if not,
\code{gorica} will return an error message. The following steps can be used to
determine if a set of hypotheses is compatible:
\enumerate{
\item	Replace a range constraint, e.g., \code{1 < a1 < 3}, by an equality
constraint in which the parameter involved is equated to the midpoint of the
range, that is, \code{a1 = 2}.
\item Replace in each hypothesis the < and > by =. For example, \code{a1 = a2
> a3 > a4} becomes \code{a1 = a2 = a3 = a4}.
\item The hypotheses are compatible if there is at least one solution to the
resulting set of equations. For the two hypotheses considered under 1. and
2., the solution is a1 = a2 = a3 = a4 = 2. An example of two non-compatible
hypotheses is \code{hypotheses <- "a = 0; a > 2;"} because there is no
solution to the equations \code{a=0} and \code{a=2}.
}

\emph{Each hypothesis in a set of hypotheses has to be non-redundant.} A
hypothesis is redundant if it can also be specified with fewer constraints.
For example, \code{a = b & a > 0 & b > 0} is redundant because it can also be
specified as \code{a = b & a > 0}. \code{gorica} will work correctly if
hypotheses specified using only < and > are redundant. \code{gorica} will
return an error message if hypotheses specified using at least one = are
redundant.

\emph{Each hypothesis in a set of hypotheses has to be possible.} An
hypothesis is impossible if estimates in agreement with the hypothesis do not
exist. For example: values for \code{a} in agreement with \code{a = 0 &
a > 2} do not exist. It is the responsibility of the user to ensure that the
hypotheses specified are possible. If not, \code{gorica} will either return an
error message or render an output table containing \code{Inf}'s.
}
\section{Contingency tables}{

When specifying hypotheses about contingency tables, the asymptotic
covariance matrix of the model estimates is derived by means of
bootstrapping. This makes it possible for users to define derived parameters:
For example, a ratio between cell probabilities. For this purpose, the
\code{\link[bain]{bain}} syntax has been enhanced with the command \code{:=}.
Thus, the syntax \code{"a := x[1,1]/(x[1,1]+x[1,2])"} defines a new parameter
\code{a} by reference to specific cells of the table \code{x}. This new
parameter can now be named in hypotheses.
}

\examples{
\dontshow{
# EXAMPLE 1. One-sample t test
ttest1 <- t_test(iris$Sepal.Length,mu=5)
gorica(ttest1,"x<5.8", iterations = 5)

# EXAMPLE 2. ANOVA
aov1 <- aov(yield ~ block-1 + N * P + K, npk)
gorica(aov1,hypothesis="block1=block5;
   K1<0", iterations = 5)

# EXAMPLE 3. glm
counts <- c(18,17,15,20,10,20,25,13,12)
outcome <- gl(3,1,9)
treatment <- gl(3,3)
fit <- glm(counts ~ outcome-1 + treatment, family = poisson())
gorica(fit, "outcome1 > (outcome2, outcome3)", iterations = 5)

# EXAMPLE 4. ANOVA
res <- lm(Sepal.Length ~ Species-1, iris)
est <- get_estimates(res)
est
gor <- gorica(res, "Speciessetosa < (Speciesversicolor, Speciesvirginica)",
comparison = "complement", iterations = 5)
gor
}
\donttest{
# EXAMPLE 1. One-sample t test
ttest1 <- t_test(iris$Sepal.Length,mu=5)
gorica(ttest1,"x<5.8")

# EXAMPLE 2. ANOVA
aov1 <- aov(yield ~ block-1 + N * P + K, npk)
gorica(aov1,hypothesis="block1=block5;
   K1<0")

# EXAMPLE 3. glm
counts <- c(18,17,15,20,10,20,25,13,12)
outcome <- gl(3,1,9)
treatment <- gl(3,3)
fit <- glm(counts ~ outcome-1 + treatment, family = poisson())
gorica(fit, "outcome1 > (outcome2, outcome3)")

# EXAMPLE 4. ANOVA
res <- lm(Sepal.Length ~ Species-1, iris)
est <- get_estimates(res)
est
gor <- gorica(res, "Speciessetosa < (Speciesversicolor, Speciesvirginica)",
comparison = "complement")
gor
}
}
\references{
Altinisik, Y., Van Lissa, C. J., Hoijtink, H., Oldehinkel, A. J.,
& Kuiper, R. M. (2021). Evaluation of inequality constrained hypotheses using
a generalization of the AIC. Psychological Methods, 26(5), 599–621.
\doi{10.31234/osf.io/t3c8g}.

Bollen, K. (1989). Structural equations with latent variables. New York, NY:
John Wiley and Sons.

Kuiper, R. M., Hoijtink, H., & Silvapulle, M. J. (2011).
An Akaike-type information criterion for model selection under inequality
constraints. Biometrika, 98, 495-501.
\doi{10.31219/osf.io/ekxsn}

Kuiper, R. M., Hoijtink, H., & Silvapulle, M. J. (2012).
Generalization of the order-restricted information criterion for multivariate
normal linear models. Journal of statistical planning and inference, 142(8),
2454-2463. \doi{10.1016/j.jspi.2012.03.007}

Vanbrabant, L., Van Loey, N., and Kuiper, R.M. (2019).
Evaluating a theory-based hypothesis against its complement using an AIC-type
information criterion with an application to facial burn injury.
Psychological Methods. \doi{10.31234/osf.io/n6ydv}

McCullagh, P. & Nelder, J. (1989). Generalized linear models (2nd ed.). Boca
Raton, FL: Chapman & Hall / CRC.

McCulloch, C. E., & Searle, S. R. (2001). Generalized linear and mixed
models. New York, NY: Wiley.
}
\author{
Caspar van Lissa, Yasin Altinisik, Rebecca Kuiper
}
