% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gsDesign.R
\name{gsBound}
\alias{gsBound}
\alias{gsBound1}
\title{Boundary derivation - low level}
\usage{
gsBound(I, trueneg, falsepos, tol = 1e-06, r = 18, printerr = 0)

gsBound1(theta, I, a, probhi, tol = 1e-06, r = 18, printerr = 0)
}
\arguments{
\item{I}{Vector containing statistical information planned at each analysis.}

\item{trueneg}{Vector of desired probabilities for crossing upper bound
assuming mean of 0.}

\item{falsepos}{Vector of desired probabilities for crossing lower bound
assuming mean of 0.}

\item{tol}{Tolerance for error (scalar; default is 0.000001). Normally this
will not be changed by the user.  This does not translate directly to number
of digits of accuracy, so use extra decimal places.}

\item{r}{Integer value (>= 1 and <= 80) controlling the number of numerical
integration grid points. Default is 18, as recommended by Jennison and
Turnbull (2000). Grid points are spread out in the tails for accurate
probability calculations. Larger values provide more grid points and greater
accuracy but slow down computation. Jennison and Turnbull (p. 350) note an
accuracy of \eqn{10^{-6}} with \code{r = 16}. This parameter is normally
not changed by users.}

\item{printerr}{If this scalar argument set to 1, this will print messages
from underlying C program.  Mainly intended to notify user when an output
solution does not match input specifications.  This is not intended to stop
execution as this often occurs when deriving a design in \code{gsDesign}
that uses beta-spending.}

\item{theta}{Scalar containing mean (drift) per unit of statistical
information.}

\item{a}{Vector containing lower bound that is fixed for use in
\code{gsBound1}.}

\item{probhi}{Vector of desired probabilities for crossing upper bound
assuming mean of theta.}
}
\value{
Both routines return a list. Common items returned by the two
routines are: \item{k}{The length of vectors input; a scalar.}
\item{theta}{As input in \code{gsBound1()}; 0 for \code{gsBound()}.}
\item{I}{As input.} \item{a}{For \code{gsbound1}, this is as input. For
\code{gsbound} this is the derived lower boundary required to yield the
input boundary crossing probabilities under the null hypothesis.}
\item{b}{The derived upper boundary required to yield the input boundary
crossing probabilities under the null hypothesis.} \item{tol}{As input.}
\item{r}{As input.} \item{error}{Error code. 0 if no error; greater than 0
otherwise.}

\code{gsBound()} also returns the following items: \item{rates}{a list
containing two items:} \item{falsepos}{vector of upper boundary crossing
probabilities as input.} \item{trueneg}{vector of lower boundary crossing
probabilities as input.}

\code{gsBound1()} also returns the following items: \item{problo}{vector of
lower boundary crossing probabilities; computed using input lower bound and
derived upper bound.} \item{probhi}{vector of upper boundary crossing
probabilities as input.}
}
\description{
\code{gsBound()} and \code{gsBound1()} are lower-level functions used to
find boundaries for a group sequential design. They are not recommended
(especially \code{gsBound1()}) for casual users. These functions do not
adjust sample size as \code{gsDesign()} does to ensure appropriate power for
a design.

\code{gsBound()} computes upper and lower bounds given boundary crossing
probabilities assuming a mean of 0, the usual null hypothesis.
\code{gsBound1()} computes the upper bound given a lower boundary, upper
boundary crossing probabilities and an arbitrary mean (\code{theta}).

The function \code{gsBound1()} requires special attention to detail and
knowledge of behavior when a design corresponding to the input parameters
does not exist.
}
\note{
The gsDesign technical manual is available at
  \url{https://keaven.github.io/gsd-tech-manual/}.
}
\examples{

# set boundaries so that probability is .01 of first crossing
# each upper boundary and .02 of crossing each lower boundary
# under the null hypothesis
x <- gsBound(
  I = c(1, 2, 3) / 3, trueneg = rep(.02, 3),
  falsepos = rep(.01, 3)
)
x

#  use gsBound1 to set up boundary for a 1-sided test
x <- gsBound1(
  theta = 0, I = c(1, 2, 3) / 3, a = rep(-20, 3),
  probhi = c(.001, .009, .015)
)
x$b

# check boundary crossing probabilities with gsProbability
y <- gsProbability(k = 3, theta = 0, n.I = x$I, a = x$a, b = x$b)$upper$prob

#  Note that gsBound1 only computes upper bound
#  To get a lower bound under a parameter value theta:
#      use minus the upper bound as a lower bound
#      replace theta with -theta
#      set probhi as desired lower boundary crossing probabilities
#  Here we let set lower boundary crossing at 0.05 at each analysis
#  assuming theta=2.2
y <- gsBound1(
  theta = -2.2, I = c(1, 2, 3) / 3, a = -x$b,
  probhi = rep(.05, 3)
)
y$b

#  Now use gsProbability to look at design
#  Note that lower boundary crossing probabilities are as
#  specified for theta=2.2, but for theta=0 the upper boundary
#  crossing probabilities are smaller than originally specified
#  above after first interim analysis
gsProbability(k = length(x$b), theta = c(0, 2.2), n.I = x$I, b = x$b, a = -y$b)
}
\references{
Jennison C and Turnbull BW (2000), \emph{Group Sequential
Methods with Applications to Clinical Trials}. Boca Raton: Chapman and Hall.
}
\seealso{
\code{vignette("gsDesignPackageOverview")}, \code{\link{gsDesign}},
\code{\link{gsProbability}}
}
\author{
Keaven Anderson \email{keaven_anderson@merck.com}
}
\keyword{design}
