% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gs_design_wlr.R
\name{gs_design_wlr}
\alias{gs_design_wlr}
\title{Group sequential design using weighted log-rank test under non-proportional hazards}
\usage{
gs_design_wlr(
  enroll_rate = define_enroll_rate(duration = c(2, 2, 10), rate = c(3, 6, 9)),
  fail_rate = tibble(stratum = "All", duration = c(3, 100), fail_rate = log(2)/c(9, 18),
    hr = c(0.9, 0.6), dropout_rate = rep(0.001, 2)),
  weight = wlr_weight_fh,
  approx = "asymptotic",
  alpha = 0.025,
  beta = 0.1,
  ratio = 1,
  info_frac = NULL,
  info_scale = c("h0_h1_info", "h0_info", "h1_info"),
  analysis_time = 36,
  binding = FALSE,
  upper = gs_b,
  upar = gsDesign(k = 3, test.type = 1, n.I = c(0.25, 0.75, 1), sfu = sfLDOF, sfupar =
    NULL)$upper$bound,
  lower = gs_b,
  lpar = c(qnorm(0.1), -Inf, -Inf),
  test_upper = TRUE,
  test_lower = TRUE,
  h1_spending = TRUE,
  r = 18,
  tol = 1e-06,
  interval = c(0.01, 100)
)
}
\arguments{
\item{enroll_rate}{Enrollment rates.}

\item{fail_rate}{Failure and dropout rates.}

\item{weight}{Weight of weighted log rank test:
\itemize{
\item \code{"1"} = unweighted.
\item \code{"n"} = Gehan-Breslow.
\item \code{"sqrtN"} = Tarone-Ware.
\item \code{"FH_p[a]_q[b]"} = Fleming-Harrington with p=a and q=b.
}}

\item{approx}{Approximate estimation method for Z statistics.
\itemize{
\item \code{"event_driven"} = only work under proportional hazard model with log rank test.
\item \code{"asymptotic"}.
}}

\item{alpha}{One-sided Type I error.}

\item{beta}{Type II error.}

\item{ratio}{Experimental:Control randomization ratio (not yet implemented).}

\item{info_frac}{Targeted information fraction at each analysis.}

\item{info_scale}{Information scale for calculation. Options are:
\itemize{
\item \code{"h0_h1_info"} (default): variance under both null and alternative hypotheses is used.
\item \code{"h0_info"}: variance under null hypothesis is used.
\item \code{"h1_info"}: variance under alternative hypothesis is used.
}}

\item{analysis_time}{Minimum time of analysis.}

\item{binding}{Indicator of whether futility bound is binding;
default of \code{FALSE} is recommended.}

\item{upper}{Function to compute upper bound.}

\item{upar}{Parameters passed to \code{upper}.}

\item{lower}{Function to compute lower bound.}

\item{lpar}{Parameters passed to \code{lower}.}

\item{test_upper}{Indicator of which analyses should include an upper
(efficacy) bound; single value of \code{TRUE} (default) indicates all analyses;
otherwise, a logical vector of the same length as \code{info} should indicate
which analyses will have an efficacy bound.}

\item{test_lower}{Indicator of which analyses should include an lower bound;
single value of \code{TRUE} (default) indicates all analyses;
single value \code{FALSE} indicated no lower bound; otherwise, a logical vector
of the same length as \code{info} should indicate which analyses will have a
lower bound.}

\item{h1_spending}{Indicator that lower bound to be set by spending
under alternate hypothesis (input \code{fail_rate})
if spending is used for lower bound.}

\item{r}{Integer value controlling grid for numerical integration as in
Jennison and Turnbull (2000); default is 18, range is 1 to 80.
Larger values provide larger number of grid points and greater accuracy.
Normally, \code{r} will not be changed by the user.}

\item{tol}{Tolerance parameter for boundary convergence (on Z-scale).}

\item{interval}{An interval that is presumed to include the time at which
expected event count is equal to targeted event.}
}
\value{
A list with input parameters, enrollment rate, analysis, and bound.
}
\description{
Group sequential design using weighted log-rank test under non-proportional hazards
}
\section{Specification}{

\if{latex}{
 \itemize{
   \item Validate if input analysis_time is a positive number or a positive increasing sequence.
   \item Validate if input info_frac is a positive number
   or positive increasing sequence on (0, 1] with final value of 1.
   \item Validate if inputs info_frac and analysis_time  have the same length if both have length > 1.
   \item Compute information at input analysis_time using \code{gs_info_wlr()}.
   \item Compute sample size and bounds using \code{gs_design_npe()}.
   \item Return a list of design enrollment, failure rates, and bounds.
  }
}
\if{html}{The contents of this section are shown in PDF user manual only.}
}

\examples{
library(dplyr)
library(mvtnorm)
library(gsDesign)
library(gsDesign2)

# set enrollment rates
enroll_rate <- define_enroll_rate(duration = 12, rate = 500 / 12)

# set failure rates
fail_rate <- define_fail_rate(
  duration = c(4, 100),
  fail_rate = log(2) / 15, # median survival 15 month
  hr = c(1, .6),
  dropout_rate = 0.001
)

# Example 1 ----
# Boundary is fixed
x <- gsSurv(
  k = 3,
  test.type = 4,
  alpha = 0.025, beta = 0.2,
  astar = 0, timing = 1,
  sfu = sfLDOF, sfupar = 0,
  sfl = sfLDOF, sflpar = 0,
  lambdaC = 0.1,
  hr = 0.6, hr0 = 1,
  eta = 0.01, gamma = 10,
  R = 12, S = NULL,
  T = 36, minfup = 24,
  ratio = 1
)

gs_design_wlr(
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  ratio = 1,
  alpha = 0.025, beta = 0.2,
  weight = function(x, arm0, arm1) {
    wlr_weight_fh(x, arm0, arm1, rho = 0, gamma = 0.5)
  },
  upper = gs_b,
  upar = x$upper$bound,
  lower = gs_b,
  lpar = x$lower$bound,
  analysis_time = c(12, 24, 36)
)

# Example 2 ----
# Boundary derived by spending function
gs_design_wlr(
  enroll_rate = enroll_rate,
  fail_rate = fail_rate,
  ratio = 1,
  alpha = 0.025, beta = 0.2,
  weight = function(x, arm0, arm1) {
    wlr_weight_fh(x, arm0, arm1, rho = 0, gamma = 0.5)
  },
  upper = gs_spending_bound,
  upar = list(sf = gsDesign::sfLDOF, total_spend = 0.025),
  lower = gs_spending_bound,
  lpar = list(sf = gsDesign::sfLDOF, total_spend = 0.2),
  analysis_time = c(12, 24, 36)
)
}
