## vim:textwidth=80:expandtab:shiftwidth=2:softtabstop=2
library(gsw)

## These tests compare with check values on the TEOS-10 website, at the 
## links provided in the R documentation of each function.

context("TEOS-10")

test_that("handle NaN values", {
          expect_equal(c(35.16579718, NA), gsw_SA_from_SP(SP=c(35, 35), p=c(100,100), longitude=c(-30,NA), latitude=c(45,30)))
          expect_equal(c(9.992854755, NA), gsw_CT_from_pt(c(35, NA), pt=c(10,11)))
})

test_that("gsw_adiabatic_lapse_rate_from_CT", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          lr <- gsw_adiabatic_lapse_rate_from_CT(SA, CT, p)
          expect_equal(lr, 1e-7*c(0.240199646230069, 0.238457486976761, 0.203635157319712,
                                  0.119829566859790, 0.100052760967308, 0.087773070307283))
})

test_that("gsw_alpha", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          alpha <- gsw_alpha(SA,CT,p)
          expect_equal(alpha, 1e-3*c(0.324464211877393, 0.322610094680523, 0.281335030247435,
                                     0.173529986885424, 0.146898108553385, 0.130265123640082))
})

test_that("gsw_alpha_on_beta", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          alpha_on_beta <- gsw_alpha_on_beta(SA,CT,p)
          expect_equal(alpha_on_beta, c(0.452468543022009, 0.449601695030057, 0.387140203094424,
                                        0.230778871228268, 0.193747796234162, 0.170946048860385))
})

test_that("alpha_wrt_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <-  c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          alpha_wrt_t_exact <- gsw_alpha_wrt_t_exact(SA,t,p)
          expect_equal(alpha_wrt_t_exact, 1e-3*c(0.325601747227247, 0.323448083851267, 0.281413883319329,
                                                 0.172825692975230, 0.145569941503599, 0.128362986933288))
})

test_that("beta = gsw_beta", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          beta <- gsw_beta(SA,CT,p)
          expect_equal(beta, 1e-3*c(0.717521909550091, 0.717657376442386, 0.726169785748549,
                                    0.750420924314564, 0.754903052075032, 0.756841573481865))
})

test_that("gsw_beta_const_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          b <- gsw_beta_const_t_exact(SA, t, p)
          expect_equal(b, 1e-3*c(0.731120837010429, 0.731071779078011, 0.736019128913071,
                                 0.753810501711847, 0.757259405338257, 0.758649268096996))
})

test_that("gsw_cabbeling", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          cabbeling <- gsw_cabbeling(SA, CT, p)
          expect_equal(cabbeling, 1e-4*c(0.086645721047423, 0.086837829466794, 0.092525582052438,
                                         0.108884336975401, 0.112971197222338, 0.115483896148927))
})

test_that("gsw_C_from_SP", {
          SP <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          C <- gsw_C_from_SP(SP, t, p)
          expect_equal(C, c(56.412599581571186, 56.316185602699953, 50.670369333973944,
                            38.134518936104350, 35.056577637635257, 32.986550607990118))
})

test_that("gsw_cp_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <-  c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          cp_t_exact <- gsw_cp_t_exact(SA, t, p)
          expect_equal(cp_t_exact, 1e3*c(4.002888003958537, 4.000980283927373, 3.995546468894633,
                                         3.985076769021370, 3.973593843482723, 3.960184084786622))
})

test_that("gsw_CT_freezing", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          saturation_fraction <- 1
          CT_freezing <- gsw_CT_freezing(SA, p, saturation_fraction)
          expect_equal(CT_freezing, c(-1.899683776424096, -1.940791867869104, -2.006240664432488,
                                      -2.092357761318778, -2.359300831770506, -2.677162675412748))
})

test_that("gsw_CT_from_pt", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          pt <- c(28.7832, 28.4209, 22.7850, 10.2305,  6.8292,  4.3245)
          CT <- gsw_CT_from_pt(SA, pt)
          expect_equal(CT, c(28.809923015982083, 28.439144260767169, 22.786246608464264,
                             10.226165605435785, 6.827183417643142,  4.323565182322069))
})

test_that("gsw_CT_from_t", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <-  c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          CT <- gsw_CT_from_t(SA, t, p)
          expect_equal(CT, c(28.809919826700281, 28.439227816091140, 22.786176893078498,
                             10.226189266620782, 6.827213633479988, 4.323575748610455))
})

test_that("gsw_deltaSA_from_SP", {
          SP =   c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          p =    c(     10,      50,     125,     250,     600,    1000)
          lat =  c(      4,       4,       4,       4,       4,       4)
          long = c(    188,     188,     188,     188,     188,     188)
          deltaSA = gsw_deltaSA_from_SP(SP,p,long,lat)
          expect_equal(deltaSA, c(0.000167203365230, 0.000268836122231, 0.000665803155705,
                                  0.002706154619403, 0.005652977406832,  0.009444734661606))
})

test_that("gsw_dynamic_enthalpy", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <-c( 28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <- c(      10,      50,     125,     250,     600,    1000)
          de <- gsw_dynamic_enthalpy(SA, CT, p)
          expect_equal(de, 1e3*c(0.097864698087770, 0.489161476686235, 1.220512192086506,
                                 2.433731199531144, 5.833880057399701, 9.711443860944032))
})

test_that("gsw_enthalpy", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <-c( 28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <- c(      10,      50,     125,     250,     600,    1000)
          e <- gsw_enthalpy(SA, CT, p)
          expect_equal(e, 1e5*c(1.151031813559086, 1.140146926828028, 0.921800138366058,
                                0.432553713026279, 0.330871609742468, 0.269706841603465))
})

test_that("gsw_enthalpy_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          e <- gsw_enthalpy_t_exact(SA, t, p)
          expect_equal(e, 1e5*c(1.151032604783763, 1.140148036012021, 0.921799209310966,
                                0.432553283808897, 0.330872159700175, 0.269705880448018))
})

test_that("gsw_entropy_from_t", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          e <- gsw_entropy_from_t(SA, t, p)
          expect_equal(e, 100*c(4.003894252787245, 3.954381784340642, 3.198664981986740,
                                1.467908815899072, 0.986473408657975, 0.627915087346090))
})

test_that("gsw_grav", {
          lat <- c(-90, -60, -30, 0)
          grav <- gsw_grav(lat)
          expect_equal(grav, c(9.832186205884799, 9.819178859991149,
                               9.793249257048750, 9.780327000000000))
})

test_that("gsw_internal_energy", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          e <- gsw_internal_energy(SA, CT, p)
          expect_equal(e, 1e5*c(1.148091576956162, 1.134013145527675, 0.909571141498779,
                                0.408593072177020, 0.273985276460357, 0.175019409258405))
})

test_that("gsw_IPV_vs_fNsquared_ratio", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          p_ref <- 0
          r <- gsw_IPV_vs_fNsquared_ratio(SA, CT, p, p_ref)
          expect_equal(r$IPV_vs_fNsquared_ratio, c(0.999742244888022, 0.996939883468178, 0.986141997098021,
                                                   0.931595598713477, 0.861224354872028))
          expect_equal(r$p_mid, c(30, 87.5, 187.5, 425, 800))
})

test_that("gsw_kappa", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <-c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          kappa <- gsw_kappa(SA, CT, p)
          expect_equal(kappa, 1e-9*c(0.411343648791300, 0.411105416128094, 0.416566236026610,
                                     0.435588650838751, 0.438782500588955, 0.439842289994702))
})

test_that("gsw_kappa_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <-c(  28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          kappa <- gsw_kappa_t_exact(SA, t, p)
          expect_equal(kappa, 1e-9*c(0.411245799180373, 0.411029072229334, 0.416539558054756,
                                     0.435668337689072, 0.438923693006423, 0.440037575765429))
})

test_that("gsw_latentheat_evap_CT", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          lh  <- gsw_latentheat_evap_CT(SA, CT)
          expect_equal(lh, 1e6*c(2.429947107462561, 2.430774073049213, 2.444220372158452,
                                 2.474127109232524, 2.482151446148560, 2.488052297193594))
})

test_that("gsw_latentheat_evap_t", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          lh = gsw_latentheat_evap_t(SA, t)
          expect_equal(lh, 1e6*c(2.429882982734836, 2.430730236218543, 2.444217294049004,
                                 2.474137411322517, 2.482156276375029, 2.488054617630297))
})

test_that("gsw_latentheat_evap_melting", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          p <- c(      10,      50,     125,     250,     600,    1000)
          lh <- gsw_latentheat_melting(SA, p)
          expect_equal(lh, 1e5*c(3.299496680271213, 3.298613352397986, 3.297125622834541,
                                 3.294973895330757, 3.288480445559747, 3.280715862416388))
})

test_that("gsw_Nsquared", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <- c(      10,      50,     125,     250,     600,    1000)
          latitude <- 4
          r <- gsw_Nsquared(SA, CT, p, latitude)
          N2 <- r$N2
          p_mid <- r$p_mid
          expect_equal(N2, 1e-3*c(0.060843209693499, 0.235723066151305, 0.216599928330380,
                                  0.012941204313372, 0.008434782795209))
          expect_equal(p_mid, c(30, 87.5, 187.5, 425, 800))
})

test_that("gsw_p_from_z", {
          z <- -c(10, 50, 125, 250, 600, 1000)
          latitude <- 4
          p <- gsw_p_from_z(z, latitude)
          expect_equal(p, 1e3*c(0.010055726724518, 0.050283543374874, 0.125731858435610,
                                0.251540299593468, 0.604210012340727, 1.007990337692001))
})

test_that("gsw_pot_rho_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          p_ref  <- 0
          prho <- gsw_pot_rho_t_exact(SA,t,p,p_ref)
          expect_equal(prho, 1e3*c(1.021798145811089, 1.022052484416980, 1.023893583651958,
                                   1.026667621124443, 1.027107230868492, 1.027409631264134))
})

test_that("gsw_pt0_from_t", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          pt0 <- gsw_pt0_from_t(SA, t, p)
          expect_equal(pt0, c(28.783196819670632, 28.420983342398962, 22.784930399117108,
                              10.230523661095731, 6.829230224409661, 4.324510571845719))
})

test_that("gsw_pt_from_CT", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          pt <- gsw_pt_from_CT(SA, CT)
          expect_equal(pt, c(28.783177048624573, 28.420955597191984, 22.784953468087107,
                             10.230534394434429, 6.829216587061605, 4.324534835990236))
})

test_that("gsw_pt_from_t", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          p_ref <- 0
          pt <- gsw_pt_from_t(SA, t, p, p_ref)
          expect_equal(pt, c(28.783196819670632, 28.420983342398962, 22.784930399117108,
                             10.230523661095731, 6.829230224409661, 4.324510571845719)) 
})

test_that("gsw_rho", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          rho <- gsw_rho(SA,CT,p)
          expect_equal(rho, 1e3*c(1.021839935738108, 1.022262457966867, 1.024427195413316,
                                  1.027790152759127, 1.029837779000189, 1.032002453224572))
})

test_that("gsw_rho_first_derivatives", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <- c(      10,      50,     125,     250,     600,    1000)
          r <- gsw_rho_first_derivatives(SA, CT, p)
          expect_equal(r$drho_dSA, c(0.733153791778356, 0.733624109867480, 0.743950957375504,
                                     0.771357282286743, 0.777581141431288, 0.781278296628328))
          expect_equal(r$drho_dCT, c(-0.331729027977015, -0.329838643311336, -0.288013324730644,
                                     -0.178012962919839, -0.150654632545556, -0.133556437868984))
          expect_equal(r$drho_dp, 1e-6*c(0.420302360738476, 0.420251070273888, 0.426773054953941,
                                         0.447763615252861, 0.452011501791479, 0.454118117103094))
})

test_that("gsw_rho_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <-  c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          rho <- gsw_rho_t_exact(SA, t, p)
          expect_equal(rho, 1e3*c(1.021840173185531, 1.022262689926782, 1.024427715941676,
                                  1.027790201811623, 1.029837714725961, 1.032002404116447))
})

test_that("gsw_SA_from_rho", {
          rho <- c(1021.8482, 1022.2647, 1024.4207, 1027.7841, 1029.8287, 1031.9916)
          CT <-c(    28.7856,   28.4329,   22.8103,   10.2600,    6.8863,    4.4036)
          p <- c(         10,        50,       125,       250,       600,      1000)
          SA <- gsw_SA_from_rho(rho, CT, p)
          expect_equal(SA, c(34.712080120418108, 34.891723808488869, 35.026202257609505,
                             34.847160842234572, 34.736398269039945, 34.732228881079742))
})

test_that("gsw_SA_from_SP", {
          SP <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
          p <- c(      10,      50,     125,     250,     600,    1000)
          lat <- c(     4,       4,       4,       4,       4,       4)
          long <- c(  188,     188,     188,     188,     188,     188)
          SA <- gsw_SA_from_SP(SP, p, long, lat)
          expect_equal(SA, c(34.711778344814114, 34.891522618230098, 35.025544862476920,
                             34.847229026189588, 34.736628474576051, 34.732363065590846))
})

test_that("gsw_SA_from_Sstar", {
          SP <- c(34.7115, 34.8912, 35.0247, 34.8436, 34.7291, 34.7197)
          p <- c(      10,      50,     125,     250,     600,    1000)
          lat <- c(     4,       4,       4,       4,       4,       4)
          long <- c(  188,     188,     188,     188,     188,     188)
          SA <- gsw_SA_from_Sstar(SP, p, long, lat)
          expect_equal(SA, c(34.711724663585905, 34.891561223296009, 35.025594598699882,
                             34.847235885385913, 34.736694493054166, 34.732387111902753))
})

test_that("gsw_sigma0", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          sigma0 <- gsw_sigma0(SA,CT)
          expect_equal(sigma0, c(21.797900819337656, 22.052215404397316, 23.892985307893923,
                                 26.667608665972011, 27.107380455119710, 27.409748977090885))
})

test_that("gsw_sigma1", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          sigma1 <- gsw_sigma1(SA,CT)
          expect_equal(sigma1, c(25.955618850310202, 26.213131422420247, 28.125423775188438,
                                 31.120360038882382, 31.637724222733368, 32.002453224572037))
})

test_that("gsw_sigma2", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          sigma2 <- gsw_sigma2(SA,CT)
          expect_equal(sigma2, c(30.023152223799116, 30.283783336283477, 32.265556840289719,
                                 35.474550881051073, 36.067289438047737, 36.492606494879510))
})

test_that("gsw_sigma3", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          sigma3 <- gsw_sigma3(SA,CT)
          expect_equal(sigma3, c(34.003747849903675, 34.267409891564057, 36.316415829697917,
                                 39.732367693977039, 40.397934186745033, 40.881795690566832))
})

test_that("gsw_sigma4", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          sigma4 <- gsw_sigma4(SA,CT)
          expect_equal(sigma4, c(37.900374609834898, 38.166979617032439, 40.280876075282549,
                                 43.896091033421953, 44.631677245327637, 45.171817312020039))
})

test_that("gsw_sound_speed", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          sound_speed <- gsw_sound_speed(SA,CT,p)
          expect_equal(sound_speed, 1e3*c(1.542426412426373, 1.542558891663385, 1.530801535436184,
                                          1.494551099295314, 1.487622786765276, 1.484271672296205))
})

test_that("gsw_sound_speed_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          sound_speed <- gsw_sound_speed_t_exact(SA,CT,p)
          expect_equal(sound_speed, 1e3*c(1.542615803587414, 1.542703534065789, 1.530844979136360,
                                          1.494409996920661, 1.487377102518027, 1.483934609078705))
})

test_that("gsw_specvol", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <- c(      10,      50,     125,     250,     600,    1000)
          specvol <- gsw_specvol(SA, CT, p)
          expect_equal(specvol, 1e-3*c(0.978626852431313, 0.978222365701325, 0.976155264597929,
                                       0.972961258011157, 0.971026719344908, 0.968989944622149))
})

test_that("gsw_specvol_anom_standard", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <- c(      10,      50,     125,     250,     600,    1000)
          a <- gsw_specvol_anom_standard(SA, CT, p)
          expect_equal(a, 1e-5*c(0.601051894897400, 0.578609769250563, 0.405600538950092,
                                 0.142190453761838, 0.104335535578967, 0.076383389577725))
})

test_that("gsw_specvol_t_exact", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          t <- c( 28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(      10,      50,     125,     250,     600,    1000)
          v <- gsw_specvol_t_exact(SA, t, p)
          expect_equal(v, 1e-3 * c(0.978626625025472, 0.978222143734527, 0.976154768597586,
                                   0.972961211575438, 0.971026779948624, 0.968989990731808))
})

test_that("gsw_SP_from_C", {
          C <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
          t <- c(28.7856, 28.4329, 22.8103, 10.2600,  6.8863,  4.4036)
          p <- c(     10,      50,     125,     250,     600,    1000)
          SP <- gsw_SP_from_C(C,t,p)
          expect_equal(SP,  c(20.009869599086951, 20.265511864874270, 22.981513062527689,
                              31.204503263727982, 34.032315787432829, 36.400308494388170))
})

test_that("gsw_SP_from_SA", {
          SA <-   c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          p <-    c(     10,      50,     125,     250,     600,    1000)
          lat <-  c(      4,       4,       4,       4,       4,       4)
          long <- c(    188,     188,     188,     188,     188,     188)
          SP <- gsw_SP_from_SA(SA,p,long,lat)
          expect_equal(SP, c(34.548721553448317, 34.727477488096639, 34.860554877708005,
                             34.680971112271791, 34.567971663653388, 34.560036751118204))
})

test_that("gsw_SP_from_SK", {
          SK <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
          SP <- gsw_SP_from_SK(SK)
          expect_equal(SP, c(34.548342096952908, 34.727295637119113, 34.860409847645435,
                             34.680755706371187, 34.567658670360110, 34.559651800554022))
          
})

test_that("gsw_SP_from_SR", {
          SR <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
          SP <- gsw_SP_from_SR(SR)
          expect_equal(SP, c(34.386552667080714, 34.564513505458834, 34.696889296869848,
                             34.518231743800094, 34.405762086435850, 34.397799632817147))
})

test_that("gsw_SP_from_Sstar", {
          Sstar <- c(34.7115, 34.8912, 35.0247, 34.8436, 34.7291, 34.7197)
          p <- c(         10,      50,     125,     250,     600,    1000)
          longitude <- 188
          latitude <- 4
          SP <- gsw_SP_from_Sstar(Sstar, p, longitude, latitude)
          expect_equal(SP, c(34.548646570969929, 34.727538423586189, 34.860549501859502,
                             34.681006826476434, 34.568065697992346, 34.560023926979518))
})

test_that("gsw_SR_from_SP", {
          SP <- c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
          SR <- gsw_SR_from_SP(SP)
          expect_equal(SR, c(34.711611927085727, 34.891255045714303, 35.024882197714305,
                             34.844535778285724, 34.731002934857159, 34.722965211428587))
})

test_that("gsw_Sstar_from_SA", {
          SA <-   c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          p <-    c(     10,      50,     125,     250,     600,    1000)
          lat <-  c(      4,       4,       4,       4,       4,       4)
          long <- c(    188,     188,     188,     188,     188,     188)
          Sstar <- gsw_Sstar_from_SA(SA,p,long,lat)
          expect_equal(Sstar, c(34.711575335926490, 34.891138777337822, 35.024705401162166,
                                34.843564118358302, 34.729005527604883, 34.719712883389462))
})

test_that("gsw_Star_from_SP", {
          SP <-   c(34.5487, 34.7275, 34.8605, 34.6810, 34.5680, 34.5600)
          p <-    c(     10,      50,     125,     250,     600,    1000)
          lat <-  c(      4,       4,       4,       4,       4,       4)
          long <- c(    188,     188,     188,     188,     188,     188)
          Sstar <- gsw_Sstar_from_SP(SP,p,long,lat)
          expect_equal(Sstar, c(34.711553680880769, 34.891161395333754, 35.024650265047370,
                                34.843593141519356, 34.729033995955525, 34.719675962471783))
})

test_that("gsw_t_freezing", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          p <- c(      10,      50,     125,     250,     600,    1000)
          saturation_fraction <- 1
          tf <- gsw_t_freezing(SA, p, saturation_fraction)
          expect_equal(tf, c(-1.902730710149803, -1.942908619287183, -2.006861069199743,
                             -2.090985086875259, -2.351293130342102, -2.660498762776720))
})

test_that("gsw_t_from_CT", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          t <- gsw_t_from_CT(SA, CT, p)
          expect_equal(t, c(28.785580227725703, 28.432872246163946, 22.810323087627076,
                            10.260010752788906, 6.886286301029376, 4.403624452383043))
})

test_that("gsw_thermobarbaric", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          tb <- gsw_thermobaric(SA, CT, p)
          expect_equal(tb, 1e-11 * c(0.152618598186650, 0.153662896162852, 0.173429325875738,
                                     0.232810160208414, 0.251984724005424, 0.266660342289558))
})

test_that("gsw_Turner_Rsubrho", {
          SA <- c(34.7118, 34.8915, 35.0256, 34.8472, 34.7366, 34.7324)
          CT <- c(28.8099, 28.4392, 22.7862, 10.2262,  6.8272,  4.3236)
          p <-  c(     10,      50,     125,     250,     600,    1000)
          r <- gsw_Turner_Rsubrho(SA, CT, p)
          expect_equal(r$Tu, c(-2.063858905281147, 41.758435216784427, 47.606966981687535,
                               53.710351151706369, 45.527063858211527)) 
          expect_equal(r$Rsubrho, 100*c(-0.009304335069039, -0.176564834348709, 0.219627771740757,
                                        0.065271424662002, 1.087044054679743)) 
          expect_equal(r$p_mid, 100*c(0.300, 0.875, 1.875, 4.250, 8.000))
})

test_that("gsw_z_from_p", {
          z <- gsw_z_from_p(c(10, 50, 125, 250, 600,1000), 4)
          expect_equal(z, 1e2*c(-0.099445834469453, -0.497180897012550, -1.242726219409978,
                                -2.484700576548589, -5.958253480356214, -9.920919060719987))
})

