/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Emodule.h" 

#include "H5private.h"   
#include "H5Epkg.h"      
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 

bool H5_PKG_INIT_VAR = false;

H5FL_EXTERN(H5E_stack_t);

hid_t
H5Eregister_class(const char *cls_name, const char *lib_name, const char *version)
{
    H5E_cls_t *cls;                         
    hid_t      ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (cls_name == NULL || lib_name == NULL || version == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "invalid string");

    
    if (NULL == (cls = H5E__register_class(cls_name, lib_name, version)))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTCREATE, H5I_INVALID_HID, "can't create error class");

    
    if ((ret_value = H5I_register(H5I_ERROR_CLASS, cls, true)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTREGISTER, H5I_INVALID_HID, "can't register error class");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eunregister_class(hid_t class_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_ERROR_CLASS != H5I_get_type(class_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an error class");

    
    if (H5I_dec_app_ref(class_id) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTDEC, FAIL, "unable to decrement ref count on error class");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Eget_class_name(hid_t class_id, char *name , size_t size)
{
    H5E_cls_t *cls;            
    ssize_t    ret_value = -1; 

    FUNC_ENTER_API((-1))

    
    if (NULL == (cls = (H5E_cls_t *)H5I_object_verify(class_id, H5I_ERROR_CLASS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "not a error class ID");

    
    if ((ret_value = H5E__get_class_name(cls, name, size)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, (-1), "can't get error class name");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eclose_msg(hid_t err_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (H5I_ERROR_MSG != H5I_get_type(err_id))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an error class");

    
    if (H5I_dec_app_ref(err_id) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTDEC, FAIL, "unable to decrement ref count on error message");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Ecreate_msg(hid_t class_id, H5E_type_t msg_type, const char *msg_str)
{
    H5E_cls_t *cls;                         
    H5E_msg_t *msg;                         
    hid_t      ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (msg_type != H5E_MAJOR && msg_type != H5E_MINOR)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "not a valid message type");
    if (msg_str == NULL)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, H5I_INVALID_HID, "message is NULL");

    
    if (NULL == (cls = (H5E_cls_t *)H5I_object_verify(class_id, H5I_ERROR_CLASS)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not an error class ID");

    
    if (NULL == (msg = H5E__create_msg(cls, msg_type, msg_str)))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTCREATE, H5I_INVALID_HID, "can't create error message");

    
    if ((ret_value = H5I_register(H5I_ERROR_MSG, msg, true)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTREGISTER, H5I_INVALID_HID, "can't register error message");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Eget_msg(hid_t msg_id, H5E_type_t *type , char *msg_str , size_t size)
{
    H5E_msg_t *msg;            
    ssize_t    ret_value = -1; 

    FUNC_ENTER_API_NOCLEAR((-1))

    
    if (NULL == (msg = (H5E_msg_t *)H5I_object_verify(msg_id, H5I_ERROR_MSG)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "not a error message ID");

    
    if ((ret_value = H5E__get_msg(msg, type, msg_str, size)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, (-1), "can't get error message text");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Ecreate_stack(void)
{
    H5E_stack_t *stk;                         
    hid_t        ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (stk = H5FL_CALLOC(H5E_stack_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, H5I_INVALID_HID, "memory allocation failed");

    
    H5E__set_default_auto(stk);

    
    if ((ret_value = H5I_register(H5I_ERROR_STACK, stk, true)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTREGISTER, H5I_INVALID_HID, "can't create error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5Eget_current_stack(void)
{
    H5E_stack_t *stk;                         
    hid_t        ret_value = H5I_INVALID_HID; 

    
    FUNC_ENTER_API_NOCLEAR(H5I_INVALID_HID)

    
    if (NULL == (stk = H5E__get_current_stack()))
        HGOTO_ERROR(H5E_ERROR, H5E_CANTCREATE, H5I_INVALID_HID, "can't create error stack");

    
    if ((ret_value = H5I_register(H5I_ERROR_STACK, stk, true)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTREGISTER, H5I_INVALID_HID, "can't create error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eset_current_stack(hid_t err_stack)
{
    H5E_stack_t *estack;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (err_stack != H5E_DEFAULT) {
        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(err_stack, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");

        
        if (H5E__set_current_stack(estack) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "unable to set error stack");

        
        if (H5I_dec_app_ref(err_stack) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTDEC, FAIL, "unable to decrement ref count on error stack");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eclose_stack(hid_t stack_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    if (H5E_DEFAULT != stack_id) {
        
        if (H5I_ERROR_STACK != H5I_get_type(stack_id))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");

        
        if (H5I_dec_app_ref(stack_id) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTDEC, FAIL, "unable to decrement ref count on error stack");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Eget_num(hid_t error_stack_id)
{
    H5E_stack_t *estack;    
    ssize_t      ret_value; 

    
    FUNC_ENTER_API_NOCLEAR((-1))

    
    if (error_stack_id == H5E_DEFAULT) {
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, (-1), "can't get current error stack");
    } 
    else {
        
        H5E_clear_stack();

        
        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(error_stack_id, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, (-1), "not an error stack ID");
    } 

    
    if ((ret_value = H5E__get_num(estack)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, (-1), "can't get number of errors");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Epop(hid_t err_stack, size_t count)
{
    H5E_stack_t *estack;
    herr_t       ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if (err_stack == H5E_DEFAULT) {
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else {
        
        H5E_clear_stack();

        
        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(err_stack, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");
    } 

    
    if (count > estack->nused)
        count = estack->nused;

    
    if (H5E__pop(estack, count) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTRELEASE, FAIL, "can't pop errors from stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Epush2(hid_t err_stack, const char *file, const char *func, unsigned line, hid_t cls_id, hid_t maj_id,
         hid_t min_id, const char *fmt, ...)
{
    H5E_stack_t *estack;              
    va_list      ap;                  
    bool         va_started = false;  
    const char  *tmp_file;            
    const char  *tmp_func;            
    herr_t       ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if (err_stack == H5E_DEFAULT) {
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get the default error stack");
    }
    else {
        
        H5E_clear_stack();

        
        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(err_stack, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");
    } 

    
    if (!estack->paused) {
        
        va_start(ap, fmt);
        va_started = true;

        
        if (NULL == (tmp_file = strdup(file)))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTALLOC, FAIL, "can't duplicate file string");
        if (NULL == (tmp_func = strdup(func)))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTALLOC, FAIL, "can't duplicate function string");

        
        if (cls_id != H5E_ERR_CLS_g)
            if (H5I_inc_ref(cls_id, false) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTINC, FAIL, "can't increment class ID");
        if (maj_id < H5E_first_maj_id_g || maj_id > H5E_last_maj_id_g)
            if (H5I_inc_ref(maj_id, false) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTINC, FAIL, "can't increment major error ID");
        if (min_id < H5E_first_min_id_g || min_id > H5E_last_min_id_g)
            if (H5I_inc_ref(min_id, false) < 0)
                HGOTO_ERROR(H5E_ERROR, H5E_CANTINC, FAIL, "can't increment minor error ID");

        
        if (H5E__push_stack(estack, true, tmp_file, tmp_func, line, cls_id, maj_id, min_id, fmt, &ap) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't push error on stack");
    }

done:
    if (va_started)
        va_end(ap);

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eclear2(hid_t err_stack)
{
    H5E_stack_t *estack;              
    herr_t       ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if (err_stack == H5E_DEFAULT)
        estack = NULL;
    else {
        
        H5E_clear_stack();

        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(err_stack, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");
    } 

    
    if (H5E__destroy_stack(estack) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't clear error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eprint2(hid_t err_stack, FILE *stream)
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if ((ret_value = H5E__print2(err_stack, stream)) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTLIST, FAIL, "can't display error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Ewalk2(hid_t err_stack, H5E_direction_t direction, H5E_walk2_t stack_func, void *client_data)
{
    H5E_stack_t  *estack;              
    H5E_walk_op_t op;                  
    herr_t        ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if (err_stack == H5E_DEFAULT) {
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else {
        
        H5E_clear_stack();

        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(err_stack, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");
    } 

    
    op.vers    = 2;
    op.u.func2 = stack_func;
    if ((ret_value = H5E__walk(estack, direction, &op, client_data)) < 0)
        HERROR(H5E_ERROR, H5E_CANTLIST, "can't walk error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eget_auto2(hid_t estack_id, H5E_auto2_t *func , void **client_data )
{
    H5E_stack_t  *estack;              
    H5E_auto_op_t op;                  
    herr_t        ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    if (estack_id == H5E_DEFAULT) {
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else {
        
        H5E_clear_stack();

        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(estack_id, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");
    } 

    
    if (H5E__get_auto(estack, &op, client_data) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get automatic error info");

#ifndef H5_NO_DEPRECATED_SYMBOLS
    
    if (!op.is_default && op.vers == 1)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "wrong API function, H5Eset_auto1 has been called");
#endif 

    if (func)
        *func = op.func2;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eset_auto2(hid_t estack_id, H5E_auto2_t func, void *client_data)
{
    H5E_stack_t  *estack;              
    H5E_auto_op_t op;                  
    herr_t        ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    if (estack_id == H5E_DEFAULT) {
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else {
        
        H5E_clear_stack();

        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(estack_id, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");
    } 

#ifndef H5_NO_DEPRECATED_SYMBOLS
    
    if (H5E__get_auto(estack, &op, NULL) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get automatic error info");

    
    if (func != op.func2_default)
        op.is_default = false;
    else
        op.is_default = true;

    op.vers = 2;
#endif 

    
    op.func2 = func;

    if (H5E__set_auto(estack, &op, client_data) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTSET, FAIL, "can't set automatic error info");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eauto_is_v2(hid_t estack_id, unsigned *is_stack)
{
    H5E_stack_t *estack;              
    herr_t       ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    if (estack_id == H5E_DEFAULT) {
        if (NULL == (estack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else {
        
        H5E_clear_stack();

        if (NULL == (estack = (H5E_stack_t *)H5I_object_verify(estack_id, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a error stack ID");
    } 

    
    if (is_stack)
#ifndef H5_NO_DEPRECATED_SYMBOLS
        *is_stack = estack->auto_op.vers > 1;
#else
        *is_stack = 1;
#endif

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eappend_stack(hid_t dst_stack_id, hid_t src_stack_id, bool close_source_stack)
{
    H5E_stack_t *dst_stack, *src_stack; 
    herr_t       ret_value = SUCCEED;   

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (dst_stack = (H5E_stack_t *)H5I_object_verify(dst_stack_id, H5I_ERROR_STACK)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "dst_stack_id not a error stack ID");
    if (NULL == (src_stack = (H5E_stack_t *)H5I_object_verify(src_stack_id, H5I_ERROR_STACK)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "src_stack_id not a error stack ID");

    
    if (H5E__append_stack(dst_stack, src_stack) < 0)
        HGOTO_ERROR(H5E_ERROR, H5E_CANTAPPEND, FAIL, "can't append stack");

    
    if (close_source_stack)
        
        if (H5I_dec_app_ref(src_stack_id) < 0)
            HGOTO_ERROR(H5E_ERROR, H5E_CANTDEC, FAIL, "unable to decrement ref count on source error stack");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eis_paused(hid_t stack_id, bool *is_paused)
{
    H5E_stack_t *stack;               
    herr_t       ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOCLEAR(FAIL)

    
    if (stack_id == H5E_DEFAULT) {
        if (NULL == (stack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else {
        
        H5E_clear_stack();

        
        if (NULL == (stack = (H5E_stack_t *)H5I_object_verify(stack_id, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an error stack ID");
    } 

    
    if (NULL == is_paused)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "is_paused parameter is NULL");

    
    *is_paused = (stack->paused > 0);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Epause_stack(hid_t stack_id)
{
    H5E_stack_t *stack;               
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (stack_id == H5E_DEFAULT) {
        if (NULL == (stack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else
        
        if (NULL == (stack = (H5E_stack_t *)H5I_object_verify(stack_id, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an error stack ID");

    
    stack->paused++;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Eresume_stack(hid_t stack_id)
{
    H5E_stack_t *stack;               
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (stack_id == H5E_DEFAULT) {
        if (NULL == (stack = H5E__get_my_stack()))
            HGOTO_ERROR(H5E_ERROR, H5E_CANTGET, FAIL, "can't get current error stack");
    } 
    else
        
        if (NULL == (stack = (H5E_stack_t *)H5I_object_verify(stack_id, H5I_ERROR_STACK)))
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not an error stack ID");

    
    if (0 == stack->paused)
        HGOTO_ERROR(H5E_ERROR, H5E_BADRANGE, FAIL, "resuming more than paused");

    
    stack->paused--;

done:
    FUNC_LEAVE_API(ret_value)
} 
