/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Tmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Tpkg.h"      
#include "H5WBprivate.h" 

void
H5T__bit_copy(uint8_t *dst, size_t dst_offset, const uint8_t *src, size_t src_offset, size_t size)
{
    size_t shift;
    size_t mask_lo, mask_hi;
    size_t s_idx, d_idx;

    FUNC_ENTER_PACKAGE_NOERR

    
    s_idx = src_offset / 8;
    d_idx = dst_offset / 8;
    src_offset %= 8;
    dst_offset %= 8;

    
    while (src_offset && size > 0) {
        size_t nbits = MIN3(size, 8 - dst_offset, 8 - src_offset);
        size_t mask  = ((size_t)1 << nbits) - 1;

        dst[d_idx] &= (uint8_t) ~(mask << dst_offset);
        dst[d_idx] = (uint8_t)(dst[d_idx] | (((src[s_idx] >> src_offset) & (uint8_t)mask) << dst_offset));

        src_offset += nbits;
        if (src_offset >= 8) {
            s_idx++;
            src_offset %= 8;
        }

        dst_offset += nbits;
        if (dst_offset >= 8) {
            d_idx++;
            dst_offset %= 8;
        }

        size -= nbits;
    }

    
    shift   = dst_offset;
    mask_lo = ((size_t)1 << (8 - shift)) - 1;
    mask_hi = (~mask_lo) & 0xff;

    for (; size > 8; size -= 8, d_idx++, s_idx++) {
        if (shift) {
            dst[d_idx + 0] &= (uint8_t)(~(mask_lo << shift));
            dst[d_idx + 0] |= (uint8_t)((src[s_idx] & mask_lo) << shift);
            dst[d_idx + 1] &= (uint8_t)(~(mask_hi >> (8 - shift)));
            dst[d_idx + 1] |= (uint8_t)((src[s_idx] & mask_hi) >> (8 - shift));
        }
        else
            dst[d_idx] = src[s_idx];
    }

    
    while (size > 0) {
        size_t nbits = (size_t)MIN3(size, 8 - dst_offset, 8 - src_offset);
        size_t mask  = ((size_t)1 << nbits) - 1;

        dst[d_idx] &= (uint8_t)(~(mask << dst_offset));
        dst[d_idx] = (uint8_t)(dst[d_idx] | (((src[s_idx] >> src_offset) & (uint8_t)mask) << dst_offset));

        src_offset += nbits;
        if (src_offset >= 8) {
            s_idx++;
            src_offset %= 8;
        }

        dst_offset += nbits;
        if (dst_offset >= 8) {
            d_idx++;
            dst_offset %= 8;
        }

        size -= nbits;
    }

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5T__bit_shift(uint8_t *buf, ssize_t shift_dist, size_t offset, size_t size)
{
    uint8_t tmp_buf[512];        
    H5WB_t *wb        = NULL;    
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(buf);
    assert(size);

    if (shift_dist) {
        size_t abs_shift_dist = (size_t)ABS(shift_dist);

        if (abs_shift_dist >= size)
            H5T__bit_set(buf, offset, size, 0);
        else {
            size_t   buf_size = (size / 8) + 1; 
            uint8_t *shift_buf;                 

            
            if (NULL == (wb = H5WB_wrap(tmp_buf, sizeof(tmp_buf))))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, FAIL, "can't wrap buffer");

            
            if (NULL == (shift_buf = (uint8_t *)H5WB_actual(wb, buf_size)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_NOSPACE, FAIL, "can't get actual buffer");

            
            if (shift_dist > 0) { 
                
                H5T__bit_copy(shift_buf, (size_t)0, buf, offset, size - abs_shift_dist);

                
                H5T__bit_copy(buf, offset + abs_shift_dist, shift_buf, (size_t)0, size - abs_shift_dist);

                
                H5T__bit_set(buf, offset, abs_shift_dist, 0);
            }
            else { 
                H5T__bit_copy(shift_buf, (size_t)0, buf, offset + abs_shift_dist, size - abs_shift_dist);
                H5T__bit_copy(buf, offset, shift_buf, (size_t)0, size - abs_shift_dist);
                H5T__bit_set(buf, offset + size - abs_shift_dist, abs_shift_dist, 0);
            }
        } 
    }     

done:
    
    if (wb && H5WB_unwrap(wb) < 0)
        HDONE_ERROR(H5E_DATATYPE, H5E_CLOSEERROR, FAIL, "can't close wrapped buffer");

    FUNC_LEAVE_NOAPI(ret_value)
} 

uint64_t
H5T__bit_get_d(const uint8_t *buf, size_t offset, size_t size)
{
    uint64_t val = 0;
    size_t   i, hs;
    uint64_t ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    assert(8 * sizeof(val) >= size);

    H5T__bit_copy((uint8_t *)&val, (size_t)0, buf, offset, size);
    switch (H5T_native_order_g) {
        case H5T_ORDER_LE:
            break;

        case H5T_ORDER_BE:
            for (i = 0, hs = sizeof(val) / 2; i < hs; i++) {
                uint8_t tmp                              = ((uint8_t *)&val)[i];
                ((uint8_t *)&val)[i]                     = ((uint8_t *)&val)[sizeof(val) - (i + 1)];
                ((uint8_t *)&val)[sizeof(val) - (i + 1)] = tmp;
            }
            break;

        case H5T_ORDER_ERROR:
        case H5T_ORDER_VAX:
        case H5T_ORDER_NONE:
        case H5T_ORDER_MIXED:
        default:
            
            assert(0 && "unknown byte order");
    }

    
    ret_value = val;

    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5T__bit_set_d(uint8_t *buf, size_t offset, size_t size, uint64_t val)
{
    size_t i, hs;

    FUNC_ENTER_PACKAGE_NOERR

    assert(8 * sizeof(val) >= size);

    switch (H5T_native_order_g) {
        case H5T_ORDER_LE:
            break;

        case H5T_ORDER_BE:
            for (i = 0, hs = sizeof(val) / 2; i < hs; i++) {
                uint8_t tmp                              = ((uint8_t *)&val)[i];
                ((uint8_t *)&val)[i]                     = ((uint8_t *)&val)[sizeof(val) - (i + 1)];
                ((uint8_t *)&val)[sizeof(val) - (i + 1)] = tmp;
            }
            break;

        case H5T_ORDER_ERROR:
        case H5T_ORDER_VAX:
        case H5T_ORDER_NONE:
        case H5T_ORDER_MIXED:
        default:
            
            assert(0 && "unknown byte order");
    }

    H5T__bit_copy(buf, offset, (uint8_t *)&val, (size_t)0, size);

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5T__bit_set(uint8_t *buf, size_t offset, size_t size, bool value)
{
    int idx;

    FUNC_ENTER_PACKAGE_NOERR

    
    idx = (int)offset / 8;
    offset %= 8;

    
    if (size && offset % 8) {
        size_t   nbits = MIN(size, 8 - offset);
        unsigned mask  = ((unsigned)1 << nbits) - 1;

        if (value)
            buf[idx] = (uint8_t)(buf[idx] | (mask << offset));
        else
            buf[idx] &= (uint8_t)(~(mask << offset));

        idx++;
        size -= nbits;
    }

    
    while (size >= 8) {
        buf[idx++] = (uint8_t)(value ? 0xff : 0x00);
        size -= 8;
    }

    
    if (size) {
        if (value)
            buf[idx] |= (uint8_t)(((unsigned)1 << size) - 1);
        else
            buf[idx] &= (uint8_t)(~(((unsigned)1 << size) - 1));
    }

    FUNC_LEAVE_NOAPI_VOID
} 

ssize_t
H5T__bit_find(const uint8_t *buf, size_t offset, size_t size, H5T_sdir_t direction, bool value)
{
    ssize_t base = (ssize_t)offset;
    ssize_t idx, i;
    size_t  iu;
    ssize_t ret_value = (-1); 

    
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(true == 1);

    switch (direction) {
        case H5T_BIT_LSB:
            
            idx = (ssize_t)(offset / 8);
            offset %= 8;

            
            if (offset) {
                for (iu = offset; iu < 8 && size > 0; iu++, size--)
                    if (value == (bool)((buf[idx] >> iu) & 0x01))
                        HGOTO_DONE(8 * idx + (ssize_t)iu - base);

                offset = 0;
                idx++;
            } 

            
            while (size >= 8) {
                if ((value ? 0x00 : 0xff) != buf[idx])
                    for (i = 0; i < 8; i++)
                        if (value == (bool)((buf[idx] >> i) & 0x01))
                            HGOTO_DONE(8 * idx + i - base);

                size -= 8;
                idx++;
            } 

            
            for (i = 0; i < (ssize_t)size; i++)
                if (value == (bool)((buf[idx] >> i) & 0x01))
                    HGOTO_DONE(8 * idx + i - base);
            break;

        case H5T_BIT_MSB:
            
            idx = (ssize_t)((offset + size - 1) / 8);
            offset %= 8;

            
            if (size > 8 - offset && (offset + size) % 8) {
                for (iu = (offset + size) % 8; iu > 0; --iu, --size)
                    if (value == (bool)((buf[idx] >> (iu - 1)) & 0x01))
                        HGOTO_DONE(8 * idx + (ssize_t)(iu - 1) - base);

                --idx;
            } 

            
            while (size >= 8) {
                if ((value ? 0x00 : 0xff) != buf[idx]) {
                    for (i = 7; i >= 0; --i)
                        if (value == (bool)((buf[idx] >> i) & 0x01))
                            HGOTO_DONE(8 * idx + i - base);
                } 

                size -= 8;
                --idx;
            } 

            
            if (size > 0) {
                for (iu = offset + size; iu > offset; --iu)
                    if (value == (bool)((buf[idx] >> (iu - 1)) & 0x01))
                        HGOTO_DONE(8 * idx + (ssize_t)(iu - 1) - base);
            }
            break;

        default:
            assert(0 && "Unknown bit search direction");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5T__bit_inc(uint8_t *buf, size_t start, size_t size)
{
    size_t   idx   = start / 8;
    unsigned carry = 1;
    unsigned acc, mask;

    
    FUNC_ENTER_PACKAGE_NOERR

    assert(buf);

    start %= 8;

    
    if (start) {
        if (size + start < 8)
            mask = ((unsigned)1 << size) - 1;
        else
            mask = ((unsigned)1 << (8 - start)) - 1;
        acc = ((unsigned)buf[idx] >> start) & mask;
        acc++;
        carry = acc & ((unsigned)1 << MIN(size, 8 - start));
        buf[idx] &= (uint8_t)(~(mask << start));
        buf[idx] = (uint8_t)(buf[idx] | ((acc & mask) << start));
        size -= MIN(size, 8 - start);
        start = 0;
        idx++;
    }

    
    while (carry && size >= 8) {
        acc = buf[idx];
        acc++;
        carry    = acc & 0x100;
        buf[idx] = (uint8_t)(acc & 0xff);
        idx++;
        size -= 8;
    }

    
    if (carry && size > 0) {
        mask = ((unsigned)1 << size) - 1;
        acc  = buf[idx] & mask;
        acc++;
        carry = acc & ((unsigned)1 << size);
        buf[idx] &= (uint8_t)(~mask);
        buf[idx] |= (uint8_t)(acc & mask);
    }

    FUNC_LEAVE_NOAPI(carry ? true : false)
} 

bool
H5T__bit_dec(uint8_t *buf, size_t start, size_t size)
{
    size_t   idx = start / 8;
    size_t   pos = start % 8;
    uint8_t  tmp;
    unsigned borrow = 0;

    
    FUNC_ENTER_PACKAGE_NOERR

    assert(buf);
    assert(size);

    
    if ((size + start - 1) / 8 > idx) {
        

        
        if (!(buf[idx] >> pos))
            borrow = 1;
        buf[idx] = (uint8_t)(buf[idx] - (1 << pos));
        idx++;
        size -= (8 - pos);

        
        while (borrow && size >= 8) {
            if (buf[idx])
                borrow = 0;
            buf[idx]--;

            idx++;
            size -= 8;
        }

        
        if (borrow && size > 0) {
            
            tmp = buf[idx];
            buf[idx]--;
            if ((buf[idx] >> size) != tmp >> size)
                buf[idx] = (uint8_t)(buf[idx] + (1 << size));
        }
    }
    else {
        

        
        tmp      = buf[idx];
        buf[idx] = (uint8_t)(buf[idx] - (1 << pos));
        if ((buf[idx] >> (pos + size)) != tmp >> (pos + size)) {
            buf[idx] = (uint8_t)(buf[idx] + (1 << (pos + size)));
            borrow   = 1;
        }
    }

    FUNC_LEAVE_NOAPI(borrow ? true : false)
} 

void
H5T__bit_neg(uint8_t *buf, size_t start, size_t size)
{
    size_t  idx = start / 8;
    size_t  pos = start % 8;
    uint8_t tmp[1];

    
    FUNC_ENTER_PACKAGE_NOERR

    assert(buf);
    assert(size);

    
    tmp[0] = (uint8_t)~buf[idx];

    
    if ((size + start - 1) / 8 > idx) { 
        H5T__bit_copy(&(buf[idx]), pos, tmp, pos, (8 - pos));
        idx++;
        size -= (8 - pos);

        
        while (size >= 8) {
            buf[idx] = (uint8_t) ~(buf[idx]);
            idx++;
            size -= 8;
        }

        
        if (size > 0) {
            
            tmp[0] = (uint8_t)~buf[idx];
            H5T__bit_copy(&(buf[idx]), (size_t)0, tmp, (size_t)0, size);
        }
    }
    else {
        
        H5T__bit_copy(&(buf[idx]), pos, tmp, pos, size);
    }

    FUNC_LEAVE_NOAPI_VOID
} 
