% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/datasets.R
\docType{data}
\name{Rohwer}
\alias{Rohwer}
\title{Rohwer Data Set}
\format{
A data frame with 69 observations on the following 10 variables.
\describe{ 
  \item{\code{group}}{a numeric vector, corresponding to SES}
  \item{\code{SES}}{Socioeconomic status, a factor with levels \code{Hi} \code{Lo}} 
  \item{\code{SAT}}{a numeric vector: score on a Student Achievement Test} 
  \item{\code{PPVT}}{a numeric vector: score on the Peabody Picture Vocabulary Test} 
  \item{\code{Raven}}{a numeric vector: score on the Raven Progressive Matrices Test} 
  \item{\code{n}}{a numeric vector: performance on a 'named' PA task} 
  \item{\code{s}}{a numeric vector: performance on a 'still' PA task} 
  \item{\code{ns}}{a numeric vector: performance on a 'named still' PA task} 
  \item{\code{na}}{a numeric vector: performance on a 'named action' PA task} 
  \item{\code{ss}}{a numeric vector: performance on a 'sentence still' PA task} 
}
}
\source{
Timm, N.H. 1975).  \emph{Multivariate Analysis with Applications in
Education and Psychology}.  Wadsworth (Brooks/Cole), Examples 4.3 (p. 281),
4.7 (p. 313), 4.13 (p. 344).
}
\description{
Data from an experiment by William D. Rohwer on kindergarten children
designed to examine how well performance on a set of paired-associate (PA)
tasks can predict performance on some measures of aptitude and achievement.
}
\details{
The variables \code{SAT}, \code{PPVT} and \code{Raven} are responses to be
potentially explained by performance on the paired-associate (PA) learning
tasks, \code{n}, \code{s}, \code{ns}, \code{na}, and \code{ss},
which differed in the syntactic and semantic relationship between the stimulus and response words in each pair.


Timm (1975) does not give a source, but the most relevant studies are Rowher & Ammons (1968) and Rohwer & Levin (1971).
The paired-associate tasks are described as:
\describe{
   \item{\code{n}}{(named): Simple paired-associate task where participants learn pairs of nouns with no additional context}
   \item{\code{s}}{(sentence): Participants learn pairs embedded within a sentence}
   \item{\code{ns}}{(named sentence): A combination where participants learn noun pairs with sentence context}
   \item{\code{na}}{(named action): Pairs are learned with an action relationship between them}
   \item{\code{ss}}{(sentence still): Similar to the sentence condition but with static presentation}
}
}
\examples{

str(Rohwer)

# Plot responses against each predictor
library(tidyr)
library(dplyr)
library(ggplot2)

yvars <- c("SAT", "PPVT", "Raven" )
xvars <- c("n", "s", "ns", "na", "ss")
Rohwer_long <- Rohwer \%>\%
  pivot_longer(cols = all_of(xvars), names_to = "xvar", values_to = "x") |>
  pivot_longer(cols = all_of(yvars), names_to = "yvar", values_to = "y") |>
  mutate(xvar = factor(xvar, xvars), yvar = factor(yvar, yvars))

ggplot(Rohwer_long, aes(x, y, color = SES, shape = SES, fill = SES)) +
  geom_point() +
  geom_smooth(method = "lm", se = FALSE, formula = y ~ x) +
  stat_ellipse(geom = "polygon", level = 0.68, alpha = 0.1) +
  facet_grid(yvar ~ xvar, scales = "free") +
  labs(x = "predictor", y = "response") +
  theme_bw(base_size = 14)


## ANCOVA, assuming equal slopes
rohwer.mod <- lm(cbind(SAT, PPVT, Raven) ~ SES + n + s + ns + na + ss, data=Rohwer)
car::Anova(rohwer.mod)

# Visualize the ANCOVA model
heplot(rohwer.mod)
# Add ellipse to test all 5 regressors
heplot(rohwer.mod, hypotheses=list("Regr" = c("n", "s", "ns", "na", "ss")))
# View all pairs
pairs(rohwer.mod, hypotheses=list("Regr" = c("n", "s", "ns", "na", "ss")))

# or 3D plot
\dontrun{
col <- c("red", "green3", "blue", "cyan", "magenta", "brown", "gray")
heplot3d(rohwer.mod, hypotheses=list("Regr" = c("n", "s", "ns", "na", "ss")), 
                     col=col, wire=FALSE)
}

## fit separate, independent models for Lo/Hi SES
rohwer.ses1 <- lm(cbind(SAT, PPVT, Raven) ~ n + s + ns + na + ss, data=Rohwer, subset=SES=="Hi")
rohwer.ses2 <- lm(cbind(SAT, PPVT, Raven) ~ n + s + ns + na + ss, data=Rohwer, subset=SES=="Lo")

# overlay the separate HE plots
heplot(rohwer.ses1, ylim=c(40,110),col=c("red", "black"))
heplot(rohwer.ses2, add=TRUE, col=c("blue", "black"), grand.mean=TRUE, error.ellipse=TRUE)


}
\references{
Friendly, M. (2007).  HE plots for Multivariate General Linear
Models.  \emph{Journal of Computational and Graphical Statistics},
\bold{16}(2) 421--444.  \url{http://datavis.ca/papers/jcgs-heplots.pdf}

Rohwer, W.D., Jr., & Levin, J.R. (1968). Action, meaning and stimulus selection
in paired-associate learning. \emph{Journal of Verbal Learning and Verbal Behavior}, \bold{7}: 137-141.

Rohwer, W. D., Jr., & Ammons, M. S. (1971). Elaboration training and paired-associate learning efficiency in children. 
\emph{Journal of Educational Psychology}, \bold{62}(5), 376-383.
}
\concept{MANCOVA}
\concept{MMRA}
\keyword{datasets}
